/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.repository;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Formatter;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jboss.deployers.client.plugins.deployment.AbstractDeployment;
import org.jboss.deployers.client.spi.Deployment;
import org.jboss.deployers.spi.attachments.MutableAttachments;
import org.jboss.deployers.spi.structure.ClassPathEntry;
import org.jboss.deployers.spi.structure.ContextInfo;
import org.jboss.deployers.spi.structure.StructureMetaData;
import org.jboss.deployers.structure.spi.DeploymentContext;
import org.jboss.deployers.structure.spi.main.MainDeployerStructure;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentContext;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedCommon;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.api.ManagedObject;
import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.system.server.profileservice.attachments.AttachmentMetaData;
import org.jboss.system.server.profileservice.attachments.AttachmentStore;
import org.jboss.system.server.profileservice.attachments.DeploymentClassPathMetaData;
import org.jboss.system.server.profileservice.attachments.DeploymentStructureMetaData;
import org.jboss.system.server.profileservice.attachments.LazyPredeterminedManagedObjects;
import org.jboss.system.server.profileservice.attachments.RepositoryAttachmentMetaData;
import org.jboss.system.server.profileservice.attachments.RepositoryAttachmentMetaDataFactory;
import org.jboss.system.server.profileservice.persistence.ManagedObjectPeristenceHandler;
import org.jboss.system.server.profileservice.persistence.xml.PersistedManagedObject;
import org.jboss.virtual.VirtualFile;

/**
 * The AbstractAttachmentStore updates and restores the persisted attachments.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision: 85526 $
 */
public class AbstractAttachmentStore implements AttachmentStore
{
   /** The attachment store root. */
   private final URI attatchmentStoreRoot;

   /** The attachment serializer. */
   protected AbstractFileAttachmentsSerializer serializer;
   
   /** The deployment factory. */
   protected VFSDeploymentFactory deploymentFactory = VFSDeploymentFactory.getInstance();
   
   /** The MainDeployerStructure. */
   protected MainDeployerStructure mainDeployer;
   
   /** The metadata name */
   public static final String METADATA_NAME = "metadata";
   
   /** The managed object persistence handler. */
   private static final ManagedObjectPeristenceHandler handler = new ManagedObjectPeristenceHandler();
   
   /** The logger. */
   private static final Logger log = Logger.getLogger(AbstractAttachmentStore.class);
   
   public AbstractAttachmentStore(File root)
   {
      if(root == null)
         throw new IllegalArgumentException("Null attachmentStoreDir");
      if(root.exists() && root.isDirectory() == false)
         throw new IllegalArgumentException("AttachmentStoreRoot is not a directory.");
      this.attatchmentStoreRoot = root.toURI();
   }
   
   public URI getAttachmentStoreRoot()
   {
      return this.attatchmentStoreRoot;
   }
   
   public VFSDeploymentFactory getDeploymentFactory()
   {
      return deploymentFactory;
   }
   
   public void setDeploymentFactory(VFSDeploymentFactory deploymentFactory)
   {
      this.deploymentFactory = deploymentFactory;
   }
   
   public MainDeployerStructure getMainDeployer()
   {
      return mainDeployer;
   }
   
   public void setMainDeployer(MainDeployerStructure mainDeployer)
   {
      this.mainDeployer = mainDeployer;
   }
   
   public AbstractFileAttachmentsSerializer getSerializer()
   {
      return serializer;
   }
   
   public void setSerializer(AbstractFileAttachmentsSerializer serializer)
   {
      this.serializer = serializer;
   }
   
   public Deployment createMCDeployment(ProfileDeployment profileDeployment) throws Exception
   {
      if(profileDeployment.getRoot() == null)
      {
         return new AbstractDeployment(profileDeployment.getName());
      }
      else
      {
         return deploymentFactory.createVFSDeployment(profileDeployment.getRoot());
      }
   }
   
   /**
    * Create a VFSDeployment with predetermined managed object.
    * 
    * @param file the deployment root.
    * @param phase the deployment phase
    * @return the VFSDeployment
    */
   public Deployment loadDeploymentData(ProfileDeployment profileDeployment) throws Exception
   {
      if(profileDeployment == null)
         throw new IllegalArgumentException("Null profile deployment.");
      
      boolean trace = log.isTraceEnabled();
      
      // Create VFS deployment
      Deployment deployment = createMCDeployment(profileDeployment);
      if(log.isTraceEnabled())
         log.trace("Created deployment: " + deployment);
      
      // TODO also handle normal Deployments
      if(profileDeployment.getRoot() == null)
         return deployment;
      
      // simpleName + hash
      String deploymentPath = createRelativeDeploymentPath(profileDeployment);
      if(trace)
         log.trace("trying to load attachment from relative path: " + deploymentPath);
      
      // Load the metadata
      RepositoryAttachmentMetaData attachmentMetaData = loadAttachmentMetaData(deploymentPath);
      
      if(attachmentMetaData == null)
      {
         log.debug("No persisted attachment found for deployment " + deployment + " with relative path: "+ deploymentPath);
         return deployment;
      }
      
      log.debug("Persisted attachment found for deployment " + deployment + " with relative path: "+ deploymentPath);
      
      try
      {
         // If the deployment has changes we skip restoring the persisted metadata.
         // TODO delete attachments ?
         // TODO check metadata locations
         if(attachPredeterminedObject(profileDeployment.getRoot(), attachmentMetaData) == false)
         {
            log.debug("Not using the persisted metadata, as the deployment was modified.");
            return deployment;
         }
      }
      catch(IOException e)
      {
         log.error("failed to get LastModified date for file, not using persisted metadata: "+ profileDeployment.getName());
         return deployment;
      }
      
      // Start with "" the root contextPath
      rebuildStructureContext(deployment, "", deploymentPath, attachmentMetaData, trace);
      
      return deployment;
   }
   
   /**
    * Determine whether to attach the PredeterminedManagedObjects or not.
    * 
    * TODO this should also check the metadata paths for the deployment. 
    * 
    * @param root the path of the deployment
    * @param metaData the repository meta data
    * @return hasBeenModified.
    * @throws IOException
    */
   protected boolean attachPredeterminedObject(VirtualFile root, RepositoryAttachmentMetaData metaData)
      throws IOException
   {
      boolean attach = true;
      if(metaData.getLastModified() < root.getLastModified())
      {
         attach = false;
      }
      // TODO check metaData locations
      return attach;
   }
   
   /**
    * Persist the updated metadata for a managedComponent and generate a metadata describing
    * the repository, if it does not exist already.
    * 
    * @param deployment the deployment.
    * @param phase the deployment phase.
    * @param comp the managed component.
    * @throws Exception
    */
   public void updateDeployment(ProfileDeployment deployment, ManagedComponent comp)
      throws Exception
   {
      if(deployment ==  null)
         throw new IllegalArgumentException("ProfileDeployment may not be null.");
      
      if(deployment.getRoot() == null)
      {
         // TODO
         log.debug("Cannot persist attachments for non VFS based deployment: " + deployment);
         return;
      }
      
      if(comp == null)
         throw new IllegalArgumentException("ManagedComponent may not be null.");
      
      boolean trace = log.isTraceEnabled();
      
      // Log 
      log.debug("updating deployment: "+ deployment + ", component: "+ comp);
      
      // simpleName + hash
      String deploymentPath = createRelativeDeploymentPath(deployment);
      RepositoryAttachmentMetaData savedMetaData = loadAttachmentMetaData(deploymentPath); 
      
      // Get parent deployment
      ManagedDeployment md = comp.getDeployment();
      String currentContextName = "";
      if(md.getParent() != null)
      {
         while( md.getParent() != null )
         {
            currentContextName = md.getSimpleName() + "/" + currentContextName;
            md = md.getParent();
         }
         currentContextName = fixName(currentContextName);
      }
      
      RepositoryAttachmentMetaData currentContextMetaData = null;
      if(savedMetaData != null)
      {
         if(trace)
            log.trace("Previous metadata found for deployment: " + deployment);
         
         // The root context
         if("".equals(currentContextName))
         {
            currentContextMetaData = savedMetaData;
         }
         else
         {
            for(RepositoryAttachmentMetaData child : savedMetaData.getChildren())
            {
               // extract the current context
               if(child.getDeploymentName().equals(currentContextName))
                  currentContextMetaData = child;
            }
         }
      }
      else
      {
         // Create the metadata
         savedMetaData = RepositoryAttachmentMetaDataFactory.createInstance(md);
         currentContextMetaData = createRepositoryMetaData(savedMetaData, currentContextName, md);
      }
      
      if(currentContextMetaData == null)
         throw new IllegalStateException("Could not create metadata");
      
      // Get the currentTimeMillis
      long lastModified = System.currentTimeMillis();
      
      // Get the parent ManagedCommon
      ManagedCommon parent = comp;
      while(parent.getParent() != null)
         parent = parent.getParent();

      // Get the managed object, as a component can also be a child of a managedObject
      ManagedObject managedObject = comp.getDeployment().getManagedObject(parent.getName());
      // Create a AttachmentMetaData for the MO
      if(managedObject != null)
      {
         String attachmentName = managedObject.getAttachmentName(); 
         // Create attachmentMetaData if needed
         AttachmentMetaData attachment = RepositoryAttachmentMetaDataFactory.findAttachment(attachmentName, currentContextMetaData.getAttachments());
         if(attachment == null)
         {
            // Add a new attachment
            attachment = new AttachmentMetaData();
            RepositoryAttachmentMetaDataFactory.addAttachment(currentContextMetaData, attachment);
         }
         
         // Is attachmentName the same as the className ?
         attachment.setName(attachmentName);
         attachment.setClassName(managedObject.getAttachment().getClass().getName());
         // Set attachment, this is transient - as it will get persisted in it's own file.
         attachment.setAttachment(managedObject);
         // Update lastModified
         currentContextMetaData.setLastModified(lastModified);
      }
      
      // Save the attachment for the root
      saveAttachmentMetaData(deploymentPath, savedMetaData);

      // Set the last modified on the root metadata
      savedMetaData.setLastModified(lastModified);
      
      // Save the repository meta data
      this.serializer.saveAttachment(getMetaDataPathName(deploymentPath), savedMetaData);
   }
   
   /**
    * Create the repository meta data from the parent ManagedDeployment.
    * 
    * @param parentMd the parent managed deployment
    * @return the RepositoryAttachmentMetaData with the structure information 
    * @throws Exception
    */
   protected RepositoryAttachmentMetaData createRepositoryMetaData(RepositoryAttachmentMetaData parentMetaData, String childPath, ManagedDeployment parentMd)
      throws Exception
   {
      // Child metadata 
      RepositoryAttachmentMetaData childMetaData = null;
      
      StructureMetaData structure = getStructureMetaData(parentMd.getName());
      if(structure == null)
         throw new IllegalStateException("Could not get the StructureMetaData.");
      
      List<ContextInfo> contextInfos = structure.getContexts();
      if(contextInfos == null)
         throw new IllegalStateException("StructureMetaData has no contexts."); // can this happen anyway ?
      
      // root context
      RepositoryAttachmentMetaDataFactory.applyStructureContext(parentMetaData, structure.getContext(""));
      // Other contexts
      for(ContextInfo info : contextInfos)
      {
         // If it is not the root path
         if(! "".equals(info.getPath()))
         {
            String childContextName = fixName(info.getPath());
            // TODO we might need to check the context itself contains a subdeployment? 
            
            RepositoryAttachmentMetaData newChild = RepositoryAttachmentMetaDataFactory.createInstance();
            newChild.setDeploymentName(childContextName);
            
            RepositoryAttachmentMetaDataFactory.applyStructureContext(newChild, info);
            RepositoryAttachmentMetaDataFactory.addChild(parentMetaData, newChild);
            
            if(childPath.equals(childContextName))
               childMetaData = newChild;
         }
      }
      
      if("".equals(childPath))
         childMetaData = parentMetaData;
      
      return childMetaData;
   }
   

   /**
    * Save the attachments based on the RepositoryAttachmentMetaData.
    * 
    * @param deploymentPath the deploymentPath
    * @param metaData the repository meta data.
    * @throws Exception
    */
   private void saveAttachmentMetaData(String deploymentPath, RepositoryAttachmentMetaData metaData)
      throws Exception
   {
      boolean trace = log.isTraceEnabled();
         
      // Save attachments for the root context
      if(metaData.getAttachments() != null && metaData.getAttachments().isEmpty() == false)
      {
         for(AttachmentMetaData attachment : metaData.getAttachments())
         {
            // Only save attachment if the attachment is present :)
            if(attachment.getAttachment() == null)
               continue;
            
            // Create xml meta data for persistence.
            ManagedObject mo = (ManagedObject) attachment.getAttachment();
            PersistedManagedObject root = createPersistedMetaData(mo);
            
            String attachmentPath = deploymentPath + attachment.getName();
            // Serialize the attachment
            serializer.saveAttachment(attachmentPath, root);
            
            if(trace)
               log.trace("Stored attachment to : " + attachmentPath);
         }
      }
      
      // Save the attachments for the childs
      if(metaData.getChildren() != null && metaData.getChildren().isEmpty() == false)
      {
         for(RepositoryAttachmentMetaData child : metaData.getChildren())
         {
            String childDeploymentPath = deploymentPath + File.separator + child.getDeploymentName() + File.separator;
            saveAttachmentMetaData(childDeploymentPath, child);
         }
      }
   }
   
   /**
    * create the xml meta data for persisting the managed object.
    * 
    * @param mo the managed object.
    * @return the xml metadata.
    */
   protected PersistedManagedObject createPersistedMetaData(ManagedObject mo)
   {
      // Return
      return handler.createPersistenceMetaData(mo);
   }
   
   /**
    * Rebuild the StructureMetaData based on the RepositoryAttachmentMetaData
    * and add predeterminedManagedObjects.
    * 
    * @param deployment the VFSDeployment
    * @param contextName the structure context path
    * @param deploymentPath the path to the attachment
    * @param attachmentMetaData the meta data
    */
   protected void rebuildStructureContext(Deployment deployment,
         String contextName,
         String deploymentPath,
         RepositoryAttachmentMetaData attachmentMetaData,
         boolean trace)
   {
      // The toplevel context
      boolean isRoot = "".equals(contextName);
      
      if(trace)
         log.trace("Rebuilding StructureMetaData for context: " + contextName);

      // Get the stored deployment structure
      DeploymentStructureMetaData structure = attachmentMetaData.getDeploymentStructure();
      
      // MetaData and ClassPath
      List<String> metaDataPaths = new ArrayList<String>();
      List<ClassPathEntry> classPath = new ArrayList<ClassPathEntry>();
      if(structure != null)
      {
         if(structure.getClassPaths() != null)
         {
            for(DeploymentClassPathMetaData md : structure.getClassPaths())
               classPath.add(deploymentFactory.createClassPathEntry(md.getPath(), md.getSuffixes()));
         }
         
         if(structure.getMetaDataPaths() != null)
            metaDataPaths = structure.getMetaDataPaths(); 
      }
      
      // Now create the ContextInfo
      ContextInfo info = deploymentFactory.addContext(deployment, contextName, metaDataPaths, classPath);
      if(structure != null)
      {
         // Set the comparator
         info.setComparatorClassName(structure.getComparatorClass());
         // Set the relative order
         info.setRelativeOrder(structure.getRelatativeOrder());
      }
      if(trace)
         log.trace("created ContextInfo: "+  info + " for deployment: "+ deployment);
      
      // Add attachments if needed 
      if(attachmentMetaData.getAttachments() != null && ! attachmentMetaData.getAttachments().isEmpty())
      {
         Set<String> availableAttachments = new HashSet<String>();
         for(AttachmentMetaData attachment : attachmentMetaData.getAttachments())
            availableAttachments.add(attachment.getClassName());

          MutableAttachments mutable =  createPredeterminedAttachment(deploymentPath, availableAttachments);
          
          // TODO is there a better way to do this ?
          if(isRoot)
          {
             deployment.setPredeterminedManagedObjects(mutable);
          }
          else
          {
             info.setPredeterminedManagedObjects(mutable);
          }
          
          if(trace)
             log.trace("Added PredetminedManagedObjects: " + availableAttachments + " to context " + contextName);
      }
      else
      {
         if(trace)
            log.trace("No PredetminedManagedObjects found for context " + contextName);         
      }
      
      // Process children
      List<RepositoryAttachmentMetaData> children = attachmentMetaData.getChildren(); 
      if(children != null && children.isEmpty() == false)
      {
         for(RepositoryAttachmentMetaData childMetaData : children)
         {
            // The structure context path
            String childContextName = contextName + "/" + childMetaData.getDeploymentName();
            // The relative path of the child attachment (therefore File.separator) 
            String relativePath = deploymentPath + childMetaData.getDeploymentName() + File.separator;
            
            if(trace)
               log.trace("Processing child context: "+ childContextName);
            
            // Rebuild the structure of the child
            rebuildStructureContext(deployment, fixName(childContextName), relativePath, childMetaData, trace);
         }
      }
   }
   
   /**
    * Create the relative path to the persisted deployment attachment meta data.
    * The string is simpleName + "-" + hash (based on the URI of the deployment)
    * 
    * @param deployment the deployment
    * @return the relative name
    * @throws Exception
    */
   protected String createRelativeDeploymentPath(ProfileDeployment deployment) throws Exception
   {
      if(deployment == null)
         throw new IllegalStateException("Null deployment.");
      
      // deployment URI toString
      String pathName = deployment.getRoot().toURI().toString();
      String fileName = deployment.getRoot().getName();
      // Generate hash
      String hash = HashGenerator.createHash(pathName);
      // simple name + "-" + hash
      return fileName + "-" + hash + File.separator;
      
   }
   
   /**
    * Create a predetermined managedObject attachment.
    * 
    * @param deploymentPath the relative deployment path
    * @param availableAttachments the available attachments
    * @return
    */
   protected MutableAttachments createPredeterminedAttachment(String deploymentPath, Set<String> availableAttachments)
   {
      return new LazyPredeterminedManagedObjects(this.serializer, deploymentPath, availableAttachments);
   }
   
   /**
    * Get the metadata path, based on a relative path.
    * 
    * @param deploymentPath the relative path to the deployment
    * @return
    */
   protected String getMetaDataPathName(String deploymentPath)
   {
      return deploymentPath.endsWith(File.separator) ? deploymentPath + METADATA_NAME : deploymentPath + File.separator + METADATA_NAME;
   }
   
   /**
    * Load the attachment metadata for a deployment.
    * 
    * @param relativeDeploymentPath the relative path
    * @return the attachment metadata
    */
   protected RepositoryAttachmentMetaData loadAttachmentMetaData(String relativeDeploymentPath)
   {
      // attachments/simpleName - hash/metadata.xml
      String fixedMetadataPath = getMetaDataPathName(relativeDeploymentPath);   
               
      try
      {  // Try to load the repository attachment metadata
         return this.serializer.loadAttachment(fixedMetadataPath, RepositoryAttachmentMetaData.class);
      }
      catch(Exception e)
      {
         log.error("Failed to load attachment metadata from relative path: "+ relativeDeploymentPath, e);
      }
      return null;
   }
   
   
   /**
    * Get the structure meta data for a deployment.
    * 
    * @param vfsDeploymentName the vfs deployment name
    * @return the StructureMetaData
    * @throws Exception
    */
   protected StructureMetaData getStructureMetaData(String vfsDeploymentName)
   {
      // Get the StructureMetaData;
      DeploymentContext deploymentContext = getDeploymentContext(vfsDeploymentName);
      if(deploymentContext == null)
         throw new IllegalStateException("Could not find deployment context for name: "+ vfsDeploymentName);
      
      return deploymentContext.getDeploymentUnit().getAttachment(StructureMetaData.class);
    }
   
   /**
    * Get deployment context.
    *
    * @param name the deployment context name
    * @return vfs deployment context or null if doesn't exist or not vfs based
    */
   @SuppressWarnings("deprecation")
   protected VFSDeploymentContext getDeploymentContext(String name)
   {
      if (mainDeployer == null)
         throw new IllegalStateException("Null main deployer.");

      DeploymentContext deploymentContext = mainDeployer.getDeploymentContext(name);
      if (deploymentContext == null || deploymentContext instanceof VFSDeploymentContext == false)
         return null;

      return (VFSDeploymentContext)deploymentContext;
   }
   
   /**
    * Make sure that the name does not start or end with /
    * 
    * @param name
    * @return
    */
   private String fixName(String name)
   {
      if(name == null)
         return null;
      
      if(name.equals(""))
         return name;
      
      if(name.startsWith("/"))
         name = name.substring(1);
      
      if(name.endsWith("/"))
         name = name.substring(0, name.length() -1);
      
      return name;
   }
   
   private static class HashGenerator
   {
      /** The digest. */
      private static MessageDigest digest;
      
      /**
       * Create a hash based on a deployment vfs path name.
       * 
       * @param deployment the deployment
       * @return a hash
       * @throws NoSuchAlgorithmException
       * @throws MalformedURLException
       * @throws URISyntaxException
       */
      public static String createHash(String pathName)
            throws NoSuchAlgorithmException, MalformedURLException, URISyntaxException
      {
         // buffer
         StringBuffer buffer = new StringBuffer();
         // formatter
         Formatter f = new Formatter(buffer);
         // get the bytez
         byte[] bytez = internalCreateHash(pathName);
         for(byte b : bytez)
         {
            // format the byte
            f.format("%02x", b);
         }
         // toString
         return f.toString();
      }
      
      protected static byte[] internalCreateHash(String pathName) throws NoSuchAlgorithmException
      {
         MessageDigest digest = getDigest();
         try
         {
            // update
            digest.update(pathName.getBytes());
            // return
            return digest.digest();
         }
         finally
         {
            // reset
            digest.reset();
         }
      }
      
      public static MessageDigest getDigest() throws NoSuchAlgorithmException
      {
         if(digest == null)
            digest = MessageDigest.getInstance("MD5");

         return digest;
      }
   }
   
}
