/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.persistence;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.jboss.beans.info.spi.PropertyInfo;
import org.jboss.logging.Logger;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.annotation.ViewUse;
import org.jboss.metatype.api.types.ArrayMetaType;
import org.jboss.metatype.api.types.CompositeMetaType;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.values.ArrayValue;
import org.jboss.metatype.api.values.CollectionValue;
import org.jboss.metatype.api.values.CompositeValue;
import org.jboss.metatype.api.values.EnumValue;
import org.jboss.metatype.api.values.GenericValue;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.MetaValueFactory;
import org.jboss.metatype.api.values.PropertiesMetaValue;
import org.jboss.metatype.api.values.SimpleValue;
import org.jboss.metatype.api.values.TableValue;
import org.jboss.system.server.profileservice.persistence.xml.ModificationInfo;
import org.jboss.system.server.profileservice.persistence.xml.NullValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedArrayValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedCollectionValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedCompositeValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedEnumValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedGenericValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedManagedObject;
import org.jboss.system.server.profileservice.persistence.xml.PersistedPair;
import org.jboss.system.server.profileservice.persistence.xml.PersistedPropertiesValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedProperty;
import org.jboss.system.server.profileservice.persistence.xml.PersistedSimpleValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedTableValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedValue;

/**
 * Create a xml representation of a Managed Object.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision: 85730 $
 */
public class ManagedObjectPeristenceHandler
{

   /** The meta value factory. */
   private MetaValueFactory metaValueFactory = MetaValueFactory.getInstance();
   
   /** The logger. */
   private static final Logger log = Logger.getLogger(ManagedObjectPeristenceHandler.class);
 
   /**
    * Create the persistence meta data for a ManagedObject.
    * 
    * @param mo the managed object
    * @return the persistence meta data
    */
   public PersistedManagedObject createPersistenceMetaData(ManagedObject mo)
   {
      if(mo == null)
         throw new IllegalArgumentException("Null managedObject.");
      
      PersistedManagedObject moElement = new PersistedManagedObject();
      processManagedObject(moElement, mo);
      return moElement;
   }
   
   /**
    * Process a managed object.
    * 
    * @param moElement the xml meta data.
    * @param mo the managed object
    * @return isModified
    */
   public void processManagedObject(PersistedManagedObject moElement, ManagedObject mo)
   {
      if(moElement == null)
         throw new IllegalArgumentException("Null rootElement.");
      if(mo == null)
         throw new IllegalArgumentException("null managedObject");
      
      boolean trace = log.isTraceEnabled();
      
      // Set the template and class-name 
      String className = mo.getAttachmentName();
      if(mo.getAttachment() != null)
      {
         Class<?> attachment = mo.getAttachment().getClass();
         className = attachment.getName();
         // Set the template name
         if(className.equals(mo.getAttachmentName()) == false)
         {
            // If the MO template is different from the actual attachment
            moElement.setTemplateName(mo.getAttachmentName());
         }
      }

      // Set the managed-object meta information
      moElement.setName(mo.getName());
      moElement.setClassName(className);
      
      // Process the properties
      List<PersistedProperty> properties = moElement.getProperties();
      if(properties == null)
      {
         properties = new ArrayList<PersistedProperty>();
         moElement.setProperties(properties);
      }
      for(String propertyName : mo.getPropertyNames())
      {
         ManagedProperty property = mo.getProperty(propertyName);
         if(property == null)
            throw new IllegalStateException("unable to find propery: "+ property);
         
         if(! property.hasViewUse(ViewUse.CONFIGURATION))
            continue;
         
         // getProperty
         PropertyInfo propertyInfo = property.getField(Fields.PROPERTY_INFO, PropertyInfo.class);
         // Skip not readable properties
         if(propertyInfo != null && propertyInfo.isReadable() == false)
         {
            if(trace)
               log.trace("skipping property "+ propertyName + "is not readable");
            continue;
         }
         
         PersistedProperty propertyElement = createPersistedProperty(
               propertyName,
               property.getValue(),
               property.getMetaType(), trace);
      
         properties.add(propertyElement);
      }
   }
   
   /**
    * Create the persisted property.
    * 
    * @param name the property name
    * @param value the meta value
    * @param metaType the meta type
    * @return the property
    */
   protected PersistedProperty createPersistedProperty(String name, MetaValue value, MetaType metaType, boolean trace)
   {
      if(name == null)
         throw new IllegalArgumentException("Null name.");
      if(metaType == null)
         throw new IllegalArgumentException("Null meta type.");
    
      if(trace)
         log.trace("processing property name="+ name + "[metaValue="+ value + ", metaType="+ metaType + "]" );
      
      // Create Property
      PersistedProperty property = new PersistedProperty(name);
      // Create persisted value
      PersistedValue persistedValue = createPersistedValue(value, metaType);
      persistedValue.setModificationInfo(ModificationInfo.MODIFIED);
      //
      property.setValue(persistedValue);
      // Return
      return property;
   }
   
   /**
    * Create the peristed xml meta data.
    * 
    * @param value the meta value
    * @param metaType the meta type
    * @return the xml value.
    */
   protected PersistedValue createPersistedValue(MetaValue value, MetaType metaType)
   {
      if(log.isTraceEnabled())
         log.trace("creating persisted value for : " + value + " with metaType " + metaType);
      
      if(value == null)
         return new NullValue();
      
      // Override the metaType e.g. the MapCompositeValueSupport
      metaType = value.getMetaType();
      
      PersistedValue persistedValue = null;
      if(metaType.isSimple())
      {
         persistedValue = createSimpleValue(
               (SimpleValue) value);
      }
      else if(metaType.isEnum())
      {
         persistedValue = createEnumValue(
               (EnumValue) value);
      }
      else if(metaType.isCollection())
      {
         persistedValue = createCollectionValue(
               (CollectionValue) value);
      }
      else if(metaType.isGeneric())
      {
         persistedValue = createGenericValue(
               (GenericValue) value);
      }
      else if(metaType.isComposite())
      {
         persistedValue = createCompositeValue(
               (CompositeValue) value,
               (CompositeMetaType) metaType);
      }
      else if(metaType.isArray())
      {
         persistedValue = createArrayValue(
               (ArrayValue) value,
               (ArrayMetaType) metaType);
      }
      else if(metaType.isTable())
      {
         persistedValue = createTableValue(
               (TableValue) value);
      }
      else if(metaType.isProperties())
      {
         persistedValue = createPropertiesValue(
               (PropertiesMetaValue) value);
      }
      else
      {
         throw new IllegalStateException("unknown metaType");
      }
      return persistedValue;
   }
   

   /**
    * Create the persistence enum value.
    * 
    * @param value the enum value
    * @return the enum xml meta data
    */
   private PersistedEnumValue createEnumValue(EnumValue value)
   {
      PersistedEnumValue persistedValue = new PersistedEnumValue();
      persistedValue.setValue(value.getValue());
      return persistedValue;
   }

   /**
    * Create the persistence simple value.
    * 
    * @param value the simple value
    * @return the simple xml meta data
    */
   private PersistedSimpleValue createSimpleValue(SimpleValue value)
   {
      PersistedSimpleValue persistedValue = new PersistedSimpleValue();
      persistedValue.setValue(convertSimple2String(value));
      return persistedValue;
   }
   
   /**
    * Create the persistence collection value.
    * 
    * @param value the collection value
    * @return the collection xml meta data
    */
   private PersistedCollectionValue createCollectionValue(CollectionValue value)
   {
      PersistedCollectionValue collection = new PersistedCollectionValue();
      for(MetaValue child : value.getElements())
      {
         PersistedValue persistedValue = createPersistedValue(child, child.getMetaType());
         collection.addValue(persistedValue);
      }
      return collection;
   }

   /**
    * Create the persistence generic value.
    * 
    * @param value the generic value
    * @return the generic xml meta data
    */
   private PersistedGenericValue createGenericValue(GenericValue value)
   {
      //
      PersistedGenericValue generic = new PersistedGenericValue();
      Object o = value.getValue();
      if(o == null)
         return generic;
      
      if(o instanceof ManagedObject)
      {
         PersistedManagedObject mo = createPersistenceMetaData((ManagedObject) o);
         generic.setManagedObject(mo);
      }
      else
      {
         throw new IllegalStateException("The value of GenericValue must be a ManagedObject: " + value);
      }
      return generic;
   }

   /**
    * Create the persistence array value.
    * 
    * @param value the array value
    * @return
    */
   private PersistedArrayValue createArrayValue(ArrayValue value, ArrayMetaType metaType)
   {
      //
      PersistedArrayValue array = new PersistedArrayValue();
      MetaType elementType = metaType.getElementType();
      for (int i = 0; i < value.getLength(); i++)
      {
         PersistedValue persistedValue = null;
         Object subElement = value.getValue(i);

         if (subElement instanceof MetaValue)
         {
            persistedValue = createPersistedValue((MetaValue) subElement, elementType);
         }
         else if (subElement != null && subElement.getClass().isArray())
         {
            persistedValue = unwrapArray(array, subElement, elementType);
         }
         // Add to parent
         array.addValue(persistedValue);
      }
      return array;
   }
   
   /**
    * Unwrap array.
    * 
    * @param array the parent array
    * @param element the array value
    * @param type the element meta type
    * @return the persistence xml meta data
    */
   protected PersistedArrayValue unwrapArray(PersistedArrayValue array, Object element, MetaType type)
   {
      PersistedArrayValue newElement = new PersistedArrayValue();
      int subSize = Array.getLength(element);
      for (int i = 0; i < subSize; i++)
      {
         PersistedValue persistedValue = null;
         Object subElement = Array.get(element, i);
         if (subElement instanceof MetaValue)
         {
            persistedValue = createPersistedValue((MetaValue) subElement, type);
         }
         else if (subElement != null && subElement.getClass().isArray())
         {
            persistedValue = unwrapArray(newElement, subElement, type);
         }

         newElement.addValue(persistedValue);
      }
      return newElement;
   }

   /**
    * Create the persistence composite value.
    * 
    * @param value the composite value
    * @param metaType the composite meta type
    * @return the persistence composite xml meta data
    */
   private PersistedCompositeValue createCompositeValue(CompositeValue value, CompositeMetaType metaType)
   {
      //
      PersistedCompositeValue composite = new PersistedCompositeValue();
      // Fix the values
      List<PersistedValue> values = composite.getValues();
      if(values == null)
      {
         values = new ArrayList<PersistedValue>();
         composite.setValues(values);
      }
      for(String item : metaType.itemSet())
      {
         MetaType itemType = metaType.getType(item);
         MetaValue itemValue = value.get(item);
         
         // Create item 
         PersistedValue persistedValue = createPersistedValue(itemValue, itemType);
         persistedValue.setName(item);
         
         values.add(persistedValue);
      }
      return composite;
   }
   
   /**
    * Create the persistence table value.
    * 
    * @param value the table value
    * @return the persistence table xml meta data
    */
   private PersistedTableValue createTableValue(TableValue value)
   {
      PersistedTableValue table = new PersistedTableValue();
      // Fix the entries
      List<PersistedCompositeValue> entries = table.getEntries();
      if(entries == null)
      {
         entries = new ArrayList<PersistedCompositeValue>();
         table.setEntries(entries);
      }
      // Process values
      Collection<CompositeValue> values = value.values();
      for(CompositeValue entry : values)
      {
         entries.add(createCompositeValue(entry, entry.getMetaType()));
      }
      return table;
   }

   private PersistedValue createPropertiesValue(PropertiesMetaValue value)
   {
      PersistedPropertiesValue properties = new PersistedPropertiesValue();
      List<PersistedPair> pairs = properties.getEntries();
      if(pairs == null)
      {
         pairs = new ArrayList<PersistedPair>();
         properties.setEntries(pairs);
      }
      for(Object key : value.keySet())
      {
         Object kvalue = value.get(key);
         PersistedPair pair = new PersistedPair(key.toString(), kvalue.toString());
         pairs.add(pair);
      }
      return properties;
   }

   /**
    * Create a emtpy xml meta data, based on the meta type
    * 
    * @param metaType the meta type
    * @return the peristence value
    */
   protected static PersistedValue emtpyPersistedValue(MetaType metaType)
   {
      if(metaType.isSimple())
      {
         return new PersistedSimpleValue(); 
      }
      else if(metaType.isEnum())
      {
         return new PersistedEnumValue();
      }
      else if(metaType.isCollection())
      {
         return new PersistedCollectionValue();
      }
      else if(metaType.isGeneric())
      {
         return new PersistedGenericValue();
      }
      else if(metaType.isComposite())
      {
         return new PersistedCompositeValue();
      }
      else if(metaType.isTable())
      {
         return new PersistedTableValue();
      }
      else if(metaType.isArray())
      {
         return new PersistedArrayValue();
      }
      else
      {
         throw new IllegalStateException("unknown metaType");
      }
   }
   
   /**
    * Convert a simple meta value to a String.
    * 
    * @param value the simple meta value.
    * @return the string.
    */
   protected String convertSimple2String(SimpleValue value)
   {       
      if(value == null)
         throw new IllegalArgumentException("Null value.");
      
      Object unwrappedValue = metaValueFactory.unwrap(value);
      if(unwrappedValue == null)
         return null; 
      // Convert to String
      return ("" + unwrappedValue);
   }
}
