/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.persistence;

import org.jboss.logging.Logger;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.values.ArrayValue;
import org.jboss.metatype.api.values.CollectionValue;
import org.jboss.metatype.api.values.CompositeValue;
import org.jboss.metatype.api.values.GenericValue;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.TableValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedGenericValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedManagedObject;
import org.jboss.system.server.profileservice.persistence.xml.PersistedProperty;
import org.jboss.system.server.profileservice.persistence.xml.PersistedValue;

/**
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class ManagedObjectOverride extends ManagedObjectRecreation
{

   /** The logger. */
   protected static final Logger log = Logger.getLogger(ManagedObjectOverride.class);

   /** The attachment property populator. */
   private AttachmentPropertyPopulator populator = new AttachmentPropertyPopulator();

   public ManagedObject update(Object attachment, PersistedManagedObject element) throws Throwable
   {
      if (attachment == null)
         throw new IllegalArgumentException("Null attachment");
      if (element == null)
         throw new IllegalArgumentException("Null persisted information.");

      // The managed object
      ManagedObject mo = buildManagedObject(attachment);
      
      //
      if(mo == null)
         return null;
      
      // Update
      updateManagedObject(mo, element, attachment);

      return mo;
   }

   public void updateManagedObject(ManagedObject mo, PersistedManagedObject element, Object attachment)
         throws Throwable
   {
      if(mo == null)
         throw new IllegalArgumentException("Null managed object.");
      if(element == null)
         throw new IllegalArgumentException("Null persisted information.");
      // 
      if(attachment == null)
         attachment = mo.getAttachment();
      if(attachment == null)
         throw new IllegalArgumentException("Null attachment.");
      
      boolean trace = log.isTraceEnabled();
      
      // Process properties
      for (PersistedProperty persisted : element.getProperties())
      {
         ManagedProperty property = mo.getProperty(persisted.getName());
         // Process
         processProperty(property, persisted, attachment, trace);
      }
   }

   protected void processProperty(ManagedProperty property, PersistedProperty propertyElement, Object attachment,
         boolean trace) throws Throwable
   {
      if (property == null)
         throw new IllegalStateException("Null property");

      // Get the original values
      MetaValue metaValue = property.getValue();
      MetaType metaType = property.getMetaType();
      if (metaValue != null)
         metaType = metaValue.getMetaType();

      //
      PersistedValue valueElement = propertyElement.getValue();
      if (valueElement.getModificationInfo() == null)
      {
         // TODO 
      }
      
      // Recreate the metaValue
      MetaValue override = createMetaValue(valueElement, metaType);

      // Process
      MetaValue value = processMetaValue(metaValue, override, trace);
      
      //
      setValue(property, value, attachment);
   }

   protected void setValue(ManagedProperty property, MetaValue value, Object attachment)
         throws Throwable
   {
      // Skip null metaValues
      if(value == null)
         return;

      // Set the values 
      property.setField(Fields.VALUE, value);
      // Update meta type
      property.setField(Fields.META_TYPE, value.getMetaType());
      
      // Populate the values
      populator.processManagedProperty(property.getName(), property, attachment);
   }

   protected MetaValue processMetaValue(MetaValue original, MetaValue override, boolean trace) throws Throwable
   {
      if (override == null)
         return original;
      if (original == null)
         return override;

      if (original.getMetaType() == null)
         throw new IllegalStateException("Original meta value is null " + original);
         

      // Now try to merge
      MetaType metaType = original.getMetaType();
      MetaValue metaValue = null;
      if (metaType.isSimple())
      {
         metaValue = override;
      }
      else if (metaType.isEnum())
      {
         metaValue = override;
      }
      else if (metaType.isCollection())
      {
         metaValue = mergeCollection(
               (CollectionValue) original,
               (CollectionValue) override, trace);
      }
      else if (metaType.isGeneric())
      {
         metaValue = mergeGenericValue(
               (GenericValue) original,
               (GenericValue) override, trace);
      }
      else if (metaType.isComposite())
      {
         metaValue = mergeCompositeValue(
               (CompositeValue) original,
               (CompositeValue) override, trace);
      }
      else if (metaType.isTable())
      {
         metaValue = mergeTableValue(
               (TableValue) original,
               (TableValue) override, trace);
      }
      else if (metaType.isArray())
      {
         metaValue = mergeArray(
               (ArrayValue) original,
               (ArrayValue) override, trace);
      }
      else
      {
         throw new IllegalStateException("unknown metaType");
      }
      return metaValue;
   }

   /**
    * Merge a composite value.
    * 
    * @param original
    * @param override
    * @return
    */
   protected CompositeValue mergeCompositeValue(CompositeValue original, CompositeValue override, boolean trace)
   {
      return override;
   }

   /**
    * Merge a table
    * 
    * @param original
    * @param override
    * @return
    */
   protected TableValue mergeTableValue(TableValue original, TableValue override, boolean trace)
   {
      return override;
   }

   /**
    * Create a merged array value.
    * 
    * @param original the original array
    * @param override the override array
    * @return the merged array
    */
   protected ArrayValue mergeArray(ArrayValue original, ArrayValue override, boolean trace)
   {
      return override;
   }

   /**
    * Create a merged collection value.
    * 
    * @param original the original collection
    * @param override the override collection
    * @return the merged collection
    * @throws Throwable 
    */
   protected CollectionValue mergeCollection(CollectionValue original, CollectionValue override, boolean trace)
         throws Throwable
   {
      return override;
   }

   /**
    * Merge a generic value.
    * 
    * @param original the original generic value
    * @param override the override generic value
    * @return the merged value
    */
   protected GenericValue mergeGenericValue(GenericValue original, GenericValue override, boolean trace)
         throws Throwable
   {
      Object v = override.getValue();
      if (v instanceof PersistedGenericValue)
      {
         PersistedGenericValue moElement = (PersistedGenericValue) override.getValue();

         return processGenericValue(original, moElement, trace);
      }
      // Don't do anything
      return null;
   }

   protected GenericValue processGenericValue(GenericValue original, PersistedGenericValue override, boolean trace)
         throws Throwable
   {
      // Don't do anything
      if(override.getManagedObject() == null)
         return null;
      
      Object o = original.getValue();
      if ((o instanceof ManagedObject))
      {
         ManagedObject originalMO = (ManagedObject) o;

         // process
         updateManagedObject(originalMO, override.getManagedObject(), originalMO.getAttachment());
         return original;
      }
      throw new RuntimeException("Cannot merge original: " + original + ", override: " + override);
   }

}
