/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.repository;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.SyncFailedException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.zip.ZipInputStream;

import org.jboss.deployers.spi.attachments.Attachments;
import org.jboss.deployers.structure.spi.DeploymentContext;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentContext;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.DeploymentContentFlags;
import org.jboss.profileservice.spi.DeploymentRepository;
import org.jboss.profileservice.spi.ModificationInfo;
import org.jboss.profileservice.spi.NoSuchDeploymentException;
import org.jboss.profileservice.spi.NoSuchProfileException;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ModificationInfo.ModifyStatus;
import org.jboss.util.file.Files;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VFSUtils;
import org.jboss.virtual.VirtualFile;
import org.jboss.virtual.VirtualFileFilter;

/**
 * An implementation of DeploymentRepository that relies on java
 * serialization to store contents on the file system.
 * 
 * + root/{name}/bootstrap - the bootstrap beans
 * + root/{name}/deployers - profile deployers
 * + root/{name}/deploy - installed deployments
 * + root/{name}/apps - post install deployments
 * + root/{name}/attachments - pre-processed attachments + admin edits to deployments
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 81888 $
 */
public class SerializableDeploymentRepository extends AbstractAttachmentStore
   implements DeploymentRepository
{
   private static final Logger log = Logger.getLogger(SerializableDeploymentRepository.class);

   /** The server root container the deployments */
   private File root;
   /** The bootstrap descriptor dir */
   private File bootstrapDir;
   /** The server static libraries */
   private File libDir;
   /** The deployers phase deployments dir */
   private File deployersDir;
   /** The application phase deployments dir */
   private File[] applicationDirs;
   /** The application phase deployment files keyed by VirtualFile URI string */
   private final Map<String, VirtualFile> applicationVFCache = new HashMap<String, VirtualFile>();
   /** The deployment post edit root location */
   private File adminEditsRoot;
   /** The profile key this repository is associated with */
   private ProfileKey key;
   /** The bootstrap VFSDeployments */
   private LinkedHashMap<String,VFSDeployment> bootstrapCtxs = new LinkedHashMap<String,VFSDeployment>();
   /** The deployer VFSDeployments */
   private LinkedHashMap<String,VFSDeployment> deployerCtxs = new LinkedHashMap<String,VFSDeployment>();
   /** The application VFSDeployments */
   private LinkedHashMap<String,VFSDeployment> applicationCtxs = new LinkedHashMap<String,VFSDeployment>();

   private Map<String, Integer> contentFlags = new ConcurrentHashMap<String, Integer>();
   /** The last time the profile was modified */
   private long lastModified;
   /** A lock for the hot deployment/{@link #getModifiedDeployments()} */
   private ReentrantReadWriteLock contentLock = new ReentrantReadWriteLock(true);
   /** Should an attempt to overwrite existing content fail in {@link #addDeploymentContent(String, ZipInputStream, DeploymentPhase)}*/
   private boolean failIfAlreadyExists = false;
   
   /** Allowed deployments filter */
   private VirtualFileFilter deploymentFilter;
   /** The metadata include filter */
   private VirtualFileFilter hotDeploymentFilter;
   /** The last modified cache */
   private Map<String, Long> lastModifiedCache;

   public SerializableDeploymentRepository(File root, URI[] appURIs, ProfileKey key)
   {
      this.root = root;
      this.key = key;
      this.setApplicationURIs(appURIs);
      this.lastModifiedCache = new ConcurrentHashMap<String, Long>();
   }

   public URI[] getApplicationURIs()
   {
      URI[] appURIs = new URI[applicationDirs.length];
      for (int n = 0; n < applicationDirs.length; n ++)
      {
         File applicationDir = applicationDirs[n];
         appURIs[n] = applicationDir.toURI();
      }
      return appURIs;
   }
   public void setApplicationURIs(URI[] uris)
   {
      applicationDirs = new File[uris.length];
      for (int n = 0; n < uris.length; n ++)
      {
         URI uri = uris[n];
         applicationDirs[n] = new File(uri);
      }
   }

   public boolean exists()
   {
      File profileRoot = new File(root, key.getName());
      return profileRoot.exists();
   }

   public String getAttachmentsRoot()
   {
      String path = null;
      if(adminEditsRoot != null)
         path = adminEditsRoot.getAbsolutePath();
      return path;
   }
   public void setAttachmentsRoot(String root)
   {
      adminEditsRoot = new File(root);
   }
   
   public VirtualFileFilter getDeploymentFilter()
   {
      return deploymentFilter;
   }
   
   public void setDeploymentFilter(VirtualFileFilter deploymentFilter)
   {
      this.deploymentFilter = deploymentFilter;
   }
   
   public VirtualFileFilter getHotDeploymentFilter()
   {
      return hotDeploymentFilter;
   }
   
   public void setHotDeploymentFilter(VirtualFileFilter hotDeploymentFilter)
   {
      this.hotDeploymentFilter = hotDeploymentFilter;
   }

   public long getLastModified()
   {
      return this.lastModified;
   }

   public URI getDeploymentURI(DeploymentPhase phase)
   {
      URI uri = null;
      switch( phase )
      {
         case BOOTSTRAP:
            uri = this.getBootstrapURI();
            break;
         case DEPLOYER:
            uri = this.getDeployersURI();
            break;
         case APPLICATION:
            uri = this.getApplicationURI();
            break;
      }
      return uri;
   }

   public void setDeploymentURI(URI uri, DeploymentPhase phase)
   {
      switch( phase )
      {
         case BOOTSTRAP:
            this.setBootstrapURI(uri);
            break;
         case DEPLOYER:
            this.setDeployersURI(uri);
            break;
         case APPLICATION:
            this.setApplicationURIs(new URI[]{uri});
            break;
      }
   }
   public Set<String> getDeploymentNames()
   {
      HashSet<String> names = new HashSet<String>();
      names.addAll(bootstrapCtxs.keySet());
      names.addAll(deployerCtxs.keySet());
      names.addAll(applicationCtxs.keySet());
      return names;
   }
   public Set<String> getDeploymentNames(DeploymentPhase phase)
   {
      HashSet<String> names = new HashSet<String>();
      switch( phase )
      {
         case BOOTSTRAP:
            names.addAll(this.bootstrapCtxs.keySet());
            break;
         case DEPLOYER:
            names.addAll(this.deployerCtxs.keySet());
            break;
         case APPLICATION:
            names.addAll(this.applicationCtxs.keySet());
            break;
      }
      return names;      
   }

   /**
    * 
    */
   public String addDeploymentContent(String name, InputStream contentIS, DeploymentPhase phase)
      throws IOException
   {
      boolean trace = log.isTraceEnabled();
      // Suspend hot deployment checking
      if( trace )
         log.trace("Aquiring content read lock");
      contentLock.writeLock().lock();
      String repositoryName = null;
      try
      {
         // Write the content out
         File contentRoot = getPhaseDir(phase);
         if(contentRoot == null)
            throw new FileNotFoundException("Failed to obtain content dir for phase: "+phase);
         File contentFile = new File(contentRoot, name);
         // TODO do we need to create the parent directory if the name is myDeployments/myRealDeployment.jar ?
         
         if(failIfAlreadyExists && contentFile.exists())
            throw new SyncFailedException("Deployment content already exists: "+contentFile.getAbsolutePath());
         FileOutputStream fos = new FileOutputStream(contentFile);
         try
         {
            byte[] tmp = new byte[4096];
            int read;
            while((read = contentIS.read(tmp)) > 0)
            {
               if (trace)
                  log.trace("write, " + read);
               fos.write(tmp, 0, read);
            }
            fos.flush();
            // Get the vfs uri for the content as the repository name
            VirtualFile contentVF = VFS.getVirtualFile(contentRoot.toURI(), name);
            repositoryName = contentVF.getName();
         }
         finally
         {
            try
            {
               fos.close();
            }
            catch (IOException ignored)
            {
            }
         }

         // Lock the content
         lockDeploymentContent(name, phase);
      }
      finally
      {
         // Allow hot deployment checking
         contentLock.writeLock().unlock();
         if(trace)
            log.trace("Released content write lock");
      }
      return repositoryName;
   }

   /**
    * @throws URISyntaxException 
    * 
    */
   public VirtualFile getDeploymentContent(String name, DeploymentPhase phase)
         throws IOException, URISyntaxException
   {
      // find the cached root
      switch( phase )
      {
         case BOOTSTRAP:
         case DEPLOYER:
            throw new IOException(phase+" content access not supported");
         case APPLICATION:
         case APPLICATION_TRANSIENT:
            break;
      }
      // get cached version
      File contentRoot = getPhaseDir(phase);
      if(contentRoot == null)
         throw new FileNotFoundException("Failed to obtain content dir for phase: "+phase);
      File contentFile = new File(contentRoot, name);
      VirtualFile content = getCachedApplicationVF(contentFile);
      if(content == null || content.exists() == false)
         throw new FileNotFoundException("Failed to find content for: "+name);
      return content;
   }

   public String[] getRepositoryNames(String[] names, DeploymentPhase phase) throws IOException
   {
      List<String> tmp = new ArrayList<String>();
      URI rootURI = getDeploymentURI(phase);
      VirtualFile root = VFS.getRoot(rootURI);
      for(String name : names)
      {
         VirtualFile content = root.getChild(name);
         try
         {
            String rname = content.toURI().toString();
            tmp.add(rname);
         }
         catch (URISyntaxException e)
         {
            log.error("Should not happen", e);
         }
      }
      String[] rnames = new String[tmp.size()];
      tmp.toArray(rnames);
      return rnames;
   }

   public int lockDeploymentContent(String vfsPath, DeploymentPhase phase)
   {
      if( log.isTraceEnabled() )
         log.trace("lockDeploymentContent, "+vfsPath); 
      int flags = setDeploymentContentFlags(vfsPath, phase, DeploymentContentFlags.LOCKED);
      return flags;
   }

   public int unlockDeploymentContent(String vfsPath, DeploymentPhase phase)
   {
      if( log.isTraceEnabled() )
         log.trace("unlockDeploymentContent, "+vfsPath); 
      int flags = clearDeploymentContentFlags(vfsPath, phase, DeploymentContentFlags.LOCKED);
      return flags;
   }

   public int getDeploymentContentFlags(String vfsPath, DeploymentPhase phase)
   {
      Integer flags = contentFlags.get(vfsPath);
      int iflags = flags != null ? flags.intValue() : 0;
      return iflags;
   }
   public synchronized int clearDeploymentContentFlags(String vfsPath,
         DeploymentPhase phase,
         int flags)
   {
      Integer dflags = contentFlags.get(vfsPath);
      if(dflags != null)
      {
         dflags &= ~flags;
         contentFlags.put(vfsPath, dflags);
      }
      int iflags = dflags != null ? dflags.intValue() : 0;
      return iflags;
   }
   public boolean hasDeploymentContentFlags(String vfsPath, DeploymentPhase phase,
         int flag)
   {
      Integer flags = contentFlags.get(vfsPath);
      boolean hasFlag = false;
      if(flags != null )
         hasFlag = (flags & flag) != 0 ? true : false;
      return hasFlag;
   }
   public int setDeploymentContentFlags(String vfsPath, DeploymentPhase phase,
         int flags)
   {
      contentFlags.put(vfsPath, flags);
      return flags;
   }

   public void acquireDeploymentContentLock()
   {
      contentLock.writeLock().lock();
      if( log.isTraceEnabled() )
         log.trace("acquireDeploymentContentLock, have write lock"); 
   }
   public void releaseDeploymentContentLock()
   {
      contentLock.writeLock().unlock();
      if( log.isTraceEnabled() )
         log.trace("releaseDeploymentContentLock, gave up write lock");       
   }

   public void addDeployment(String vfsPath, VFSDeployment d, DeploymentPhase phase)
      throws Exception
   {
      switch( phase )
      {
         case BOOTSTRAP:
            this.addBootstrap(vfsPath, d);
            break;
         case DEPLOYER:
            this.addDeployer(vfsPath, d);
            break;
         case APPLICATION:
            this.addApplication(vfsPath, d);
            break;
         case APPLICATION_TRANSIENT:
            this.addApplication(vfsPath, d);
            break;
      }
      lastModified = System.currentTimeMillis();
   }

   public VFSDeployment getDeployment(String name, DeploymentPhase phase)
      throws Exception, NoSuchDeploymentException
   {
      VFSDeployment ctx = null;
      if( phase == null )
      {
         // Try all phases
         try
         {
            ctx = this.getBootstrap(name);
         }
         catch(NoSuchDeploymentException ignore)
         {
         }
         try
         {
            if( ctx == null )
               ctx = this.getDeployer(name);
         }
         catch(NoSuchDeploymentException ignore)
         {
         }
         try
         {
            if( ctx == null )
               ctx = this.getApplication(name);
         }
         catch(NoSuchDeploymentException ignore)
         {
         }
      }
      else
      {
         switch( phase )
         {
            case BOOTSTRAP:
               ctx = this.getBootstrap(name);
               break;
            case DEPLOYER:
               ctx = this.getDeployer(name);
               break;
            case APPLICATION:
               ctx = this.getApplication(name);
               break;
         }
      }
      // Make sure we don't return null
      if( ctx == null )
         throw new NoSuchDeploymentException("name="+name+", phase="+phase);
      return ctx;
   }

   public Collection<VFSDeployment> getDeployments()
   {
      HashSet<VFSDeployment> deployments = new HashSet<VFSDeployment>();
      deployments.addAll(this.bootstrapCtxs.values());
      deployments.addAll(this.deployerCtxs.values());
      deployments.addAll(this.applicationCtxs.values());
      return Collections.unmodifiableCollection(deployments);
   }

   /**
    * Scan the applications for changes.
    */
   public synchronized Collection<ModificationInfo> getModifiedDeployments()
      throws Exception
   {
      ArrayList<ModificationInfo> modified = new ArrayList<ModificationInfo>();
      Collection<VFSDeployment> apps = getApplications();
      boolean trace = log.isTraceEnabled();
      if( trace )
         log.trace("Checking applications for modifications");
      if( trace )
         log.trace("Aquiring content read lock");
      contentLock.readLock().lock();
      try
      {
         if( apps != null )
         {
            Iterator<VFSDeployment> iter = apps.iterator();
            int ignoreFlags = DeploymentContentFlags.LOCKED | DeploymentContentFlags.DISABLED;
            while( iter.hasNext() )
            {
               VFSDeployment ctx = iter.next();
               VirtualFile root = ctx.getRoot();
               String name = root.getPathName();
               // Ignore locked or disabled applications
               if(this.hasDeploymentContentFlags(name, DeploymentPhase.APPLICATION, ignoreFlags))
               {
                  if(trace)
                     log.trace("Ignoring locked application: "+root);
                  continue;
               }
               // Check for removal
               if( root.exists() == false )
               {
                  long rootLastModified = root.getLastModified();
                  ModificationInfo info = new ModificationInfo(ctx, rootLastModified, ModifyStatus.REMOVED);
                  modified.add(info);
                  iter.remove();
                  // Remove last modified cache
                  this.lastModifiedCache.remove(root.getPathName());
                  if( trace )
                     log.trace(name + " was removed");
               }
               // Check for modification
               else if( hasBeenModified(root) || 
                     hasDeploymentContentFlags(root.getPathName(), DeploymentPhase.APPLICATION, DeploymentContentFlags.MODIFIED) )
               {
                  long rootLastModified = root.getLastModified();
                  if( trace )
                     log.trace(name + " was modified: " + rootLastModified);
                  // Need to create a duplicate ctx
                  VFSDeployment ctx2 = loadDeploymentData(root, DeploymentPhase.APPLICATION);
                  ModificationInfo info = new ModificationInfo(ctx2, rootLastModified, ModifyStatus.MODIFIED);
                  modified.add(info);
               }
               // TODO: this could check metadata files modifications as well
            }
            // Now check for additions
            for (File applicationDir : applicationDirs)
            {
               VirtualFile deployDir = getCachedApplicationVF(applicationDir);
               ArrayList<VirtualFile> added = new ArrayList<VirtualFile>();
               addedDeployments(added, deployDir);
               for(VirtualFile vf : added)
               {
                  if(this.hasDeploymentContentFlags(vf.getPathName(), DeploymentPhase.APPLICATION, ignoreFlags))
                  {
                     if(trace)
                        log.trace("Ignoring locked application: "+ vf);
                     continue;
                  }
                  VFSDeployment ctx = loadDeploymentData(vf, DeploymentPhase.APPLICATION);
                  ModificationInfo info = new ModificationInfo(ctx, vf.getLastModified(), ModifyStatus.ADDED);
                  modified.add(info);
                  applicationCtxs.put(ctx.getName(), ctx);
               }
            }
         }
      }
      finally
      {
         contentLock.readLock().unlock();
         if( trace )
            log.trace("Released content read lock");
      }

      if(modified.size() > 0)
         lastModified = System.currentTimeMillis();
      return modified;
   }
   
   /**
    * Check if the deployment has been modified.
    *
    * @param root the virtual file root
    * @return true if modifed
    * @throws Exception for any error
    */
   protected boolean hasBeenModified(VirtualFile root) throws Exception
   {
      // get file:/ schema
      URI uri = VFSUtils.getCompatibleURI(root);
      File file = new File(uri);
      // if root is file check its modification
      if (file.isFile())
         return root.hasBeenModified();

      // else check metadata
      String name = root.toURI().toString();
      VFSDeploymentContext deploymentContext = getDeploymentContext(name);
      if (deploymentContext != null)
         return hasBeenModified(deploymentContext);

      log.trace("Falling back to root name: " + root);
      deploymentContext = getDeploymentContext(root.getName());
      if (deploymentContext != null)
         return hasBeenModified(deploymentContext);

      return false;
   }
   
   /**
    * Check for modifications.
    * 
    * @param deploymentContext
    * @return
    * @throws IOException
    */
   protected boolean hasBeenModified(VFSDeploymentContext deploymentContext) throws IOException
   {
      List<VirtualFile> metadataLocations = deploymentContext.getMetaDataLocations();
      if (metadataLocations != null && metadataLocations.isEmpty() == false)
      {
         for(VirtualFile metadataLocation : metadataLocations)
         {
            List<VirtualFile> children = metadataLocation.getChildren(hotDeploymentFilter);
            if (children != null && children.isEmpty() == false)
            {
               for(VirtualFile child : children)
               {
                  String pathName = child.getPathName();
                  Long timestamp = lastModifiedCache.get(pathName);
                  long lastModified = child.getLastModified();
                  lastModifiedCache.put(pathName, lastModified);
                  if (timestamp != null && timestamp < lastModified)
                  {
                     if (log.isTraceEnabled())
                        log.trace("Metadata location modified: " + child);
                     return true;
                  }
               }
            }
         }
      }
      List<DeploymentContext> childContexts = deploymentContext.getChildren();
      if (childContexts != null && childContexts.isEmpty() == false)
      {
         for (DeploymentContext childContext : childContexts)
         {
            if (childContext instanceof VFSDeploymentContext)
            {
               if (hasBeenModified((VFSDeploymentContext)childContext))
                  return true;
            }
         }
      }
      return false;
   }
   

   public Collection<VFSDeployment> getDeployments(DeploymentPhase phase)
      throws Exception
   {
      Collection<VFSDeployment> ctxs = null;
      switch( phase )
      {
         case BOOTSTRAP:
            ctxs = this.getBootstraps();
            break;
         case DEPLOYER:
            ctxs = this.getDeployers();
            break;
         case APPLICATION:
            ctxs = this.getApplications();
            break;
      }
      return ctxs;
   }

   public VFSDeployment removeDeployment(String name, DeploymentPhase phase)
      throws Exception
   {
      VFSDeployment ctx = null;
      switch( phase )
      {
         case BOOTSTRAP:
            ctx = this.removeBootstrap(name);
            break;
         case DEPLOYER:
            ctx = this.removeDeployer(name);
            break;
         case APPLICATION:
            ctx = this.removeApplication(name);
            break;
      }
      if(ctx != null)
         lastModified = System.currentTimeMillis();
      return ctx;
   }
   
   public String toString()
   {
      StringBuilder tmp = new StringBuilder(super.toString());
      tmp.append("(root=");
      tmp.append(root);
      tmp.append(", key=");
      tmp.append(key);
      tmp.append(")");
      return tmp.toString();
   }

   /**
    * Create a profile deployment repository
    * 
    * @throws IOException
    */
   public void create() throws Exception
   {
      File profileRoot = new File(root, key.getName());
      if( profileRoot.exists() == true )
         throw new IOException("Profile root already exists: "+profileRoot);
      if( profileRoot.mkdirs() == false )
         throw new IOException("Failed to create profile root: "+profileRoot);
      // server/{name}/bootstrap
      bootstrapDir = new File(profileRoot, "bootstrap");
      if( bootstrapDir.mkdirs() == false )
         throw new IOException("Failed to create profile bootstrap dir: "+bootstrapDir);

      // server/{name}/deployers
      deployersDir = new File(profileRoot, "deployers");
      if( deployersDir.mkdirs() == false )
         throw new IOException("Failed to create profile deployers dir: "+deployersDir);

      // server/{name}/deploy
      for (File applicationDir : applicationDirs)
      {
         if( applicationDir.mkdirs() == false )
            throw new IOException("Failed to create profile deploy dir: "+applicationDir);
      }
      // server/{name}/lib
      libDir = new File(profileRoot, "lib");
      if( libDir.mkdirs() == false )
         throw new IOException("Failed to create profile lib dir: "+libDir);

      if( adminEditsRoot.exists() == false && adminEditsRoot.mkdirs() == false )
         throw new IOException("Failed to create profile adminEdits dir: "+adminEditsRoot);
   }

   /**
    * Load the profile deployments
    * 
    * @throws IOException
    * @throws NoSuchProfileException
    */
   public void load() throws Exception, NoSuchProfileException
   {
      if( serializer == null )
         throw new IllegalStateException("serializer has not been set");

      File profileRoot = new File(root, key.getName());
      if( profileRoot.exists() == false )
         throw new NoSuchProfileException("Profile root does not exists: "+profileRoot);
      // server/{name}/bootstrap
      bootstrapDir = new File(profileRoot, "bootstrap");
      if( bootstrapDir.exists() == false )
      {
         //throw new FileNotFoundException("Profile contains no bootstrap dir: "+bootstrapDir);
         // fallback to conf/jboss-service.xml for now
         bootstrapDir = null;
      }

      // server/{name}/deployers
      deployersDir = new File(profileRoot, "deployers");
      if( deployersDir.exists() == false )
         throw new FileNotFoundException("Profile contains no deployers dir: "+deployersDir);

      // server/{name}/deploy
      for (File applicationDir : applicationDirs)
      {
         if( applicationDir.exists() == false )
            throw new FileNotFoundException("Profile contains no deploy dir: "+applicationDir);
      }

      if(this.serializer instanceof AbstractFileAttachmentsSerializer)
         ((AbstractFileAttachmentsSerializer) this.serializer).setAttachmentsStoreDir(adminEditsRoot);

      if( bootstrapDir != null )
      {
         VFS bootstrapVFS = VFS.getVFS(bootstrapDir.toURI());
         loadBootstraps(bootstrapVFS.getRoot());
      }
      else
      {
         // hack to load conf/jboss-service.xml until its removed
         loadBootstraps(null);         
      }
      VFS deployersVFS = VFS.getVFS(deployersDir.toURI());
      loadDeployers(deployersVFS.getRoot());
      for (File applicationDir : applicationDirs)
      {
         VirtualFile dirVF = getCachedApplicationVF(applicationDir);
         loadApplications(dirVF);
      }

      this.lastModified = System.currentTimeMillis();
   }

   /**
    * Get virtual file for app dir.
    *
    * @param applicationDir the app dir
    * @return virtual file representing app dir
    * @throws IOException for any error
    */
   protected VirtualFile getCachedApplicationVF(File applicationDir) throws IOException
   {
      URI uri = applicationDir.toURI();
      return getCachedApplicationVF(uri);
   }

   /**
    * Get virtual file for app uri.
    *
    * @param uri the app uri
    * @return virtual file representing app uri
    * @throws IOException for any error
    */
   protected VirtualFile getCachedApplicationVF(URI uri) throws IOException   
   {
      String uriString = uri.toString();
      VirtualFile dir;
      synchronized (applicationVFCache)
      {
         dir = applicationVFCache.get(uriString);
         if (dir == null)
         {
            dir = VFS.getCachedFile(uri);
            applicationVFCache.put(uriString, dir);
         }
      }
      return dir;
   }

   /**
    * Remove the contents of the profile repository
    * @throws IOException
    * @throws NoSuchProfileException
    */
   public void remove() throws IOException, NoSuchProfileException
   {
      File profileRoot = new File(root, key.getName());
      Files.delete(profileRoot);
   }

   protected void addBootstrap(String vfsPath, VFSDeployment ctx)
      throws Exception
   {
      this.bootstrapCtxs.put(vfsPath, ctx);
   }

   // Managed object attachments for a deployment
   public void addManagedObject(String vfsPath, Attachments edits)
      throws IOException
   {
      // TODO what should this do ?
      throw new UnsupportedOperationException("addManagedObject");
//      Map<String, Object> map = edits.getAttachments();
//      File attachments = new File(adminEditsRoot, vfsPath+".edits");
//      ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(attachments));
//      try
//      {
//         oos.writeObject(map);
//      }
//      finally
//      {
//         try
//         {
//            oos.close();
//         }
//         catch (IOException ignore)
//         {
//         }
//      }
//      lastModified = System.currentTimeMillis();
   }

   protected void addDeployer(String vfsPath, VFSDeployment ctx)
      throws Exception
   {
      this.deployerCtxs.put(vfsPath, ctx);
   }

   protected void addApplication(String vfsPath, VFSDeployment ctx) throws Exception
   {
      log.info("Adding application: " + vfsPath + " / " + ctx);
      this.applicationCtxs.put(vfsPath, ctx);
   }

   protected VFSDeployment getBootstrap(String vfsPath) throws Exception
   {
      VFSDeployment ctx = bootstrapCtxs.get(vfsPath);
      if( ctx == null )
         throw new NoSuchDeploymentException(vfsPath);
      return ctx;
   }

   protected Collection<VFSDeployment> getBootstraps() throws Exception
   {
      return bootstrapCtxs.values();
   }

   protected File getPhaseDir(DeploymentPhase phase)
   {
      File dir = null;
      switch( phase )
      {
         case BOOTSTRAP:
            dir = bootstrapDir;
            break;
         case DEPLOYER:
            dir = deployersDir;
            break;
         case APPLICATION:
            dir = applicationDirs[0];
            break;
         case APPLICATION_TRANSIENT:
            // TODO
            break;
      }
      return dir;
   }

   protected URI getBootstrapURI()
   {
      return bootstrapDir.toURI();
   }
   protected URI getDeployersURI()
   {
      return deployersDir.toURI();
   }
   protected URI getApplicationURI()
   {
      File applicationDir = applicationDirs[0];
      return applicationDir.toURI();
   }
   protected VFSDeployment getDeployer(String vfsPath)
      throws Exception
   {
      VFSDeployment ctx = deployerCtxs.get(vfsPath);
      if( ctx == null )
         throw new NoSuchDeploymentException(vfsPath);
      return ctx;
   }

   protected Collection<VFSDeployment> getDeployers()
      throws Exception
   {
      return deployerCtxs.values();
   }

   protected VFSDeployment getApplication(String vfsPath) throws Exception
   {
      boolean trace = log.isTraceEnabled();
      VFSDeployment ctx = applicationCtxs.get(vfsPath);
      if( ctx == null )
      {
         // Try to find the simple name
         if(trace)
            log.trace("Failed to find application for: "+vfsPath+", scanning for simple name");
         for(VFSDeployment deployment : applicationCtxs.values())
         {
            if(trace)
               log.trace("Checking: "+deployment.getSimpleName());
            if(deployment.getSimpleName().equals(vfsPath))
            {
               if(trace)
                  log.trace("Matched to simple name of deployment:"+deployment);
               ctx = deployment;
               break;
            }
         }
         if(ctx == null)
         {
            log.debug("Failed to find application for: "+vfsPath+", available: " + applicationCtxs.values());
            throw new NoSuchDeploymentException(vfsPath);
         }
      }
      return ctx;
   }

   protected Collection<VFSDeployment> getApplications()
      throws Exception
   {
      return applicationCtxs.values();
   }

   protected VFSDeployment removeBootstrap(String vfsPath) throws IOException
   {
      VFSDeployment vfsDeployment = bootstrapCtxs.get(vfsPath);
      if(vfsDeployment == null)
         throw new IllegalStateException("Deployment not found: " + vfsPath + ", available: " + bootstrapCtxs.values());
      VirtualFile root = vfsDeployment.getRoot();
      if(root.delete() == false)
         throw new IOException("Failed to delete: " + root);
      return bootstrapCtxs.remove(vfsPath);
   }

   // this is an infinite loop
   protected VFSDeployment removeDeployer(String vfsPath) throws IOException
   {
      VFSDeployment vfsDeployment = deployerCtxs.get(vfsPath);
      if(vfsDeployment == null)
         throw new IllegalStateException("Deployment not found: " + vfsPath + ", available: " + deployerCtxs.values());
      VirtualFile root = vfsDeployment.getRoot();
      if(root.delete() == false)
         throw new IOException("Failed to delete: " + root);
      return deployerCtxs.remove(vfsPath);
   }
   protected VFSDeployment removeApplication(String vfsPath) throws Exception
   {
      VFSDeployment vfsDeployment = getApplication(vfsPath);
      VirtualFile root = vfsDeployment.getRoot();
      if(root.delete() == false)
         throw new IOException("Failed to delete: " + root);
      return this.applicationCtxs.remove(vfsPath);
   }
   protected void setBootstrapURI(URI uri)
   {
      bootstrapDir = new File(uri);
   }
   protected void setDeployersURI(URI uri)
   {
      deployersDir = new File(uri);
   }

   /**
    * Load the bootstrap descriptors under bootstrapDir:
    * 
    * @param bootstrapDir
    * @throws IOException
    */
   private void loadBootstraps(VirtualFile bootstrapDir)
      throws IOException
   {
      if( bootstrapDir != null )
      {
         List<VirtualFile> children = bootstrapDir.getChildren();
         for(VirtualFile vf : children)
         {
            VFSDeployment vfCtx = loadDeploymentData(vf, DeploymentPhase.BOOTSTRAP);
            bootstrapCtxs.put(vfCtx.getName(), vfCtx);       
         }
      }
      else
      {
         // fallback to conf/jboss-service.xml for now
         File profileRoot = new File(root, key.getName());
         File confDir = new File(profileRoot, "conf");
         VirtualFile confVF = VFS.getRoot(confDir.toURI());
         VirtualFile jbossServiceVF = confVF.getChild("jboss-service.xml");
         if(jbossServiceVF == null)
            throw new FileNotFoundException("Failed to find jboss-service.xml under conf");
         VFSDeployment vfCtx = loadDeploymentData(jbossServiceVF, DeploymentPhase.BOOTSTRAP);
         bootstrapCtxs.put(vfCtx.getName(), vfCtx);                
      }
   }

   /**
    * Load all the deployments under the deployersDir.
    * 
    * @param deployersDir
    * @throws IOException
    */
   private void loadDeployers(VirtualFile deployersDir)
      throws IOException
   {
      List<VirtualFile> children = deployersDir.getChildren();
      for(VirtualFile vf : children)
      {
         VFSDeployment vfCtx = loadDeploymentData(vf, DeploymentPhase.DEPLOYER);
         deployerCtxs.put(vfCtx.getName(), vfCtx);       
      }
   }

   /**
    * Load all the applications under the applicationDir.
    * 
    * @param applicationDir
    * @throws IOException
    */
   private void loadApplications(VirtualFile applicationDir)
      throws Exception
   {
      ArrayList<VirtualFile> added = new ArrayList<VirtualFile>();
      addedDeployments(added, applicationDir);
      for (VirtualFile vf : added)
      {
         VFSDeployment vfCtx = loadDeploymentData(vf, DeploymentPhase.APPLICATION);
         applicationCtxs.put(vfCtx.getName(), vfCtx);
      }
   }

   /**
    * Added deployments for DeploymentPhase.APPLICATION.
    * 
    * @param list
    * @param root
    * @throws Exception
    */
   private void addedDeployments(List<VirtualFile> list, VirtualFile root)
      throws Exception
   {
      List<VirtualFile> components = root.getChildren();

      for (VirtualFile component : components)
      {
         // Excluding files from scanning
         if(! this.deploymentFilter.accepts(component))
         {
            log.trace("ignoring "+ component);
            continue;
         }
         
         String key = component.toURI().toString();
         if (applicationCtxs.containsKey(key) == true)
            continue;

         if (component.isLeaf())
         {
            list.add(component);
         }
         else if (component.getName().indexOf('.') == -1)
         {
            // recurse if not '.' in name and recursive search is enabled
            addedDeployments(list, component);
         }
         else
         {
            list.add(component);
         }
      }
   }

   /**
    * Update a deployment.
    * 
    * @param d the deployment
    * @param phase the deployment phase
    * @param comp the managed component
    * @throws Exception
    */
   public void updateDeployment(VFSDeployment d, DeploymentPhase phase,
         ManagedComponent comp) throws Exception
   {
      if(comp != null)
      {
         // update component
         super.updateDeployment(d, phase, comp);
         // Update last modified
         this.lastModified = System.currentTimeMillis();
      }
      else
      {
         log.error("no metadata attached.");
      }
   }
   
   protected VFSDeployment loadDeploymentData(VirtualFile file, DeploymentPhase phase)
   {
      try
      {
         return super.loadDeploymentData(file, phase);
      }
      catch(Exception e)
      {
         throw new RuntimeException("Could not load deployment data: "+ file, e);
      }
   }

}
