/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.management;

import java.lang.annotation.Annotation;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jboss.aop.Dispatcher;
import org.jboss.aop.advice.Interceptor;
import org.jboss.aspects.remoting.InvokeRemoteInterceptor;
import org.jboss.aspects.remoting.MergeMetaDataInterceptor;
import org.jboss.aspects.remoting.Remoting;
import org.jboss.aspects.security.SecurityClientInterceptor;
import org.jboss.deployers.client.spi.Deployment;
import org.jboss.deployers.client.spi.main.MainDeployer;
import org.jboss.deployers.spi.management.DeploymentTemplate;
import org.jboss.deployers.spi.management.ManagementView;
import org.jboss.deployers.spi.management.NameMatcher;
import org.jboss.deployers.spi.management.RuntimeComponentDispatcher;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.DeploymentTemplateInfo;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedOperation;
import org.jboss.managed.api.ManagedParameter;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.MutableManagedObject;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.managed.api.annotation.ManagementComponent;
import org.jboss.managed.api.annotation.ManagementObject;
import org.jboss.managed.api.annotation.ManagementObjectID;
import org.jboss.managed.api.annotation.ManagementObjectRef;
import org.jboss.managed.api.annotation.ManagementProperty;
import org.jboss.managed.api.annotation.ViewUse;
import org.jboss.managed.plugins.ManagedComponentImpl;
import org.jboss.managed.plugins.factory.AbstractManagedObjectFactory;
import org.jboss.metatype.api.types.ArrayMetaType;
import org.jboss.metatype.api.types.CollectionMetaType;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.values.ArrayValue;
import org.jboss.metatype.api.values.CollectionValue;
import org.jboss.metatype.api.values.GenericValue;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.MetaValueFactory;
import org.jboss.metatype.api.values.SimpleValue;
import org.jboss.profileservice.spi.NoSuchDeploymentException;
import org.jboss.profileservice.spi.NoSuchProfileException;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ProfileService;
import org.jboss.remoting.InvokerLocator;
import org.jboss.virtual.VirtualFile;

/**
 * The default ManagementView implementation.
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @author ales.justin@jboss.org
 * @version $Revision: 82070 $
 */
@ManagementObject(name="ManagementView", componentType=@ManagementComponent(type="MCBean", subtype="*"))
public class ManagementViewImpl implements ManagementView
{
   private static Logger log = Logger.getLogger(ManagementViewImpl.class);
   private static final String BUNDLE_NAME = "org.jboss.profileservice.management.messages"; //$NON-NLS-1$

   /** The deployment factory */
   private VFSDeploymentFactory deploymentFactory = VFSDeploymentFactory.getInstance();
   
   /** The ProfileService for loading profiles */
   private ProfileService ps;
   /** The currently loaded profile */
   private Profile activeProfile;
   private long activeProfileLastModified;
   /** The MainDeployer used to process profile changes */
   private MainDeployer mainDeployer;

   private InvokerLocator locator;

   /** The deployment templates that have been registered with the MV */
   private HashMap<String, DeploymentTemplate> templates = new HashMap<String, DeploymentTemplate>();
   /** The internationalization resource bundle */
   private ResourceBundle i18n;
   /** the Locale for the i18n messages */
   private Locale currentLocale;
   /** The formatter used for i18n messages */
   private MessageFormat formatter = new MessageFormat("");
   /** An index of ManagedComponent by ComponentType */
   private HashMap<ComponentType, Set<ManagedComponent>> compByCompType = new HashMap<ComponentType, Set<ManagedComponent>>();
   /** id/type key to ManagedObject map */
   private Map<String, ManagedObject> moRegistry = new HashMap<String, ManagedObject>();
   /** The deployment name to ManagedDeployment map */
   private Map<String, ManagedDeployment> managedDeployments = new HashMap<String, ManagedDeployment>();
   /** The bootstrap deployment name to ManagedDeployment map */
   private Map<String, ManagedDeployment> bootstrapManagedDeployments = Collections.emptyMap();
   /** The ManagedPropertys with unresolved ManagementObjectRefs */
   private Map<String, Set<ManagedProperty>> unresolvedRefs = new HashMap<String, Set<ManagedProperty>>();
   /** A map of runtime ManagedObjects needing to be merged with their
    * matching ManagedObject.
    */
   private Map<String, ManagedObject> runtimeMOs = new HashMap<String, ManagedObject>();
   /** The dispatcher handles ManagedOperation dispatches */
   private RuntimeComponentDispatcher dispatcher;
   private MetaValueFactory metaValueFactory = MetaValueFactory.getInstance();

   public ManagementViewImpl()
   {
      currentLocale = Locale.getDefault();
      formatter.setLocale(currentLocale);
      i18n = ResourceBundle.getBundle(BUNDLE_NAME, currentLocale);
   }

   /**
    * Load and associate the given profile with the ManagementView
    * for future operations. This must be called before the other ManagementView
    * operations may be used. Currently there can only be one actively loaded
    * profile against which a client is performing management operations.
    * 
    * @param key - the profile to load
    * @throws NoSuchProfileException
    */
   public void loadProfile(ProfileKey key)
      throws Exception
   {
      // If the profile is unmodified do nothing
      if(activeProfile != null && activeProfile.getKey().equals(key))
      {
         if(activeProfile.getLastModified() <= activeProfileLastModified)
         {
            log.debug("Not reloading profile: "+key+", "+activeProfileLastModified);
            return;
         }
      }

      activeProfile = ps.getProfile(key);
      if( activeProfile == null )
      {
         formatter.applyPattern(i18n.getString("ManagementView.NoSuchProfileException")); //$NON-NLS-1$
         Object[] args = {key};
         String msg = formatter.format(args);
         throw new NoSuchProfileException(msg);
      }
      // Cleanup
      this.compByCompType.clear();
      this.managedDeployments.clear();
      this.moRegistry.clear();
      this.runtimeMOs.clear();
      this.unresolvedRefs.clear();

      // Process the deployments
      boolean trace = log.isTraceEnabled();
      Collection<VFSDeployment> deployments = activeProfile.getDeployments();
      for(VFSDeployment deployment : deployments)
      {
         try
         {
            ManagedDeployment md = getManagedDeployment(deployment);
            processManagedDeployment(md, 0, trace);
         }
         catch(Exception e)
         {
            log.warn("Failed to create ManagedDeployment for: " + deployment.getName(), e);
         }
      }
      // Process the bootstrap deployments
      for(ManagedDeployment md : bootstrapManagedDeployments.values())
      {
         try
         {
            processManagedDeployment(md, 0, trace);
         }
         catch(Exception e)
         {
            log.warn("Failed to process ManagedDeployment for: " + md.getName(), e);
         }
      }
      if(this.runtimeMOs.size() > 0)
         log.warn("Failed to merged the following runtime ManagedObjects: "+runtimeMOs);
      activeProfileLastModified = activeProfile.getLastModified();
   }
   public void reloadProfile() throws Exception
   {
      activeProfileLastModified = 0;
      loadProfile(activeProfile.getKey());
   }

   /**
    * Process managed deployment.
    *
    * @param md the managed deployment
    * @param level depth level
    * @param trace is trace enabled
    * @throws Exception for any error
    */
   protected void processManagedDeployment(ManagedDeployment md, int level, boolean trace) throws Exception
   {
      String name = md.getName();
      if (trace)
         log.trace(name + " ManagedDeployment_" + level + ": " + md);
      Map<String, ManagedObject> mos = md.getManagedObjects();
      if (trace)
         log.trace(name + " ManagedObjects_ " + level + ": " + mos);
      for(ManagedObject mo : mos.values())
      {
         processManagedObject(mo, md);
      }
      managedDeployments.put(name, md);

      // Process children
      List<ManagedDeployment> mdChildren = md.getChildren();
      if(mdChildren != null && mdChildren.isEmpty() == false)
      {
         for(ManagedDeployment mdChild : mdChildren)
         {
            processManagedDeployment(mdChild, level + 1, trace);
         }
      }
   }

   /**
    * Process managed object.
    *
    * @param mo the managed object
    * @param md the managed deployment
    */
   protected void processManagedObject(ManagedObject mo, ManagedDeployment md)
      throws Exception
   {
      String key = mo.getName() + "/" + mo.getNameType();
      if(mo.getName().equals("org.jboss.security.plugins.SecurityConfig"))
         log.info("Saw SecurityConfig MO");
      log.debug("ID for ManagedObject: "+key+", attachmentName: "+mo.getAttachmentName());

      // See if this is a runtime ManagedObject
      Map<String, Annotation> moAnns = mo.getAnnotations();
      ManagementObject managementObject = (ManagementObject) moAnns.get(ManagementObject.class.getName());
      if (managementObject.isRuntime())
      {
         // Merge this with the ManagedObject
         ManagedObject parentMO = moRegistry.get(key);
         if (parentMO == null)
         {
            log.debug("Deferring resolution of runtime ManagedObject: "+managementObject);
            // Save the runtime mo for merging
            runtimeMOs.put(key, mo);
         }
         else
         {
            mergeRuntimeMO(parentMO, mo);
            runtimeMOs.remove(key);
         }
         // There is no further processing of runtime ManagedObjects
         return;
      }
      else
      {
         // See if there is runtime info to merge
         ManagedObject runtimeMO = runtimeMOs.get(key);
         if (runtimeMO != null)
         {
            mergeRuntimeMO(mo, runtimeMO);
            runtimeMOs.remove(key);
         }
      }

      // Update the MO registry
      // TODO - does this make sense? In case of a MetaType.isCollection we could get different results then
//      ManagedObject prevMO = moRegistry.put(key, mo);
//      if( prevMO != null )
//      {
//         // This should only matter for ManagedObjects that have a ManagementObjectID
//         log.debug("Duplicate mo for key: "+key+", prevMO: "+prevMO);
//         return;
//      }
      // Check for unresolved refs
      checkForReferences(key, mo);

      // Map any existing ManagedComponent types
      for(ManagedComponent comp : md.getComponents().values())
      {
         log.debug("Processing ManagementComponent: "+comp);
         ComponentType type = comp.getType();
         Set<ManagedComponent> typeComps = compByCompType.get(type);
         if (typeComps == null)
         {
            typeComps = new HashSet<ManagedComponent>();
            compByCompType.put(type, typeComps);
         }
         typeComps.add(comp);
      }

      // Create ManagedComponents for ManagedObjects annotated with ManagementComponent
      ManagementComponent mc = (ManagementComponent) moAnns.get(ManagementComponent.class.getName());
      if (mc != null)
      {
         ComponentType type = new ComponentType(mc.type(), mc.subtype());
         ManagedComponentImpl comp = new ManagedComponentImpl(type, md, mo);
         md.addComponent(mo.getName(), comp);
         log.debug("Processing ManagementComponent: "+comp);
         Set<ManagedComponent> typeComps = compByCompType.get(type);
         if (typeComps == null)
         {
            typeComps = new HashSet<ManagedComponent>();
            compByCompType.put(type, typeComps);
         }
         typeComps.add(comp);
      }

      // Scan for @ManagementObjectRef
      for(ManagedProperty prop : mo.getProperties().values())
      {
         log.debug("Checking property: "+prop);
         // See if this is a ManagementObjectID
         Map<String, Annotation> pannotations = prop.getAnnotations();
         if (pannotations != null && pannotations.isEmpty() == false)
         {
            ManagementObjectID id = (ManagementObjectID) pannotations.get(ManagementObjectID.class.getName());
            if (id != null)
            {
               Object refName = getRefName(prop.getValue());
               if (refName == null)
                  refName = id.name();
               String propKey = refName + "/" + id.type();
               log.debug("ManagedProperty level ID for ManagedObject: "+propKey+", attachmentName: "+mo.getAttachmentName());
               moRegistry.put(propKey, mo);
               checkForReferences(propKey, mo);
            }
            // See if this is a ManagementObjectRef
            ManagementObjectRef ref = (ManagementObjectRef) pannotations.get(ManagementObjectRef.class.getName());
            if ( ref != null )
            {
               // The reference key is the prop value + ref.type()
               log.debug("Property("+prop.getName()+") references: "+ref);
               Object refName = getRefName(prop.getValue());
               if (refName == null)
                  refName = ref.name();
               String targetKey = refName + "/" + ref.type();
               ManagedObject target = moRegistry.get(targetKey);
               if (target != null)
               {
                  log.debug("Resolved property("+prop.getName()+") reference to: "+targetKey);
                  prop.setTargetManagedObject(target);
               }
               else
               {
                  Set<ManagedProperty> referers =  unresolvedRefs.get(targetKey);
                  if (referers == null)
                  {
                     referers = new HashSet<ManagedProperty>();
                     unresolvedRefs.put(targetKey, referers);
                  }
                  referers.add(prop);
               }
            }
         }

         MetaType propType = prop.getMetaType();
         if (propType == AbstractManagedObjectFactory.MANAGED_OBJECT_META_TYPE)
         {
            processGenericValue ((GenericValue)prop.getValue(), md);
         }
         else if (propType.isArray())
         {
            ArrayMetaType amt = (ArrayMetaType) propType;
            MetaType etype = amt.getElementType();
            if (etype == AbstractManagedObjectFactory.MANAGED_OBJECT_META_TYPE)
            {
               ArrayValue avalue = (ArrayValue) prop.getValue();
               int length = avalue != null ? avalue.getLength() : 0;
               for(int n = 0; n < length; n ++)
                  processGenericValue((GenericValue) avalue.getValue(n), md);
            }
         }
         else if (propType.isCollection())
         {
            CollectionMetaType amt = (CollectionMetaType) propType;
            MetaType etype = amt.getElementType();
            if (etype == AbstractManagedObjectFactory.MANAGED_OBJECT_META_TYPE)
            {
               CollectionValue avalue = (CollectionValue) prop.getValue();
               if(avalue != null)
               {
                  MetaValue[] elements = avalue.getElements();
                  for(int n = 0; n < avalue.getSize(); n ++)
                  {
                     GenericValue gv = (GenericValue) elements[n];
                     ManagedObject propMO = (ManagedObject) gv.getValue();
                     if(propMO != null)
                        processManagedObject(propMO, md);
                  }
               }
            }
         }
      }
   }

   /**
    * Get ref name.
    *
    * @param value property value
    * @return plain value
    */
   protected Object getRefName(Object value)
   {
      if (value instanceof MetaValue)
      {
         MetaValue metaValue = (MetaValue)value;
         if (metaValue.getMetaType().isSimple() == false)
            throw new IllegalArgumentException("Can only get ref from simple value: " + value);
         SimpleValue svalue = (SimpleValue) metaValue;
         return svalue.getValue();
      }
      return value;
   }

   /**
    * Process generic value.
    *
    * @param genericValue the generic value
    * @param md the managed deployment
    * @throws Exception for any error
    */
   protected void processGenericValue(GenericValue genericValue, ManagedDeployment md) throws Exception
   {
      // TODO: a null is probably an error condition
      if (genericValue != null)
      {
         ManagedObject propMO = (ManagedObject) genericValue.getValue();
         // TODO: a null is probably an error condition
         if (propMO != null)
            processManagedObject(propMO, md);
      }
   }

   
   public Map<String, ManagedDeployment> getBootstrapManagedDeployments()
   {
      return bootstrapManagedDeployments;
   }
   public void setBootstrapManagedDeployments(
         Map<String, ManagedDeployment> bootstrapManagedDeployments)
   {
      this.bootstrapManagedDeployments = bootstrapManagedDeployments;
   }

   public Map<String, Set<ManagedProperty>> getUnresolvedRefs()
   {
      return unresolvedRefs;
   }

   public ProfileService getProfileService()
   {
      return ps;
   }

   public void setProfileService(ProfileService ps)
   {
      this.ps = ps;
      log.debug("setProfileService: "+ps);
   }

   public InvokerLocator getLocator()
   {
      return locator;
   }

   public void setLocator(InvokerLocator locator)
   {
      this.locator = locator;
   }

   public MainDeployer getMainDeployer()
   {
      return mainDeployer;
   }

   public void setMainDeployer(MainDeployer mainDeployer)
   {
      this.mainDeployer = mainDeployer;
      log.debug("setMainDeployer: "+mainDeployer);
   }

   public MetaValueFactory getMetaValueFactory()
   {
      return metaValueFactory;
   }
   public void setMetaValueFactory(MetaValueFactory metaValueFactory)
   {
      this.metaValueFactory = metaValueFactory;
   }

   public void setDispatcher(RuntimeComponentDispatcher dispatcher)
   {
      this.dispatcher = dispatcher;
   }

   /**
    * Get the names of the deployment in the profile.
    */
   public Set<String> getDeploymentNames()
   {
      Set<String> names = new HashSet<String>();
      Set<String> profileNames = activeProfile.getDeploymentNames();
      if(profileNames != null)
         names.addAll(profileNames);
      if(bootstrapManagedDeployments != null)
         names.addAll(bootstrapManagedDeployments.keySet());
      return names;
   }

   /**
    * Get the names of the deployment in the profile that have the
    * given deployment type.
    * 
    * @param type - the deployment type
    */
   public Set<String> getDeploymentNamesForType(String type)
   {
      HashSet<String> matches = new HashSet<String>();
      for(ManagedDeployment md : managedDeployments.values())
      {
         String name = md.getName();
         Set<String> types = md.getTypes();
         if(types != null)
         {
            if(types.contains(type))
            {
               log.debug(name+" matches type: "+type+", types:"+types);
               matches.add(name);
            }
         }
      }
      for(ManagedDeployment md : bootstrapManagedDeployments.values())
      {
         String name = md.getName();
         Set<String> types = md.getTypes();
         if(types != null)
         {
            if(types.contains(type))
            {
               log.debug(name+" matches type: "+type+", types:"+types);
               matches.add(name);
            }
         }
      }
      return matches;
   }

   public Set<String> getMatchingDeploymentName(String regex)
      throws NoSuchDeploymentException
   {
      Set<String> names = getDeploymentNames();
      HashSet<String> matches = new HashSet<String>();
      Pattern p = Pattern.compile(regex);
      for(String name : names)
      {
         Matcher m = p.matcher(name);
         if( m.matches() )
            matches.add(name);
      }
      if( matches.size() == 0 )
      {
         formatter.applyPattern(i18n.getString("ManagementView.NoSuchDeploymentException")); //$NON-NLS-1$
         Object[] args = {regex};
         String msg = formatter.format(args);
         throw new NoSuchDeploymentException(msg);
      }
      return matches;
   }
   public Set<ManagedDeployment> getMatchingDeployments(String name, NameMatcher<ManagedDeployment> matcher)
      throws NoSuchDeploymentException, Exception
   {
      Set<ManagedDeployment> matches = new HashSet<ManagedDeployment>();
      // TODO
      return matches;
   }

   public Set<String> getTemplateNames()
   {
      return new HashSet<String>(templates.keySet());
   }

   public void addTemplate(DeploymentTemplate template)
   {
      this.templates.put(template.getInfo().getName(), template);
      log.debug("addTemplate: "+template);
   }

   public void removeTemplate(DeploymentTemplate template)
   {
      this.templates.remove(template.getInfo().getName());
      log.debug("removeTemplate: "+template);
   }

   /**
    * 
    * @param key
    * @param name
    * @return
    */
   public ManagedDeployment getDeployment(String name, DeploymentPhase phase) throws NoSuchDeploymentException,
         Exception
   {
      // Resolve internally.
      ManagedDeployment md = this.managedDeployments.get(name);
      if (md == null)
      {
         // Check the bootstrap deployments
         md = this.bootstrapManagedDeployments.get(name);
      }
      
      // Let the DeploymentRepository try to resolve the deployment (simpleName)
      if(md == null)
      {
         Deployment ctx = activeProfile.getDeployment(name, phase);
         md = this.managedDeployments.get(ctx.getName());
      }
      
      // Do not return null
      if (md == null)
         throw new NoSuchDeploymentException("Managed deployment: " + name + " not found.");

      return md;
   }

   /**
    * 
    * @param key
    * @param type
    * @return
    * @throws NoSuchProfileException
    */
   public Set<ManagedDeployment> getDeploymentsForType(String type)
      throws Exception
   {
      Set<String> names = getDeploymentNamesForType(type);
      HashSet<ManagedDeployment> mds = new HashSet<ManagedDeployment>();
      for(String name : names)
      {
         Deployment ctx = activeProfile.getDeployment(name, null);
         ManagedDeployment md = getManagedDeployment(ctx);
         mds.add(md);
      }
      return mds;
   }

   /**
    * 
    * @param key
    * @param type
    * @return 
    * @throws NoSuchProfileException
    */
   public Set<ManagedComponent> getComponentsForType(ComponentType type)
      throws Exception
   {
      Set<ManagedComponent> comps = compByCompType.get(type);
      if(comps == null)
         comps = Collections.emptySet();
      return comps;
   }
   
   public ManagedComponent getComponent(String name, ComponentType type)
      throws Exception
   {
      Set<ManagedComponent> components = compByCompType.get(type);
      ManagedComponent comp = null;
      if(components != null)
      {
         for(ManagedComponent mc : components)
         {
            if(mc.getName().equals(name))
            {
               comp = mc;
               break;
            }
         }
      }
      if(comp != null)
      {
         Map<String, ManagedProperty> props = comp.getProperties();
         Set<ManagedOperation> ops = comp.getOperations();
         log.debug("Component"
               +"(ops.size="
               +ops != null ? ops.size() : 0
               +",props.size=)"
               +props != null ? props.size() : 0);
      }
      return comp;
   }
   public Set<ManagedComponent> getMatchingComponents(String name, ComponentType type,
         NameMatcher<ManagedComponent> matcher)
      throws Exception
   {
      Set<ManagedComponent> components = compByCompType.get(type);
      Set<ManagedComponent> matched = new HashSet<ManagedComponent>();
      if(components != null)
      {
         for(ManagedComponent mc : components)
         {
            if(matcher.matches(mc, name))
               matched.add(mc);
         }
      }
      if(matched.size() > 0)
      {
         log.debug("getComponents matched: "+matched);
      }
      return matched;
   }

   public DeploymentTemplateInfo getTemplate(String name)
      throws NoSuchDeploymentException
   {
      DeploymentTemplate template = templates.get(name);
      if( template == null )
      {
         formatter.applyPattern(i18n.getString("ManagementView.NoSuchTemplate")); //$NON-NLS-1$
         Object[] args = {name};
         String msg = formatter.format(args);
         throw new IllegalStateException(msg);
      }

      DeploymentTemplateInfo info = template.getInfo();
      log.debug("getTemplate, "+info);
      return info;
   }

   public void applyTemplate(DeploymentPhase phase, String deploymentBaseName, DeploymentTemplateInfo info)
      throws Exception
   {
      DeploymentTemplate template = templates.get(info.getName());
      if( template == null )
      {
         formatter.applyPattern(i18n.getString("ManagementView.NoSuchTemplate")); //$NON-NLS-1$
         Object[] args = {info.getName()};
         String msg = formatter.format(args);
         throw new IllegalStateException(msg);
      }

      // Create a deployment base from the template
      VirtualFile root = activeProfile.getRootFile(phase);
      if( log.isTraceEnabled() )
         log.trace("applyTemplate, profile="+activeProfile+", deploymentBaseName="+deploymentBaseName+", phase="+phase+", info="+info);
      VirtualFile vf = template.applyTemplate(root, deploymentBaseName, info);
      VFSDeployment ctx = deploymentFactory.createVFSDeployment(vf);
      activeProfile.addDeployment(ctx, phase);
      mainDeployer.addDeployment(ctx);
      // Seems useless, what was the original point of this?
      template.updateTemplateDeployment(ctx, info);
      mainDeployer.process();

      
      /* Scan through the template properties to see if there is a
         property with an ManagementObjectID annotation that needs
         to be used to update the associated ManagedObject name.
      */
      for(ManagedProperty prop : info.getProperties().values())
      {
         // Skip null values
         if( prop.getValue() == null )
            continue;
         Map<String, Annotation> pannotations = prop.getAnnotations();
         if (pannotations != null)
         {
            ManagementObjectID id = (ManagementObjectID) pannotations.get(ManagementObjectID.class.getName());
            if (id != null)
            {
               Object refName = getRefName(prop.getValue());
               if (refName == null)
                  refName = id.name();
               String name = "" + refName;
               log.debug("Updating template ManagedObject name to:"+name+" from property: "+prop);
               ManagedObject mo = prop.getManagedObject();
               if(mo instanceof MutableManagedObject)
               {
                  MutableManagedObject mmo = (MutableManagedObject) mo;
                  mmo.setName(name);
               }
               else
               {
                  formatter.applyPattern(i18n.getString("ManagementView.InvalidTemplatePropertyMO")); //$NON-NLS-1$
                  Object[] args = {prop.getName()};
                  String msg = formatter.format(args);
                  throw new IllegalArgumentException(msg);
               }
            }
         }
      }

      // Now apply the managed properties to get the deployment ManagedObjects
      Map<String, ManagedObject> mos = mainDeployer.getManagedObjects(ctx.getName());
      log.debug("applyTemplate, profile="+activeProfile+", deploymentBaseName="+deploymentBaseName+", phase="+phase+", :"+mos);
      // Map the 
      String propName = info.getRootManagedPropertyName();
      if(propName != null)
      {
         // Flatten out the root objects
         ManagedObject rootMO = mos.get(ctx.getName());
         if(rootMO != null)
            flattenRootObject(rootMO, propName, mos);
      }
      for(ManagedProperty prop : info.getProperties().values())
      {
         // Skip null values
         if( prop.getValue() == null )
            continue;

         ManagedObject mo = prop.getManagedObject();
         if (mo == null)
            throw new IllegalArgumentException("Null managed object: " + prop);

         ManagedObject ctxMO = mos.get(mo.getName());
         if( ctxMO == null )
         {
            formatter.applyPattern(i18n.getString("ManagementView.InvalidTemplateKey")); //$NON-NLS-1$
            Object[] args = {mo.getName()};
            String msg = formatter.format(args);
            throw new IllegalArgumentException(msg);
         }
         ManagedProperty ctxProp = ctxMO.getProperty(prop.getName());
         // Check for a mapped name
         if( ctxProp == null )
         {
            String mappedName = prop.getMappedName();
            if( mappedName != null )
               ctxProp = ctxMO.getProperty(mappedName);
         }
         if( ctxProp == null )
         {
            formatter.applyPattern(i18n.getString("ManagementView.InvalidTemplateProperty")); //$NON-NLS-1$
            Object[] args = {prop.getName()};
            String msg = formatter.format(args);
            throw new IllegalArgumentException(msg);
         }
         // The property value must be a MetaValue
         Object value = prop.getValue();
         if ((value instanceof MetaValue) == false)
         {
            formatter.applyPattern(i18n.getString("ManagementView.InvalidPropertyValue")); //$NON-NLS-1$
            Object[] args = {prop.getName(), value.getClass()};
            String msg = formatter.format(args);
            throw new IllegalArgumentException(msg);
         }
         MetaValue metaValue = (MetaValue)value;
         ctxProp.setValue(metaValue);

         // todo - should this also dispatch to runtime component?
         Object componentName = getComponentName(ctxProp);
         if (componentName != null)
            dispatcher.set(componentName, ctxProp.getName(), metaValue);
      }
   }

   public void removeDeployment(String deploymentName, DeploymentPhase phase)
      throws NoSuchProfileException, NoSuchDeploymentException, Exception
   {
      log.debug("removeDeployment, "+deploymentName+", phase: "+phase);
      Deployment ctx = activeProfile.removeDeployment(deploymentName, phase);
      if( ctx == null )
      {
         formatter.applyPattern(i18n.getString("ManagementView.NoSuchDeploymentException")); //$NON-NLS-1$
         Object[] args = {deploymentName};
         String msg = formatter.format(args);
         throw new NoSuchDeploymentException(msg);
      }

      if( mainDeployer.removeDeployment(ctx.getName()) == false )
      {
         formatter.applyPattern(i18n.getString("ManagementView.MainDeployerRemoveException")); //$NON-NLS-1$
         Object[] args = {deploymentName};
         String msg = formatter.format(args);
         throw new NoSuchDeploymentException(msg);
         
      }
      // Remove the ManagedDeployment/ManagedComponents
      ManagedDeployment md = managedDeployments.remove(deploymentName);
      if(md != null)
      {
         log.debug("removeDeployment, md: "+md);
         Map<String, ManagedComponent> comps = md.getComponents();
         if(comps != null)
         {
            for(ManagedComponent mc : comps.values())
            {
               ComponentType type = mc.getType();
               Set<ManagedComponent> compsForType = compByCompType.get(type);
               if(compsForType != null)
               {
                  if(compsForType.remove(mc) == true)
                     log.debug("Removed mc: "+mc);
                  else
                     log.debug(mc+" was not found in set: "+compsForType);
               }
            }
         }
      }
   }

   /**
    * Process the changes made to the profile.
    * 
    * @throws Exception
    */
   public void process() throws Exception
   {
      mainDeployer.process();
      mainDeployer.checkComplete();
      activeProfileLastModified = 0;
   }

   /**
    * Process a component update.
    */
   public void updateComponent(ManagedComponent comp)
      throws Exception
   {
      // Find the comp deployment
      ManagedDeployment md = comp.getDeployment();

      // Get the parent
      while( md.getParent() != null )
         md = md.getParent();
         
      String name = md.getName();
      DeploymentPhase phase = md.getDeploymentPhase();
      VFSDeployment compDeployment = activeProfile.getDeployment(name, phase);
      if( compDeployment == null )
      {
         formatter.applyPattern(i18n.getString("ManagementView.NoSuchDeploymentException")); //$NON-NLS-1$
         Object[] args = {name};
         String msg = formatter.format(args);
         throw new NoSuchDeploymentException(msg);
      }

      // Apply the managed properties to the server ManagedDeployment/ManagedComponent
      ManagedDeployment compMD = managedDeployments.get(md.getName());
      log.debug("updateComponent, profile="+activeProfile+", deploymentName="+name+", phase="+phase+", :"+compMD);
      
      ManagedComponent serverComp = null;
      // Find the managed component again
      if(comp.getDeployment().getParent() == null)
      {
         serverComp = compMD.getComponent(comp.getName());
      }
      else
      {
         // Look at the children
         // TODO - support more levels of nested deployments ?
         for(ManagedDeployment child : compMD.getChildren())
         {
            if(serverComp != null)
               break;
            
            serverComp = child.getComponent(comp.getName());
         }
      }
      if(serverComp == null)
      {
         log.debug("Name: "+comp.getName()+" does not map to existing ManagedComponet in ManagedDeployment: "+md.getName()
               + ", components: "+compMD.getComponents());
         formatter.applyPattern(i18n.getString("ManagementView.InvalidComponentName")); //$NON-NLS-1$
         Object[] args = {comp.getName(), md.getName()};
         String msg = formatter.format(args);
         throw new IllegalArgumentException(msg);
      }

      // Dispatch any runtime component property values
      for(ManagedProperty prop : comp.getProperties().values())
      {
         // Skip null values && non-CONFIGURATION values
         if( prop.getValue() == null || prop.hasViewUse(ViewUse.CONFIGURATION) == false)
            continue;

         ManagedProperty ctxProp = serverComp.getProperties().get(prop.getName());
         // Check for a mapped name
         if( ctxProp == null )
         {
            String mappedName = prop.getMappedName();
            if( mappedName != null )
               ctxProp = serverComp.getProperties().get(mappedName);
         }
         if( ctxProp == null )
         {
            formatter.applyPattern(i18n.getString("ManagementView.InvalidTemplateProperty")); //$NON-NLS-1$
            Object[] args = {prop.getName()};
            String msg = formatter.format(args);
            throw new IllegalArgumentException(msg);
         }
         // The property value must be a MetaValue
         Object value = prop.getValue();
         if ((value instanceof MetaValue) == false)
         {
            formatter.applyPattern(i18n.getString("ManagementView.InvalidPropertyValue")); //$NON-NLS-1$
            Object[] args = {prop.getName(), value.getClass()};
            String msg = formatter.format(args);
            throw new IllegalArgumentException(msg);
         }
         MetaValue metaValue = (MetaValue)value;
         ctxProp.setValue(metaValue);
         // Dispatch any runtime component property values
         Object componentName = getComponentName(ctxProp);
         if (componentName != null)
            dispatcher.set(componentName, ctxProp.getName(), metaValue);
      }
      
      
      // Update the repository deployment attachments
      activeProfile.updateDeployment(compDeployment, phase, serverComp);
   }

   /**
    * Get the component name from managed property.
    *
    * @param property the managed property
    * @return component name or null if no coresponding component
    */
   protected Object getComponentName(ManagedProperty property)
   {
      // first check target
      ManagedObject targetObject = property.getTargetManagedObject();
      if (targetObject != null)
         return targetObject.getComponentName();

      // check owner
      targetObject = property.getManagedObject();
      return targetObject != null ? targetObject.getComponentName() : null;
   }

   protected void checkForReferences(String key, ManagedObject mo)
   {
      Set<ManagedProperty> referers =  unresolvedRefs.get(key);
      log.debug("checkForReferences, "+key+" has referers: "+referers);
      if (referers != null)
      {
         for(ManagedProperty prop : referers)
         {
            prop.setTargetManagedObject(mo);
         }
         unresolvedRefs.remove(key);
      }      
   }

   /**
    * Merge the runtime props and ops
    * TODO: need a plugin to access the ManagedObject impl, JBMAN-23
    * @param mo
    * @param runtimeMO
    */
   protected void mergeRuntimeMO(ManagedObject mo, ManagedObject runtimeMO)
      throws Exception
   {
      Map<String, ManagedProperty> runtimeProps = runtimeMO.getProperties();
      Set<ManagedOperation> runtimeOps = runtimeMO.getOperations();
      log.debug("Merging runtime: "+runtimeMO.getName());
      Map<String, ManagedProperty> moProps = mo.getProperties();
      Set<ManagedOperation> moOps = mo.getOperations();
      HashMap<String, ManagedProperty> props = new HashMap<String, ManagedProperty>(moProps);
      HashSet<ManagedOperation> ops = new HashSet<ManagedOperation>(moOps);

      if (runtimeProps != null && runtimeProps.size() > 0)
      {
         // Get the runtime MO component name
         Object componentName = runtimeMO.getComponentName();
         log.debug("Properties before:"+props);
         // We need to pull the runtime values for stats
         for(ManagedProperty prop : runtimeProps.values())
         {
            Map<String, Annotation> pannotations = prop.getAnnotations();
            if(pannotations != null)
            {
               ManagementProperty mpa = (ManagementProperty) pannotations.get(ManagementProperty.class.getName());
               ViewUse[] uses = mpa.use();
               for(ViewUse use : uses)
               {
                  if(use == ViewUse.STATISTIC)
                  {
                     String propName = prop.getMappedName();
                     try
                     {
                        MetaValue propValue = dispatcher.get(componentName, propName);
                        if(propValue != null)
                           prop.setValue(propValue);
                     }
                     catch(Throwable t)
                     {
                        log.debug("Failed to get stat value, "+componentName+":"+propName);
                     }
                  }
               }
            }
         }
         props.putAll(runtimeProps);
         log.debug("Properties after:"+props);
      }
      if (runtimeOps != null && runtimeOps.size() > 0)
      {
         log.debug("Ops before:"+ops);
         runtimeOps = createOperationProxies(runtimeMO, runtimeOps);
         ops.addAll(runtimeOps);
         log.debug("Ops after:"+ops);
      }

      MutableManagedObject moi = (MutableManagedObject) mo;
      moi.setProperties(props);
      moi.setOperations(ops);
   }

   /**
    * Create ManagedOperation wrapper to intercept
    * its invocation, pushing the actual invocation
    * to runtime component.
    *
    * @param mo the managed object
    * @param ops the managed operations
    * @return set of wrapped managed operations
    * @throws Exception for any error
    * @see #
    */
   protected Set<ManagedOperation> createOperationProxies(ManagedObject mo, Set<ManagedOperation> ops)
      throws Exception
   {
      if (dispatcher == null)
         throw new IllegalArgumentException("Missing RuntimeComponentDispatcher.");

      ClassLoader loader = getClass().getClassLoader();
      ArrayList<Interceptor> interceptors = new ArrayList<Interceptor>();
      interceptors.add(SecurityClientInterceptor.singleton);
      interceptors.add(MergeMetaDataInterceptor.singleton);
      interceptors.add(InvokeRemoteInterceptor.singleton);

      // Create the ManagedOperation proxies
      HashSet<ManagedOperation> opProxies = new HashSet<ManagedOperation>();
      Class<?>[] ifaces = {ManagedOperation.class};
      for (ManagedOperation op : ops)
      {
         // This is just a unique name registered with the dispatcher and remoting layers
         String dispatchName = "ProfileService.ManagedOperation@"+System.identityHashCode(op);
         Dispatcher.singleton.registerTarget(dispatchName, new ManagedOperationDelegate(op, mo));
         ManagedOperation opProxy = (ManagedOperation) Remoting.createRemoteProxy(dispatchName,
               loader, ifaces, locator, interceptors, "ProfileService");
         opProxies.add(opProxy);
      }
      return opProxies;
   }

   private ManagedDeployment getManagedDeployment(Deployment ctx)
      throws Exception
   {
      return mainDeployer.getManagedDeployment(ctx.getName());
   }
   /**
    * Take the root managed object and the associated deployment unit
    * managed object map, and look to the root MO for a collection type
    * managed property named rootProperty to which the template property
    * map.
    * 
    * @param rootMO
    * @param rootProperty
    * @param managedObjects
    */
   private void flattenRootObject(ManagedObject rootMO, String rootProperty,
         Map<String,ManagedObject> managedObjects)
   {
      ManagedProperty deployments = rootMO.getProperty(rootProperty);
      MetaType propType = deployments.getMetaType();
      if (propType.isCollection())
      {
         CollectionMetaType amt = (CollectionMetaType) propType;
         MetaType etype = amt.getElementType();
         if (etype == AbstractManagedObjectFactory.MANAGED_OBJECT_META_TYPE)
         {
            CollectionValue avalue = (CollectionValue) deployments.getValue();
            if(avalue != null)
            {
               MetaValue[] elements = avalue.getElements();
               for(int n = 0; n < avalue.getSize(); n ++)
               {
                  GenericValue gv = (GenericValue) elements[n];
                  ManagedObject propMO = (ManagedObject) gv.getValue();
                  if(propMO != null)
                     managedObjects.put(propMO.getName(), propMO);
               }
            }
         }
      }
   }

   public class ManagedOperationDelegate implements ManagedOperation
   {
      private ManagedOperation delegate;
      private ManagedObject managedObject;

      public ManagedOperationDelegate(ManagedOperation delegate, ManagedObject managedObject)
      {
         if (delegate == null)
            throw new IllegalArgumentException("Null delegate.");
         this.delegate = delegate;
         if (managedObject == null)
            throw new IllegalArgumentException("Null managed object.");
         this.managedObject = managedObject;
      }

      public MetaValue invoke(MetaValue... metaValues)
      {
         Object componentName = managedObject.getComponentName();
         MetaValue result = null;
         if (componentName != null)
         {
            Object value = dispatcher.invoke(componentName, delegate.getName(), metaValues);
            if (value != null)
               result = metaValueFactory.create(value);
         }

         return result;
      }

      public String getDescription()
      {
         return delegate.getDescription();
      }

      public String getName()
      {
         return delegate.getName();
      }

      public Impact getImpact()
      {
         return delegate.getImpact();
      }

      public MetaType getReturnType()
      {
         return delegate.getReturnType();
      }

      public ManagedParameter[] getParameters()
      {
         return delegate.getParameters();
      }
   }
}
