/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.management;

import java.lang.reflect.UndeclaredThrowableException;

import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.registry.KernelBus;
import org.jboss.metatype.api.values.EnumValue;
import org.jboss.metatype.api.values.EnumValueSupport;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.MetaValueFactory;
import org.jboss.profileservice.spi.types.ControllerStateMetaType;

/**
 * Microcontainer KernelBus runtime component dispatcher.
 *
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 * @version $Revision: 81410 $
 */
public class KernelBusRuntimeComponentDispatcher extends AbstractRuntimeComponentDispatcher
{
   private KernelBus bus;
   private Kernel kernel;
   private String statePropertyName = "state";

   @Deprecated
   public KernelBusRuntimeComponentDispatcher(KernelBus bus)
   {
      this(bus, null);
   }

   @Deprecated
   public KernelBusRuntimeComponentDispatcher(KernelBus bus, MetaValueFactory valueFactory)
   {
      super(valueFactory);
      if (bus == null)
         throw new IllegalArgumentException("Null kernel bus.");

      this.bus = bus;
      this.kernel = bus.getKernel();
   }

   public KernelBusRuntimeComponentDispatcher(Kernel kernel)
   {
      this(kernel, null);
   }

   public KernelBusRuntimeComponentDispatcher(Kernel kernel, MetaValueFactory valueFactory)
   {
      super(valueFactory);
      if (kernel == null)
         throw new IllegalArgumentException("Null kernel");

      this.kernel = kernel;
      this.bus = kernel.getBus();
   }

   /**
    * Check kernel and bus.
    */
   public void start()
   {
      if (kernel == null)
         throw new IllegalArgumentException("Null kernel");
      if (bus == null)
         throw new IllegalArgumentException("Null kernel bus");
   }

   /**
    * Set the state property name.
    *
    * @param statePropertyName the state property name
    */
   public void setStatePropertyName(String statePropertyName)
   {
      this.statePropertyName = statePropertyName;
   }

   public MetaValue get(Object componentName, String propertyName)
   {
      try
      {
         if(propertyName.equals(statePropertyName))
            return getState(componentName);

         return create(bus.get(componentName, propertyName));
      }
      catch (Throwable t)
      {
         throw new UndeclaredThrowableException(t);
      }
   }

   public void set(Object componentName, String propertyName, MetaValue value)
   {
      try
      {
         bus.set(componentName, propertyName, unwrap(value));
      }
      catch (Throwable t)
      {
         throw new UndeclaredThrowableException(t);
      }
   }

   public Object invoke(Object componentName, String methodName, MetaValue... param)
   {
      try
      {
         return bus.invoke(componentName, methodName, toArguments(param), toSignature(param));
      }
      catch (Throwable t)
      {
         throw new UndeclaredThrowableException(t);
      }
   }

   /**
    * Get the state of a bean
    * 
    * @param name the bean name
    * @return state enum value
    */
   protected EnumValue getState(Object name)
   {
      KernelController controller = kernel.getController();
      ControllerContext context = controller.getContext(name, null);
      if (context == null)
         throw new IllegalStateException("Context not installed: " + name);

      ControllerState state = context.getState();
      return new EnumValueSupport(ControllerStateMetaType.TYPE, state.getStateString());
   }
}
