/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.singleton;

import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.List;

import org.jboss.ha.framework.interfaces.ClusterNode;

/**
 * Election policy that chooses the node where the singleton should run based on 
 * the given preferred master node in ip_address:port_number or 
 * host_name:port_number format. If the preferred master is null, or its 
 * ip_address does not resolve to a valid host name, or the port number is 
 * invalid, it delegates to the standard policy.  
 * 
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 */
public class PreferredMasterElectionPolicy extends HASingletonElectionPolicySimple 
   implements PreferredMasterElectionPolicyMBean
{
   private String preferredMaster;
   
   /**
    * @see PreferredMasterElectionPolicyMBean#setPreferredMaster(String)
    */
   public void setPreferredMaster(String node)
   {
      preferredMaster = node;
   }
   
   /**
    * @see PreferredMasterElectionPolicyMBean#getPreferredMaster()
    */
   public String getPreferredMaster()
   {
      return preferredMaster;
   }

   @Override
   protected ClusterNode elect(List<ClusterNode> candidates)
   {
      // If preferred master is defined and contained in cluster, return it
      if (preferredMaster != null) 
      {
         log.debug("Checking if " + preferredMaster + " is in candidate list " + candidates);
         
         // First just match on names
         for (ClusterNode node : candidates)
         {
            if (node.getName().equals(preferredMaster))
            {
               return node;
            }
         }
         
         // No match. Check for situation where preferred master uses a hostname
         // and the ClusterNode uses dotted decimal (or vice versa)
         
         // Create a URI out of the preferred master and try retrieving host 
         // and port.
         URI uri = createUri(preferredMaster);
         // See if we can parse out an InetAddress and port from preferredMaster.
         InetAddress addr = parseInetAddress(uri);
                 
         if (addr != null)
         {
            int port = uri.getPort();
            String rewritten = addr.getHostAddress() + ":" + port;
            if (preferredMaster.equals(rewritten))
            {
               rewritten = addr.getHostName() + ":" + port;
            }
            
            for (ClusterNode node : candidates)
            {
               if (node.getName().equals(rewritten))
               {
                  return node;
               }
            }
         }
      }
      
      return super.elect(candidates);
   }
   
   /**
    * Create a URI instance from the given preferred master. 
    * 
    * @param str contains the String format of the preferred master.
    * @return an instance of URI, or null if the str cannot be parsed.
    */
   protected URI createUri(String str)
   {
      try
      {
         return new URI("cluster://" + preferredMaster);
      }
      catch (URISyntaxException use)
      {
         log.debug("Cannot extract URI from " + preferredMaster, use);
      }
      
      return null;
   }
   
   /**
    * Parse an URI into an InetAddress
    * 
    * @param uri URI representation of the address. It can be null.
    * @return InetAddress representation or null if the host in the URI is 
    * unknown or given URI is null. 
    */
   protected InetAddress parseInetAddress(URI uri)
   {
      if (uri != null)
      {
         // Returns either the host, or null if host undefined
         String host = uri.getHost();
         // Returns either the port, or -1 if port undefined or not numeric
         int port = uri.getPort();         
         
         if (host != null && port != -1)
         {
            try
            {
               InetAddress addr = InetAddress.getByName(host);
               log.debug("Parsed " + preferredMaster + " into " + addr + " and " + port);
               return addr;
            }
            catch (UnknownHostException uhe)
            {
               log.debug("Cannot extract InetAddress from " + preferredMaster, uhe);
            }
         }             
      }
      
      return null;
   }
}