/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ha.jndi;

import java.util.Collection;

import javax.naming.Name;
import javax.naming.NamingException;

import org.jnp.interfaces.Naming;

import org.jboss.cache.Fqn;
import org.jboss.cache.Cache;
import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.logging.Logger;

/** 
 *  This class utilizes JBossCache to provide a distributed JNDI implementation.
 *  Lookups will look for Names in HAJNDI then delegate to the local InitialContext.
 *
 *  @author <a href="mailto:bill@burkecentral.com">Bill Burke</a>
 *  @author Scott.Stark@jboss.org
 *  @version $Revision: 58573 $
 */
public class HAJNDI
   implements org.jnp.interfaces.Naming
{
   /** @since 1.12.2.4, jboss-3.2.2 */
   static final long serialVersionUID = -6277328603304171620L;
  
   public static final String ROOT = "__HA_JNDI__";
   public static final Fqn ROOTFQN = new Fqn(new Object[] { ROOT });
   
   // Attributes --------------------------------------------------------
   private static Logger log = Logger.getLogger(HAJNDI.class);
   
   protected HAPartition partition;
   protected TreeHead delegate;
   protected Naming haStub;

   // Constructor --------------------------------------------------------  
  
   public HAJNDI(HAPartition partition, Cache cache)
   throws NamingException
   {
      if (partition == null)
         throw new IllegalArgumentException("Null partition");
      if (cache == null)
         throw new IllegalArgumentException("Null cache");
      this.partition = partition;
      delegate = new TreeHead(cache, HAJNDI.ROOTFQN);
      delegate.setPartition(this.partition);
      delegate.setHARMIHead(this);

   }
   
   // Public --------------------------------------------------------

   public void init() throws Exception
   {
      delegate.init();
   }

   public void stop() throws Exception
   {
      delegate.stop();
   }

   public void destroy() throws Exception
   {
      delegate.destroy();
   }

   public void setHAStub (Naming stub)
   {
      this.haStub = stub;
   }

   public Naming getHAStub ()
   {
      return this.haStub;
   }

   // Naming implementation -----------------------------------------
   

   public synchronized void bind(Name name, Object obj, String className)
      throws NamingException
   {
      delegate.bind (name, obj, className);
   }

   public synchronized void rebind(Name name, Object obj, String className)
      throws NamingException
   {
      delegate.rebind (name, obj, className);
   }

   public synchronized void unbind(Name name)
      throws NamingException
   {
      delegate.unbind (name);
   }

   public Object lookup(Name name)
      throws NamingException
   {
      return delegate.lookup (name);
   }

   public Collection list(Name name)
      throws NamingException
   {
      return delegate.list(name) ;
   }
    
   public Collection listBindings(Name name)
      throws NamingException
   {
      return delegate.listBindings(name);
   }
   
   public javax.naming.Context createSubcontext(Name name)
      throws NamingException
   {
      return delegate.createSubcontext(name);
   }
}
