/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ha.framework.server;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.lang.ref.WeakReference;
import java.net.InetAddress;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NameNotFoundException;
import javax.naming.Reference;
import javax.naming.StringRefAddr;

import org.jboss.cache.Cache;
import org.jboss.ha.framework.interfaces.ClusterNode;
import org.jboss.ha.framework.interfaces.DistributedReplicantManager;
import org.jboss.ha.framework.interfaces.DistributedState;
import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.invocation.MarshalledValueInputStream;
import org.jboss.invocation.MarshalledValueOutputStream;
import org.jboss.logging.Logger;
import org.jboss.naming.NonSerializableFactory;
import org.jboss.system.ServiceMBeanSupport;
import org.jgroups.Channel;
import org.jgroups.ChannelFactory;
import org.jgroups.ExtendedMessageListener;
//import org.jgroups.JChannel;
import org.jgroups.ExtendedMembershipListener;
import org.jgroups.MembershipListener;
import org.jgroups.MergeView;
import org.jgroups.Message;
import org.jgroups.MessageListener;
import org.jgroups.Version;
import org.jgroups.View;
import org.jgroups.blocks.GroupRequest;
import org.jgroups.blocks.MethodCall;
import org.jgroups.blocks.RpcDispatcher;
import org.jgroups.stack.IpAddress;
import org.jgroups.util.Rsp;
import org.jgroups.util.RspList;

/**
 * {@link HAPartition} implementation based on a 
 * <a href="http://www.jgroups.com/">JGroups</a> <code>RpcDispatcher</code> 
 * and a multiplexed <code>JChannel</code>.
 *
 * @author <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 * @author <a href="mailto:bill@burkecentral.com">Bill Burke</a>.
 * @author Scott.Stark@jboss.org
 * @author brian.stansberry@jboss.com
 * @version $Revision: 69549 $
 */
public class ClusterPartition
   extends ServiceMBeanSupport
   implements ExtendedMembershipListener, HAPartition, 
              AsynchEventHandler.AsynchEventProcessor,
              ClusterPartitionMBean
{
   private static final byte EOF_VALUE   = -1;
   private static final byte NULL_VALUE   = 0;
   private static final byte SERIALIZABLE_VALUE = 1;
   // TODO add Streamable support
   // private static final byte STREAMABLE_VALUE = 2;
   
   /**
    * Returned when an RPC call arrives for a service that isn't registered.
    */
   private static class NoHandlerForRPC implements Serializable
   {
      static final long serialVersionUID = -1263095408483622838L;
   }
   
   private static class StateStreamEnd implements Serializable
   {
      /** The serialVersionUID */
      private static final long serialVersionUID = -3705345735451504946L;      
   }
   
   /**
    * Used internally when an RPC call requires a custom classloader for unmarshalling
    */
   private static class HAServiceResponse implements Serializable
   {
      private static final long serialVersionUID = -6485594652749906437L;
      private final String serviceName;
      private final byte[] payload;
           
      public HAServiceResponse(String serviceName, byte[] payload)
      {
         this.serviceName = serviceName;
         this.payload = payload;
      }
           
      public String getServiceName()
      {
         return serviceName;
      }
           
      public byte[] getPayload()
      {
         return payload;
      }
   }

   // Constants -----------------------------------------------------

   // final MethodLookup method_lookup_clos = new MethodLookupClos();

   // Attributes ----------------------------------------------------

   protected ClusterPartitionConfig config;
   protected HashMap<String, Object> rpcHandlers = new HashMap<String, Object>();
   protected HashMap stateHandlers = new HashMap();
   /** Do we send any membership change notifications synchronously? */
   protected boolean allowSyncListeners = false;
   /** The HAMembershipListener and HAMembershipExtendedListeners */
   protected ArrayList synchListeners = new ArrayList();
   /** The asynch HAMembershipListener and HAMembershipExtendedListeners */
   protected ArrayList asynchListeners = new ArrayList();
   /** The handler used to send membership change notifications asynchronously */
   protected AsynchEventHandler asynchHandler;
   /** The current cluster partition members */
   protected Vector members = null;
   protected Vector jgmembers = null;
   protected ConcurrentHashMap<String, WeakReference<ClassLoader>> clmap =
                                          new ConcurrentHashMap<String, WeakReference<ClassLoader>>();

   public Vector history = null;

   /** The partition members other than this node */
   protected Vector otherMembers = null;
   protected Vector jgotherMembers = null;
   /** the local JG IP Address */
   protected org.jgroups.stack.IpAddress localJGAddress = null;
   /** The cluster transport protocol address string */
   protected String nodeName;
   /** me as a ClusterNode */
   protected ClusterNode me = null;
   /** The JGroups partition channel */
   protected Channel channel;
   /** The cluster replicant manager */
   protected DistributedReplicantManagerImpl replicantManager;
   /** The DistributedState service we manage */
   protected DistributedStateImpl distributedState;
   /** The cluster instance log category */
   protected Logger log;
   protected Logger clusterLifeCycleLog;   
   /** The current cluster view id */
   protected long currentViewId = -1;
   /** Whether to bind the partition into JNDI */
   protected boolean bindIntoJndi = true;
   
   private final ThreadGate flushBlockGate = new ThreadGate();
   
   private RpcDispatcher dispatcher = null;

   /**
    * True if serviceState was initialized during start-up.
    */
   protected boolean isStateSet = false;

   /**
    * An exception occuring upon fetch serviceState.
    */
   protected Exception setStateException;
   private final Object channelLock = new Object();
   private final MessageListenerAdapter messageListener = new MessageListenerAdapter();

   // Static --------------------------------------------------------
   
   private static Channel createMuxChannel(ClusterPartitionConfig config)
   {
      ChannelFactory factory = config.getMultiplexer();
      if (factory == null)
         throw new IllegalStateException("HAPartitionConfig has no JChannelFactory");
      String stack = config.getMultiplexerStack();
      if (stack == null)
         throw new IllegalStateException("HAPartitionConfig has no multiplexer stack");
      try
      {
         return factory.createMultiplexerChannel(stack, config.getPartitionName());
      }
      catch (RuntimeException e)
      {
         throw e;
      }
      catch (Exception e)
      {
         throw new RuntimeException("Failure creating multiplexed Channel", e);
      }
   }

    // Constructors --------------------------------------------------
   
   public ClusterPartition(ClusterPartitionConfig config)
   {
      if (config == null)
         throw new IllegalArgumentException("config cannot be null");

      this.config = config;
      setupLoggers(config.getPartitionName());
      this.history = new Vector();
      logHistory ("Partition object created");      
   }

   // ------------------------------------------------------------ ServiceMBean
   
   // ----------------------------------------------------------------- Service
   
   protected void createService() throws Exception
   {
      if (config == null)
         throw new IllegalArgumentException("config cannot be null");
      
      if (replicantManager == null)
         throw new IllegalStateException("DistributedReplicantManager property must be set before creating ClusterPartition service");
      
      replicantManager.createService();
      
      if (distributedState != null)
      {
         distributedState.setClusteredCache(getClusteredCache());
         distributedState.createService();
      }         
      
      // Create the asynchronous handler for view changes
      asynchHandler = new AsynchEventHandler(this, "AsynchViewChangeHandler");
      
      log.debug("done initializing partition");
   }
   
   protected void startService() throws Exception
   {
      logHistory ("Starting partition");
      
      log.debug("Creating Multiplexer Channel for partition " + getPartitionName() +
            " using stack " + getMultiplexerStack());

      channel = createMuxChannel(config);
      
      channel.setOpt(Channel.AUTO_RECONNECT, Boolean.TRUE);
      channel.setOpt(Channel.AUTO_GETSTATE, Boolean.TRUE);
      
      log.info("Initializing partition " + getPartitionName());
      logHistory ("Initializing partition " + getPartitionName());
      
      dispatcher = new RpcHandler(channel, null, null, new Object(), config.getDeadlockDetection());
      
      // Subscribe to events generated by the org.jgroups. protocol stack
      log.debug("setMembershipListener");
      dispatcher.setMembershipListener(this);
      log.debug("setMessageListener");
      dispatcher.setMessageListener(messageListener);
      dispatcher.setRequestMarshaller(new RequestMarshallerImpl());
      dispatcher.setResponseMarshaller(new ResponseMarshallerImpl());
      
      channel.connect(getPartitionName());
      
      try
      {
         // get current JG group properties         
         log.debug("get nodeName");
         this.localJGAddress = (IpAddress)channel.getLocalAddress();
         this.me = new ClusterNodeImpl(this.localJGAddress);
         this.nodeName = this.me.getName();

         log.debug("Get current members");
         waitForView();                

         verifyNodeIsUnique();

         fetchState();
         
         replicantManager.startService();
         
         if (distributedState != null)
         {
            distributedState.startService();
         }
         
         // Start the asynch listener handler thread
         asynchHandler.start();
         
         // Bind ourself in the public JNDI space if configured to do so
         if (bindIntoJndi)
         {
            Context ctx = new InitialContext();
            this.bind("/HAPartition/" + getPartitionName(), this, ClusterPartition.class, ctx);
            log.debug("Bound in JNDI under /HAPartition/" + getPartitionName());
         }
      }
      catch (Throwable t)
      {
         log.debug("Caught exception after channel connected; closing channel -- " + t.getLocalizedMessage());
         channel.close();
         channel = null;
         throw (t instanceof Exception) ? (Exception) t : new RuntimeException(t);
      }
      
   }

   protected void stopService() throws Exception
   {
      logHistory ("Stopping partition");
      log.info("Stopping partition " + getPartitionName());

      try
      {
         asynchHandler.stop();
      }
      catch( Exception e)
      {
         log.warn("Failed to stop asynchHandler", e);
      }    
      
      if (distributedState != null)
      {
         distributedState.stopService();
      }

      replicantManager.stopService();
      
//    NR 200505 : [JBCLUSTER-38] replace channel.close() by a disconnect and
//    add the destroyPartition() step
      try
      {
         if (channel != null && channel.isConnected())
            channel.disconnect();
      }
      catch (Exception e)
      {
         log.error("channel disconnection failed", e);
      }

      if (bindIntoJndi)
      {
         String boundName = "/HAPartition/" + getPartitionName();
         InitialContext ctx = null;
         try
         {
            // the following statement fails when the server is being shut down (07/19/2007)
            ctx = new InitialContext();
            ctx.unbind(boundName);
         }
         catch (Exception e) {
            log.error("partition unbind operation failed", e);
         }
         finally
         {
            if (ctx != null)
               ctx.close();
         }
         NonSerializableFactory.unbind (boundName);         
      }

      log.info("Partition " + getPartitionName() + " stopped.");
   }
   
   protected void destroyService()  throws Exception
   {
      log.debug("Destroying HAPartition: " + getPartitionName()); 
      
      if (distributedState != null)
      {
         distributedState.destroyService();
      }   

      replicantManager.destroyService();

      try
      {
         if (channel != null && channel.isOpen())
            channel.close();
      }
      catch (Exception e)
      {
         log.error("Closing channel failed", e);
      }

      log.info("Partition " + getPartitionName() + " destroyed.");
   }
   
   // ---------------------------------------------------------- State Transfer 


   protected void fetchState() throws Exception
   {
      log.info("Fetching serviceState (will wait for " + getStateTransferTimeout() + 
            " milliseconds):");
      long start, stop;
      isStateSet = false;
      start = System.currentTimeMillis();
      boolean rc = channel.getState(null, getStateTransferTimeout());
      if (rc)
      {
         synchronized (channelLock)
         {
            while (!isStateSet)
            {
               if (setStateException != null)
                  throw setStateException;

               try
               {
                  channelLock.wait();
               }
               catch (InterruptedException iex)
               {
               }
            }
         }
         stop = System.currentTimeMillis();
         log.info("serviceState was retrieved successfully (in " + (stop - start) + " milliseconds)");
      }
      else
      {
         // No one provided us with serviceState.
         // We need to find out if we are the coordinator, so we must
         // block until viewAccepted() is called at least once

         synchronized (members)
         {
            while (members.size() == 0)
            {
               log.debug("waiting on viewAccepted()");
               try
               {
                  members.wait();
               }
               catch (InterruptedException iex)
               {
               }
            }
         }

         if (isCurrentNodeCoordinator())
         {
            log.info("State could not be retrieved (we are the first member in group)");
         }
         else
         {
            throw new IllegalStateException("Initial serviceState transfer failed: " +
               "Channel.getState() returned false");
         }
      }
   }

   private void getStateInternal(OutputStream stream) throws IOException
   {
      MarshalledValueOutputStream mvos = null; // don't create until we know we need it
      
      for (Iterator keys = stateHandlers.entrySet().iterator(); keys.hasNext(); )
      {
         Map.Entry entry = (Map.Entry)keys.next();
         HAPartition.HAPartitionStateTransfer subscriber = 
            (HAPartition.HAPartitionStateTransfer) entry.getValue();
         log.debug("getState for " + entry.getKey());
         Object state = subscriber.getCurrentState();
         if (state != null)
         {
            if (mvos == null)
            {
               // This is our first write, so need to write the header first
               stream.write(SERIALIZABLE_VALUE);
               
               mvos = new MarshalledValueOutputStream(stream);
            }
            
            mvos.writeObject(entry.getKey());
            mvos.writeObject(state);
         }
      }
      
      if (mvos == null)
      {
         // We never wrote any serviceState, so write the NULL header
         stream.write(NULL_VALUE);
      }
      else
      {
         mvos.writeObject(new StateStreamEnd());
         mvos.flush();
         mvos.close();
      }
      
   }
   
   private void setStateInternal(InputStream stream) throws IOException, ClassNotFoundException
   {
      byte type = (byte) stream.read();
      
      if (type == EOF_VALUE)
      {
         log.debug("serviceState stream is empty");
         return;
      }
      else if (type == NULL_VALUE)
      {
         log.debug("serviceState is null");
         return;
      }
      
      long used_mem_before, used_mem_after;
      Runtime rt=Runtime.getRuntime();
      used_mem_before=rt.totalMemory() - rt.freeMemory();
      
      MarshalledValueInputStream mvis = new MarshalledValueInputStream(stream);
      
      while (true)
      {
         Object obj = mvis.readObject(); 
         if (obj instanceof StateStreamEnd)
            break;
         
         String key = (String) obj;
         log.debug("setState for " + key);
         Object someState = mvis.readObject();
         HAPartition.HAPartitionStateTransfer subscriber = (HAPartition.HAPartitionStateTransfer)stateHandlers.get(key);
         if (subscriber != null)
         {
            try
            {
               subscriber.setCurrentState((Serializable)someState);
            }
            catch (Exception e)
            {
               // Don't let issues with one subscriber affect others
               // unless it is DRM, which is really an internal function
               // of the HAPartition
               // FIXME remove this once DRM is JBC-based
               if (DistributedReplicantManagerImpl.SERVICE_NAME.equals(key))
               {
                  if (e instanceof RuntimeException)
                     throw (RuntimeException) e;
                  else
                     throw new RuntimeException(e);
               }
               else
               {
                  log.error("Caught exception setting serviceState to " + subscriber, e);
               }
            }
         }
         else
         {
            log.debug("There is no stateHandler for: " + key);
         }      
      }
      
      try
      {
         stream.close();
      }
      catch(Exception e)
      {
         log.error("Caught exception closing serviceState stream", e);
      }

      used_mem_after=rt.totalMemory() - rt.freeMemory();
      log.debug("received serviceState; expanded memory by " +
            (used_mem_after - used_mem_before) + " bytes (used memory before: " + used_mem_before +
            ", used memory after: " + used_mem_after + ")");
   }

   private void recordSetStateFailure(Throwable t)
   {
      log.error("failed setting serviceState", t);
      if (t instanceof Exception)
         setStateException = (Exception) t;
      else
         setStateException = new Exception(t);
   }

   private void notifyChannelLock()
   {
      synchronized (channelLock)
      {         
         channelLock.notifyAll();
      }
   }
   
   // org.jgroups.MembershipListener implementation ----------------------------------------------
   
   public void suspect(org.jgroups.Address suspected_mbr)
   {      
      logHistory ("Node suspected: " + (suspected_mbr==null?"null":suspected_mbr.toString()));
      if (isCurrentNodeCoordinator ())
         clusterLifeCycleLog.info ("Suspected member: " + suspected_mbr);
      else
         log.info("Suspected member: " + suspected_mbr);
   }

   public void block() 
   {       
       flushBlockGate.close();           
       log.debug("Block processed at " + me);	  
   }
   
   public void unblock()
   {
       flushBlockGate.open();           
       log.debug("Unblock processed at " + me);
   }
   
   /** Notification of a cluster view change. This is done from the JG protocol
    * handlder thread and we must be careful to not unduly block this thread.
    * Because of this there are two types of listeners, synchronous and
    * asynchronous. The synchronous listeners are messaged with the view change
    * event using the calling thread while the asynchronous listeners are
    * messaged using a seperate thread.
    *
    * @param newView
    */
   public void viewAccepted(View newView)
   {
      try
      {
         // we update the view id
         this.currentViewId = newView.getVid().getId();

         // Keep a list of other members only for "exclude-self" RPC calls
         this.jgotherMembers = (Vector)newView.getMembers().clone();
         this.jgotherMembers.remove (channel.getLocalAddress());
         this.otherMembers = translateAddresses (this.jgotherMembers); // TRANSLATE!
         Vector translatedNewView = translateAddresses ((Vector)newView.getMembers().clone());
         logHistory ("New view: " + translatedNewView + " with viewId: " + this.currentViewId +
                     " (old view: " + this.members + " )");


         // Save the previous view and make a copy of the new view
         Vector oldMembers = this.members;

         Vector newjgMembers = (Vector)newView.getMembers().clone();
         Vector newMembers = translateAddresses(newjgMembers); // TRANSLATE
         this.members = newMembers;
         this.jgmembers = newjgMembers;
         
         if (oldMembers == null)
         {
            // Initial viewAccepted
            log.debug("ViewAccepted: initial members set for partition " + getPartitionName() + ": " +
                     this.currentViewId + " (" + this.members + ")");
            
            log.info("Number of cluster members: " + members.size());
            for(int m = 0; m > members.size(); m ++)
            {
               Object node = members.get(m);
               log.debug(node);
            }
            log.info ("Other members: " + this.otherMembers.size ());
            
            // Wake up the deployer thread blocking in waitForView
            notifyChannelLock();
            return;
         }        
         
         int difference = 0;
         if (oldMembers == null)
            difference = newMembers.size () - 1;
         else
            difference = newMembers.size () - oldMembers.size ();
         
         if (isCurrentNodeCoordinator ())
            clusterLifeCycleLog.info ("New cluster view for partition " + getPartitionName() + " (id: " +
                                      this.currentViewId + ", delta: " + difference + ") : " + this.members);
         else
            log.info("New cluster view for partition " + getPartitionName() + ": " +
                     this.currentViewId + " (" + this.members + " delta: " + difference + ")");

         // Build a ViewChangeEvent for the asynch listeners
         ViewChangeEvent event = new ViewChangeEvent();
         event.viewId = currentViewId;
         event.allMembers = translatedNewView;
         event.deadMembers = getDeadMembers(oldMembers, event.allMembers);
         event.newMembers = getNewMembers(oldMembers, event.allMembers);
         event.originatingGroups = null;
         // if the new view occurs because of a merge, we first inform listeners of the merge
         if(newView instanceof MergeView)
         {
            MergeView mergeView = (MergeView) newView;
            event.originatingGroups = mergeView.getSubgroups();
         }

         log.debug("membership changed from " + 
                  (oldMembers == null ? 0 : oldMembers.size()) + " to " + 
                  event.allMembers.size());
         // Put the view change to the asynch queue
         this.asynchHandler.queueEvent(event);

         // Broadcast the new view to the synchronous view change listeners
         if (this.allowSyncListeners)
         {
            this.notifyListeners(synchListeners, event.viewId, event.allMembers,
                  event.deadMembers, event.newMembers, event.originatingGroups);
         }
      }
      catch (Exception ex)
      {
         log.error("ViewAccepted failed", ex);
      }
   }

   private void waitForView() throws Exception
   {
      synchronized (channelLock)
      {
         if (this.members == null)
         {
            try
            {
               channelLock.wait(getMethodCallTimeout());
            }
            catch (InterruptedException iex)
            {
            }
            
            if (this.members == null)
               throw new IllegalStateException("No view received from Channel");
         }
      }
   }

   // HAPartition implementation ----------------------------------------------
   
   public String getNodeName()
   {
      return nodeName;
   }
   
   public String getPartitionName()
   {
      return (config == null ? null : config.getPartitionName());
   }  
   
   public DistributedReplicantManager getDistributedReplicantManager()
   {
      return replicantManager;
   }
   
   public DistributedState getDistributedStateService()
   {
      return distributedState;
   }

   public long getCurrentViewId()
   {
      return this.currentViewId;
   }
   
   public Vector getCurrentView()
   {
      Vector result = new Vector (this.members.size());
      for (int i = 0; i < members.size(); i++)
      {
         result.add( ((ClusterNode) members.elementAt(i)).getName() );
      }
      return result;
   }

   public ClusterNode[] getClusterNodes ()
   {
      synchronized (members)
      {
         ClusterNode[] nodes = new ClusterNode[this.members.size()];
         nodes = (ClusterNode[]) this.members.toArray(nodes);
         return nodes;
      }
   }

   public ClusterNode getClusterNode ()
   {
      return me;
   }

   public boolean isCurrentNodeCoordinator ()
   {
      if(this.members == null || this.members.size() == 0 || this.me == null)
         return false;
     return this.members.elementAt (0).equals (this.me);
   }

   // ***************************
   // ***************************
   // RPC multicast communication
   // ***************************
   // ***************************
   //
   public void registerRPCHandler(String objName, Object subscriber)
   {
      rpcHandlers.put(objName, subscriber);
   }
   
   public void registerRPCHandler(String objName, Object subscriber, ClassLoader classloader)
   {
      registerRPCHandler(objName, subscriber);
      clmap.put(objName, new WeakReference<ClassLoader>(classloader));
   }
   
   public void unregisterRPCHandler(String objName, Object subscriber)
   {
      rpcHandlers.remove(objName);
      clmap.remove(objName);
   }      

   /**
    * This function is an abstraction of RpcDispatcher.
    */
   public ArrayList callMethodOnCluster(String objName, String methodName,
      Object[] args, Class[] types, boolean excludeSelf) throws Exception
   {
      return callMethodOnCluster(objName, methodName, args, types, excludeSelf, getMethodCallTimeout());
   }


   public ArrayList callMethodOnCluster(String objName, String methodName,
       Object[] args, Class[] types, boolean excludeSelf, long methodTimeout) throws Exception
   {
      RspList rsp = null;
      boolean trace = log.isTraceEnabled();

      MethodCall m = new MethodCall(objName + "." + methodName, args, types);
      
      if(channel.flushSupported())
      {
     	 flushBlockGate.await(getStateTransferTimeout());
      }
      if (excludeSelf)
      {
         if( trace )
         {
            log.trace("callMethodOnCluster(true), objName="+objName
               +", methodName="+methodName+", members="+jgotherMembers);
         }
         rsp = dispatcher.callRemoteMethods(this.jgotherMembers, m, GroupRequest.GET_ALL, methodTimeout);
      }
      else
      {
         if( trace )
         {
            log.trace("callMethodOnCluster(false), objName="+objName
               +", methodName="+methodName+", members="+members);
         }
         rsp = dispatcher.callRemoteMethods(null, m, GroupRequest.GET_ALL, methodTimeout);
      }

      return processResponseList(rsp, trace);
    }

   /**
    * Calls method on Cluster coordinator node only.  The cluster coordinator node is the first node to join the
    * cluster.
    * and is replaced
    * @param objName
    * @param methodName
    * @param args
    * @param types
    * @param excludeSelf
    * @return an array of responses from remote nodes
    * @throws Exception
    */
   public ArrayList callMethodOnCoordinatorNode(String objName, String methodName,
          Object[] args, Class[] types,boolean excludeSelf) throws Exception
   {
      return callMethodOnCoordinatorNode(objName,methodName,args,types,excludeSelf, getMethodCallTimeout());
   }

   /**
    * Calls method on Cluster coordinator node only.  The cluster coordinator node is the first node to join the
    * cluster.
    * and is replaced
    * @param objName
    * @param methodName
    * @param args
    * @param types
    * @param excludeSelf
    * @param methodTimeout
    * @return an array of responses from remote nodes
    * @throws Exception
    */
   public ArrayList callMethodOnCoordinatorNode(String objName, String methodName,
          Object[] args, Class[] types,boolean excludeSelf, long methodTimeout) throws Exception
   {
      boolean trace = log.isTraceEnabled();

      MethodCall m = new MethodCall(objName + "." + methodName, args, types);
      
      if( trace )
      {
         log.trace("callMethodOnCoordinatorNode(false), objName="+objName
            +", methodName="+methodName);
      }

      // the first cluster view member is the coordinator
      Vector coordinatorOnly = new Vector();
      // If we are the coordinator, only call ourself if 'excludeSelf' is false
      if (false == isCurrentNodeCoordinator () ||
          false == excludeSelf)
      {
         coordinatorOnly.addElement(this.jgmembers.elementAt (0));
      }
      
      RspList rsp = dispatcher.callRemoteMethods(coordinatorOnly, m, GroupRequest.GET_ALL, methodTimeout);

      return processResponseList(rsp, trace);
   }

    /**
     * Calls method synchrounously on target node only.
     * @param serviceName Name of the target service name on which calls are de-multiplexed
     * @param methodName name of the Java method to be called on remote services
     * @param args array of Java Object representing the set of parameters to be
     * given to the remote method
     * @param types The types of the parameters
     * node of the partition or only on remote nodes
     * @param targetNode is the target of the call
     * @return the value returned by the target method
     * @throws Exception Throws if a communication exception occurs
     */
   public Object callMethodOnNode(String serviceName, String methodName,
           Object[] args, Class[] types, long methodTimeout, ClusterNode targetNode) throws Throwable
    {
       if (!(targetNode instanceof ClusterNodeImpl))
          throw new IllegalArgumentException("targetNode " + targetNode + " is not an instance of " + 
                                          ClusterNodeImpl.class + " -- only targetNodes provided by this HAPartition should be used");
       MethodCall m;
       boolean trace = log.isTraceEnabled();
       if(types != null)
          m=new MethodCall(serviceName + "." + methodName, args, types);
       else
          m=new MethodCall(serviceName + "." + methodName, args);
       if( trace )
       {
          log.trace("callMethodOnNode( objName="+serviceName
             +", methodName="+methodName);
       }
       Object rc = dispatcher.callRemoteMethod(((ClusterNodeImpl)targetNode).getOriginalJGAddress(), m, GroupRequest.GET_FIRST, methodTimeout);
       if (rc != null)
       {
          Object item = rc;
          if (item instanceof Rsp)
          {
             Rsp response = (Rsp) item;
             // Only include received responses
             boolean wasReceived = response.wasReceived();
             if( wasReceived == true )
             {
                item = response.getValue();
                if (!(item instanceof NoHandlerForRPC))
                   rc = item;
                }
                else if( trace )
                   log.trace("Ignoring non-received response: "+response);
             }
             else
             {
                if (!(item instanceof NoHandlerForRPC))
                   rc = item;
                else if( trace )
                   log.trace("Ignoring NoHandlerForRPC");
             }
          }
       return rc;
     }


   /**
     * Calls method on target node only.
     * @param serviceName Name of the target service name on which calls are de-multiplexed
     * @param methodName name of the Java method to be called on remote services
     * @param args array of Java Object representing the set of parameters to be
     * given to the remote method
     * @param types The types of the parameters
     * node of the partition or only on remote nodes
     * @param targetNode is the target of the call
     * @return none
     * @throws Exception Throws if a communication exception occurs
     */
   public void callAsyncMethodOnNode(String serviceName, String methodName,
           Object[] args, Class[] types, long methodTimeout, ClusterNode targetNode) throws Throwable
   {
      if (!(targetNode instanceof ClusterNodeImpl))
         throw new IllegalArgumentException("targetNode " + targetNode + " is not an instance of " + 
                                         ClusterNodeImpl.class + " -- only targetNodes provided by this HAPartition should be used");
      MethodCall m;
       boolean trace = log.isTraceEnabled();
       if(types != null)
          m=new MethodCall(serviceName + "." + methodName, args, types);
       else
          m=new MethodCall(serviceName + "." + methodName, args);
       if( trace )
       {
          log.trace("callAsyncMethodOnNode( objName="+serviceName
             +", methodName="+methodName);
       }
       dispatcher.callRemoteMethod(((ClusterNodeImpl)targetNode).getOriginalJGAddress(), m, GroupRequest.GET_NONE, methodTimeout);
   }

   private ArrayList processResponseList(RspList rsp, boolean trace)
   {
      ArrayList rtn = new ArrayList();
      if (rsp != null)
      {
         for (Object item : rsp.values())
         {
            if (item instanceof Rsp)
            {
               Rsp response = (Rsp) item;
               // Only include received responses
               boolean wasReceived = response.wasReceived();
               if( wasReceived == true )
               {
                  item = response.getValue();
                  if (!(item instanceof NoHandlerForRPC))
                     rtn.add(item);
               }
               else if( trace )
                  log.trace("Ignoring non-received response: "+response);
            }
            else
            {
               if (!(item instanceof NoHandlerForRPC))
                  rtn.add(item);
               else if( trace )
                  log.trace("Ignoring NoHandlerForRPC");
            }
         }
         
      }
      return rtn;
   }

   /**
    * This function is an abstraction of RpcDispatcher for asynchronous messages
    */
   public void callAsynchMethodOnCluster(String objName, String methodName,
      Object[] args, Class[] types, boolean excludeSelf) throws Exception
   {
      boolean trace = log.isTraceEnabled();

      MethodCall m = new MethodCall(objName + "." + methodName, args, types);

      if(channel.flushSupported())
      {
     	 flushBlockGate.await(getStateTransferTimeout());
      }
      if (excludeSelf)
      {
         if( trace )
         {
            log.trace("callAsynchMethodOnCluster(true), objName="+objName
               +", methodName="+methodName+", members="+jgotherMembers);
         }
         dispatcher.callRemoteMethods(this.jgotherMembers, m, GroupRequest.GET_NONE, getMethodCallTimeout());
      }
      else
      {
         if( trace )
         {
            log.trace("callAsynchMethodOnCluster(false), objName="+objName
               +", methodName="+methodName+", members="+members);
         }
         dispatcher.callRemoteMethods(null, m, GroupRequest.GET_NONE, getMethodCallTimeout());
      }
   }
   
   // *************************
   // *************************
   // State transfer management
   // *************************
   // *************************
   //      
   public void subscribeToStateTransferEvents(String objectName, HAPartitionStateTransfer subscriber)
   {
      stateHandlers.put(objectName, subscriber);
   }
   
   public void unsubscribeFromStateTransferEvents(String objectName, HAPartitionStateTransfer subscriber)
   {
      stateHandlers.remove(objectName);
   }
   
   // *************************
   // *************************
   // Group Membership listeners
   // *************************
   // *************************
   //   
   public void registerMembershipListener(HAMembershipListener listener)
   {
      boolean isAsynch = (this.allowSyncListeners == false) 
            || (listener instanceof AsynchHAMembershipListener)
            || (listener instanceof AsynchHAMembershipExtendedListener);
      if( isAsynch ) {
         synchronized(this.asynchListeners) {
            this.asynchListeners.add(listener);
         }
      }
      else  { 
         synchronized(this.synchListeners) {
            this.synchListeners.add(listener);
         }
      }
   }
   
   public void unregisterMembershipListener(HAMembershipListener listener)
   {
      boolean isAsynch = (this.allowSyncListeners == false) 
            || (listener instanceof AsynchHAMembershipListener)
            || (listener instanceof AsynchHAMembershipExtendedListener);
      if( isAsynch ) {
         synchronized(this.asynchListeners) {
            this.asynchListeners.remove(listener);
         }
      }
      else  { 
         synchronized(this.synchListeners) {
            this.synchListeners.remove(listener);
         }
      }
   }
   
   public boolean getAllowSynchronousMembershipNotifications()
   {
      return allowSyncListeners;
   }

   public void setAllowSynchronousMembershipNotifications(boolean allowSync)
   {      
      this.allowSyncListeners = allowSync;
   }
   
   // AsynchEventHandler.AsynchEventProcessor -----------------------

   public void processEvent(Object event)
   {
      ViewChangeEvent vce = (ViewChangeEvent) event;
      notifyListeners(asynchListeners, vce.viewId, vce.allMembers,
            vce.deadMembers, vce.newMembers, vce.originatingGroups);
      
   }
   
   
   // Public ------------------------------------------------------------------
   
   public void setDistributedStateImpl(DistributedStateImpl distributedState)
   {
      this.distributedState = distributedState;      
   }
   
   public void setDistributedReplicantManagerImpl(DistributedReplicantManagerImpl drm)
   {
      if (this.replicantManager != null  && !(replicantManager == drm))
         throw new IllegalStateException("DistributedReplicantManager already set");

      this.replicantManager = drm;
      if (this.replicantManager != null)
      {
         this.replicantManager.setHAPartition(this);
      }
   }
   
   
   // Protected -----------------------------------------------------

   protected void verifyNodeIsUnique () throws IllegalStateException
   {
      ClusterNodeImpl matched = null;
      for (ClusterNode member : getClusterNodes())
      {
         if (member.equals(me))
         {
            if (matched == null)
            {
               // We of course are in the view, so we expect one match
               // Just track that we've had one
               matched = (ClusterNodeImpl) member;
            }
            else
            {
               // Two nodes in view match us; try to figure out which one isn't us
               ClusterNodeImpl other = matched;
               if (other.getOriginalJGAddress().equals(((ClusterNodeImpl)me).getOriginalJGAddress()))
               {
                  other = (ClusterNodeImpl) member;
               }
               throw new IllegalStateException("Found member " + other + 
                     " in current view that duplicates us (" + me + "). This" +
                     " node cannot join partition until duplicate member has" +
                     " been removed");
            }
         }
      }
   }

   /**
    * Helper method that binds the partition in the JNDI tree.
    * @param jndiName Name under which the object must be bound
    * @param who Object to bind in JNDI
    * @param classType Class type under which should appear the bound object
    * @param ctx Naming context under which we bind the object
    * @throws Exception Thrown if a naming exception occurs during binding
    */   
   protected void bind(String jndiName, Object who, Class classType, Context ctx) throws Exception
   {
      // Ah ! This service isn't serializable, so we use a helper class
      //
      NonSerializableFactory.bind(jndiName, who);
      Name n = ctx.getNameParser("").parse(jndiName);
      while (n.size () > 1)
      {
         String ctxName = n.get (0);
         try
         {
            ctx = (Context)ctx.lookup (ctxName);
         }
         catch (NameNotFoundException e)
         {
            log.debug ("creating Subcontext " + ctxName);
            ctx = ctx.createSubcontext (ctxName);
         }
         n = n.getSuffix (1);
      }

      // The helper class NonSerializableFactory uses address type nns, we go on to
      // use the helper class to bind the service object in JNDI
      //
      StringRefAddr addr = new StringRefAddr("nns", jndiName);
      Reference ref = new Reference(classType.getName (), addr, NonSerializableFactory.class.getName (), null);
      ctx.rebind (n.get (0), ref);
   }
   
   /**
    * Helper method that returns a vector of dead members from two input vectors: new and old vectors of two views.
    * Dead members are old - new members.
    * @param oldMembers Vector of old members
    * @param newMembers Vector of new members
    * @return Vector of members that have died between the two views, can be empty.
    */   
   protected Vector getDeadMembers(Vector oldMembers, Vector newMembers)
   {
      if(oldMembers == null) oldMembers=new Vector();
      if(newMembers == null) newMembers=new Vector();
      Vector dead=(Vector)oldMembers.clone();
      dead.removeAll(newMembers);
      log.debug("dead members: " + dead);
      return dead;
   }
   
   /**
    * Helper method that returns a vector of new members from two input vectors: new and old vectors of two views.
    * @param oldMembers Vector of old members
    * @param allMembers Vector of new members
    * @return Vector of members that have joined the partition between the two views
    */   
   protected Vector getNewMembers(Vector oldMembers, Vector allMembers)
   {
      if(oldMembers == null) oldMembers=new Vector();
      if(allMembers == null) allMembers=new Vector();
      Vector newMembers=(Vector)allMembers.clone();
      newMembers.removeAll(oldMembers);
      return newMembers;
   }

   protected void notifyListeners(ArrayList theListeners, long viewID,
      Vector allMembers, Vector deadMembers, Vector newMembers,
      Vector originatingGroups)
   {
      log.debug("Begin notifyListeners, viewID: "+viewID);
      synchronized(theListeners)
      {
         // JBAS-3619 -- don't hold synch lock while notifying
         theListeners = (ArrayList) theListeners.clone();
      }
      
      for (int i = 0; i < theListeners.size(); i++)
      {
         HAMembershipListener aListener = null;
         try
         {
            aListener = (HAMembershipListener) theListeners.get(i);
            if(originatingGroups != null && (aListener instanceof HAMembershipExtendedListener))
            {
               HAMembershipExtendedListener exListener = (HAMembershipExtendedListener) aListener;
               exListener.membershipChangedDuringMerge (deadMembers, newMembers,
                  allMembers, originatingGroups);
            }
            else
            {
               aListener.membershipChanged(deadMembers, newMembers, allMembers);
            }
         }
         catch (Throwable e)
         {
            // a problem in a listener should not prevent other members to receive the new view
            log.warn("HAMembershipListener callback failure: "+aListener, e);
         }
      }
      
      log.debug("End notifyListeners, viewID: "+viewID);
   }
   
   /*
    * Allows caller to specify whether the partition instance should be bound into JNDI.  Default value is true.
    * This method must be called before the partition is started as the binding occurs during startup.
    * 
    * @param bind  Whether to bind the partition into JNDI.
    */
   public void setBindIntoJndi(boolean bind)
   {
       bindIntoJndi = bind;
   }
   
   /*
    * Allows caller to determine whether the partition instance has been bound into JNDI.
    * 
    * @return true if the partition has been bound into JNDI.
    */
   public boolean getBindIntoJndi()
   {
       return bindIntoJndi;
   }


   protected Vector translateAddresses (Vector jgAddresses)
   {
      if (jgAddresses == null)
         return null;

      Vector result = new Vector (jgAddresses.size());
      for (int i = 0; i < jgAddresses.size(); i++)
      {
         IpAddress addr = (IpAddress) jgAddresses.elementAt(i);
         result.add(new ClusterNodeImpl(addr));
      }

      return result;
   }

   public void logHistory (String message)
   {
      try
      {
         history.add(new SimpleDateFormat().format (new Date()) + " : " + message);
      }
      catch (Exception ignored){}
   }

   // --------------------------------------------------- ClusterPartitionMBean
   
   public String showHistory ()
   {
      StringBuffer buff = new StringBuffer();
      Vector data = new Vector (this.history);
      for (java.util.Iterator row = data.iterator(); row.hasNext();)
      {
         String info = (String) row.next();
         buff.append(info).append("\n");
      }
      return buff.toString();
   }

   public String showHistoryAsXML ()
   {
      StringBuffer buff = new StringBuffer();
      buff.append("<events>\n");
      Vector data = new Vector (this.history);
      for (java.util.Iterator row = data.iterator(); row.hasNext();)
      {
         buff.append("   <event>\n      ");
         String info = (String) row.next();
         buff.append(info);
         buff.append("\n   </event>\n");
      }
      buff.append("</events>\n");
      return buff.toString();
   }
   
   public Cache getClusteredCache()
   {
      return config.getClusteredCache();
   }

   public boolean getDeadlockDetection()
   {
      return config.getDeadlockDetection();
   }

   public HAPartition getHAPartition()
   {
      return this;
   }

   public String getJGroupsVersion()
   {
      return Version.description + "( " + Version.cvs + ")";
   }

   public ChannelFactory getMultiplexer()
   {
      return config.getMultiplexer();
   }

   public String getMultiplexerStack()
   {
      return config.getMultiplexerStack();
   }

   public InetAddress getNodeAddress()
   {
      return config.getNodeAddress();
   }

   public long getStateTransferTimeout() {
      return config.getStateTransferTimeout();
   }

   public long getMethodCallTimeout() {
      return config.getMethodCallTimeout();
   }

   public void setMethodCallTimeout(long timeout)
   {
      config.setMethodCallTimeout(timeout);      
   }

   public void setStateTransferTimeout(long timeout)
   {
      config.setStateTransferTimeout(timeout);
   }

   public String getNodeUniqueId()
   {
      return config.getNodeUniqueId();
   }

   // Protected --------------------------------------------------------------
      
   /**
    * Creates an object from a byte buffer
    */
   protected Object objectFromByteBufferInternal (byte[] buffer) throws Exception
   {
      if(buffer == null) 
         return null;

      ByteArrayInputStream bais = new ByteArrayInputStream(buffer);
      MarshalledValueInputStream mvis = new MarshalledValueInputStream(bais);
      return mvis.readObject();
   }
   
   /**
    * Serializes an object into a byte buffer.
    * The object has to implement interface Serializable or Externalizable
    */
   protected byte[] objectToByteBufferInternal (Object obj) throws Exception
   {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      MarshalledValueOutputStream mvos = new MarshalledValueOutputStream(baos);
      mvos.writeObject(obj);
      mvos.flush();
      return baos.toByteArray();
   }
   
   /**
    * Creates a response object from a byte buffer - optimized for response marshalling
    */
   protected Object objectFromByteBufferResponseInternal (byte[] buffer) throws Exception
   {
      if(buffer == null) 
         return null;

      if (buffer[0] == NULL_VALUE)
         return null;

      ByteArrayInputStream bais = new ByteArrayInputStream(buffer);
      // read past the null/serializable byte
      bais.read();
      MarshalledValueInputStream mvis = new MarshalledValueInputStream(bais);
      return mvis.readObject();
   }
   
   /**
    * Serializes a response object into a byte buffer, optimized for response marshalling.
    * The object has to implement interface Serializable or Externalizable
    */
   protected byte[] objectToByteBufferResponseInternal (Object obj) throws Exception
   {
      if (obj == null)
         return new byte[]{NULL_VALUE};

      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      // write a marker to stream to distinguish from null value stream
      baos.write(SERIALIZABLE_VALUE);
      MarshalledValueOutputStream mvos = new MarshalledValueOutputStream(baos);
      mvos.writeObject(obj);
      mvos.flush();
      return baos.toByteArray();
   }
   
   // Private -------------------------------------------------------
   
   // Inner classes -------------------------------------------------

   private class MessageListenerAdapter
         implements ExtendedMessageListener
   {
      
      public void getState(OutputStream stream)
      {
         logHistory ("getState called on partition");
         
         log.debug("getState called.");
         try
         {
            getStateInternal(stream);
         }
         catch (Exception ex)
         {
            log.error("getState failed", ex);
         }
         
      }
      
      public void getState(String state_id, OutputStream ostream)
      {
         throw new UnsupportedOperationException("Not implemented; see http://jira.jboss.com/jira/browse/JBAS-3594");
      }

      public byte[] getState(String state_id)
      {
         throw new UnsupportedOperationException("Not implemented; see http://jira.jboss.com/jira/browse/JBAS-3594");
      }
      
      public void setState(InputStream stream)
      {
         logHistory ("setState called on partition");
         try
         {
            if (stream == null)
            {
               log.debug("transferred serviceState is null (may be first member in cluster)");
            }
            else
            {
               setStateInternal(stream);
            }
            
            isStateSet = true;
         }
         catch (Throwable t)
         {
            recordSetStateFailure(t);
         }
         finally
         {
            // Notify waiting thread that serviceState has been set.
            notifyChannelLock();
         }
      }

      public byte[] getState()
      {
         logHistory ("getState called on partition");
         
         log.debug("getState called.");
         try
         {
            ByteArrayOutputStream baos = new ByteArrayOutputStream(1024);
            getStateInternal(baos);
            return baos.toByteArray();
         }
         catch (Exception ex)
         {
            log.error("getState failed", ex);
         }
         return null; // This will cause the receiver to get a "false" on the channel.getState() call
      }

      public void setState(String state_id, byte[] state)
      {
         throw new UnsupportedOperationException("Not implemented; see http://jira.jboss.com/jira/browse/JBAS-3594");
      }

      public void setState(String state_id, InputStream istream)
      {
         throw new UnsupportedOperationException("Not implemented; see http://jira.jboss.com/jira/browse/JBAS-3594");
      }

      public void receive(org.jgroups.Message msg)
      { /* complete */}
      
      public void setState(byte[] obj)
      {
         logHistory ("setState called on partition");
         try
         {
            if (obj == null)
            {
               log.debug("transferred serviceState is null (may be first member in cluster)");
            }
            else
            {
               ByteArrayInputStream bais = new ByteArrayInputStream(obj);
               setStateInternal(bais);
               bais.close();
            }
            
            isStateSet = true;
         }
         catch (Throwable t)
         {
            recordSetStateFailure(t);
         }
         finally
         {
            // Notify waiting thread that serviceState has been set.
            notifyChannelLock();
         }
      }
      
   }

   /** 
    * A simple data class containing the view change event needed to
    * notify the HAMembershipListeners
    */
   private static class ViewChangeEvent
   {
      long viewId;
      Vector deadMembers;
      Vector newMembers;
      Vector allMembers;
      Vector originatingGroups;
   }
   
   private class RequestMarshallerImpl implements org.jgroups.blocks.RpcDispatcher.Marshaller
   {

      public Object objectFromByteBuffer(byte[] buf) throws Exception
      {
         return objectFromByteBufferInternal(buf);
      }

      public byte[] objectToByteBuffer(Object obj) throws Exception
      {
         // wrap MethodCall in Object[service_name, byte[]] so that service name is available during demarshalling
         if (obj instanceof MethodCall)
         {
            String name = ((MethodCall)obj).getName();
            int idx = name.lastIndexOf('.');
            String serviceName = name.substring(0, idx);
            return objectToByteBufferInternal(new Object[]{serviceName, objectToByteBufferInternal(obj)});           
         }
         else // this shouldn't occur
            return objectToByteBufferInternal(obj);
      }      
   }
   
   private class ResponseMarshallerImpl implements org.jgroups.blocks.RpcDispatcher.Marshaller
   {
      
      public Object objectFromByteBuffer(byte[] buf) throws Exception
      {
         boolean trace = log.isTraceEnabled();
         Object retval = objectFromByteBufferResponseInternal(buf);
         // HAServiceResponse is only received when a scoped classloader is required for unmarshalling
         if (!(retval instanceof HAServiceResponse))
         {
            return retval;
         }
          
         String serviceName = ((HAServiceResponse)retval).getServiceName();
         byte[] payload = ((HAServiceResponse)retval).getPayload();   

         ClassLoader previousCL = null;
         boolean overrideCL = false;
         try
         {
            WeakReference<ClassLoader> weak = clmap.get(serviceName);
            if (weak != null) // this should always be true since we only use HAServiceResponse when classloader is specified
            {
               previousCL = Thread.currentThread().getContextClassLoader();
               ClassLoader loader = weak.get();
               if( trace )
                  log.trace("overriding response Thread ContextClassLoader for service " + serviceName);            
               overrideCL = true;
               Thread.currentThread().setContextClassLoader(loader);
            }
            retval = objectFromByteBufferResponseInternal(payload);
   
            return retval;
         }
         finally
         {
            if (overrideCL == true)
            {
               log.trace("resetting response classloader");
               Thread.currentThread().setContextClassLoader(previousCL);
            }
         }
      }

      public byte[] objectToByteBuffer(Object obj) throws Exception
      {
         return objectToByteBufferResponseInternal(obj);
      }      
   }
   
   /**
    * Overrides RpcDispatcher.Handle so that we can dispatch to many 
    * different objects.
    */
   private class RpcHandler extends RpcDispatcher
   {      
      private RpcHandler(Channel channel, MessageListener l, MembershipListener l2, Object server_obj,
            boolean deadlock_detection)
      {
         super(channel, l, l2, server_obj, deadlock_detection);
      }
      
      /**
       * Analyze the MethodCall contained in <code>req</code> to find the 
       * registered service object to invoke against, and then execute it 
       * against *that* object and return result.
       *
       * This overrides RpcDispatcher.Handle so that we can dispatch to many different objects.
       * @param req The org.jgroups. representation of the method invocation
       * @return The serializable return value from the invocation
       */
      public Object handle(Message req)
      {
         Object body = null;
         Object retval = null;
         Object handler = null;
         boolean trace = log.isTraceEnabled();
         boolean overrideCL = false;
         ClassLoader previousCL = null;
         String service = null;
         byte[] request_bytes = null;
         
         if( trace )
            log.trace("Partition " + getPartitionName() + " received msg");
         if(req == null || req.getBuffer() == null)
         {
            log.warn("Partition " + getPartitionName() + " message or message buffer is null!");
            return null;
         }
         
         try
         {
            Object wrapper = objectFromByteBufferInternal(req.getBuffer());
            if(wrapper == null || !(wrapper instanceof Object[]))
            {
               log.warn("Partition " + getPartitionName() + " message wrapper does not contain Object[] object!");
               return null;
            }

            // wrapper should be Object[]{service_name, byte[]}
            Object[] temp = (Object[])wrapper;
            service = (String)temp[0];
            request_bytes = (byte[])temp[1];

            // see if this node has registered to handle this service
            handler = rpcHandlers.get(service);
            if (handler == null)
            {
               if( trace )
                  log.trace("Partition " + getPartitionName() + " no rpc handler registered under service " + service);
               return new NoHandlerForRPC();
            }
         }
         catch(Exception e)
         {
            log.warn("Partition " + getPartitionName() + " failed unserializing message buffer (msg=" + req + ")", e);
            return null;
         }
         
         try
         {            
            // If client registered the service with a classloader, override the thread classloader here
            WeakReference<ClassLoader> weak = clmap.get(service);
            if (weak != null)
            {
               if( trace )
                  log.trace("overriding Thread ContextClassLoader for RPC service " + service);
               previousCL = Thread.currentThread().getContextClassLoader();
               ClassLoader loader = weak.get();
               overrideCL = true;
               Thread.currentThread().setContextClassLoader(loader);
            }
            body = objectFromByteBufferInternal(request_bytes);
         }
         catch (Exception e)
         {
            log.warn("Partition " + getPartitionName() + " failed extracting message body from request bytes", e);
            return null;
         }
         finally
         {
            if (overrideCL)
            {
               log.trace("resetting Thread ContextClassLoader");
               Thread.currentThread().setContextClassLoader(previousCL);
            }
         }
         
         if(body == null || !(body instanceof MethodCall))
         {
            log.warn("Partition " + getPartitionName() + " message does not contain a MethodCall object!");
            return null;
         }
         
         // get method call information
         MethodCall method_call = (MethodCall)body;
         String methodName = method_call.getName();      
         
         if( trace )
            log.trace("full methodName: " + methodName);
         
         int idx = methodName.lastIndexOf('.');
         String handlerName = methodName.substring(0, idx);
         String newMethodName = methodName.substring(idx + 1);
         if( trace ) 
         {
            log.trace("handlerName: " + handlerName + " methodName: " + newMethodName);
            log.trace("Handle: " + methodName);
         }
         
         // prepare method call
         method_call.setName(newMethodName);

         /* Invoke it and just return any exception with trace level logging of
         the exception. The exception semantics of a group rpc call are weak as
         the return value may be a normal return value or the exception thrown.
         */
         try
         {
            retval = method_call.invoke(handler);
            if (overrideCL)
            {
               // wrap the response so that the service name can be accessed during unmarshalling of the response
               byte[] retbytes = objectToByteBufferResponseInternal(retval);
               retval = new HAServiceResponse(handlerName, retbytes);
            }
            if( trace )
               log.trace("rpc call return value: " + retval);
         }
         catch (Throwable t)
         {
            if( trace )
               log.trace("Partition " + getPartitionName() + " rpc call threw exception", t);
            retval = t;
         }

         return retval;
      }
      
   }
   /**
    * Copyright (c) 2005 Brian Goetz and Tim Peierls
    * Released under the Creative Commons Attribution License
    * (http://creativecommons.org/licenses/by/2.5)
    * Official home: http://www.jcip.net
    * 
    * ThreadGate <p/> Recloseable gate using wait and notifyAll
    * 
    * @author Brian Goetz and Tim Peierls
    */

   private static class ThreadGate {
       // CONDITION-PREDICATE: opened-since(n) (isOpen || generation>n)
       private boolean isOpen;

       private int generation;

       public synchronized void close() 
       {
           isOpen = false;
       }

       public synchronized void open() 
       {
           ++generation;
           isOpen = true;
           notifyAll();
       }

       // BLOCKS-UNTIL: opened-since(generation on entry)
       public synchronized void await() throws InterruptedException 
       {
           int arrivalGeneration = generation;
           while(!isOpen && arrivalGeneration == generation)
               wait();
       }
       
       // BLOCKS-UNTIL: opened-since(generation on entry)
       public synchronized void await(long timeout) throws InterruptedException 
       {
           int arrivalGeneration = generation;
           while(!isOpen && arrivalGeneration == generation)
               wait(timeout);
       }
   }
   
   private void setupLoggers(String partitionName)
   {
      if (partitionName == null)
      {
         this.log = Logger.getLogger(HAPartition.class.getName());
         this.clusterLifeCycleLog = Logger.getLogger(HAPartition.class.getName() + ".lifecycle");
      }
      else
      {
         this.log = Logger.getLogger(HAPartition.class.getName() + "." + partitionName);
         this.clusterLifeCycleLog = Logger.getLogger(HAPartition.class.getName() + ".lifecycle." + partitionName);
      }
   }
   
}
