/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session;

import java.io.Serializable;

import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.Transaction;
import org.jboss.identity.idm.api.PersistenceManager;
import org.jboss.identity.idm.api.RelationshipManager;
import org.jboss.identity.idm.api.AttributesManager;
import org.jboss.identity.idm.api.RoleManager;
import org.jboss.identity.idm.api.query.UserQuery;
import org.jboss.identity.idm.api.query.GroupQuery;
import org.jboss.identity.idm.api.query.RoleQuery;
import org.jboss.identity.idm.exception.IdentityException;
import org.jboss.identity.idm.exception.FeatureNotSupportedException;
import org.jboss.identity.idm.spi.store.IdentityStoreSession;
import org.jboss.identity.idm.spi.store.IdentityStoreInvocationContext;
import org.jboss.identity.idm.spi.repository.IdentityStoreRepository;
import org.jboss.identity.idm.impl.store.SimpleIdentityStoreInvocationContext;
import org.jboss.identity.idm.impl.api.session.context.IdentitySessionContext;
import org.jboss.identity.idm.impl.api.session.context.IdentitySessionContextImpl;
import org.jboss.identity.idm.impl.api.session.context.IdentityStoreInvocationContextResolver;
import org.jboss.identity.idm.impl.api.session.mapper.IdentityObjectTypeMapper;
import org.jboss.identity.idm.impl.api.session.managers.PersistenceManagerImpl;
import org.jboss.identity.idm.impl.api.session.managers.RelationshipManagerImpl;
import org.jboss.identity.idm.impl.api.session.managers.AttributesManagerImpl;
import org.jboss.identity.idm.impl.api.session.managers.RoleManagerImpl;
import org.jboss.identity.idm.impl.api.session.SimpleTransactionImpl;
import org.jboss.identity.idm.impl.api.query.UserQueryImpl;
import org.jboss.identity.idm.impl.api.query.GroupQueryImpl;
import org.jboss.identity.idm.impl.api.query.RoleQueryImpl;
import org.jboss.identity.idm.impl.NotYetImplementedException;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class IdentitySessionImpl implements IdentitySession, Serializable
{

   private static final long serialVersionUID = 7615238887627699243L;

   private final String realmName;

   private final IdentitySessionContext sessionContext;

   private final PersistenceManager persistenceManager;

   private final RelationshipManager relationshipManager;

   private final AttributesManager profileManager;

   private final RoleManager roleManager;

   public IdentitySessionContext getSessionContext()
   {
      return sessionContext;
   }

   public IdentitySessionImpl(String realmName,
                              IdentityStoreRepository repository,
                              IdentityObjectTypeMapper typeMapper) throws IdentityException
   {
      this.realmName = realmName;

      IdentityStoreSession storeSession = repository.createIdentityStoreSession();
      final IdentityStoreInvocationContext invocationCtx = new SimpleIdentityStoreInvocationContext(storeSession, realmName);

      IdentityStoreInvocationContextResolver resolver = new IdentityStoreInvocationContextResolver()
      {
         public IdentityStoreInvocationContext resolveInvocationContext()
         {
            return invocationCtx;
         }
      };

      sessionContext = new IdentitySessionContextImpl(repository, typeMapper, resolver);

      this.persistenceManager = new PersistenceManagerImpl(this);
      this.relationshipManager = new RelationshipManagerImpl(this);
      this.profileManager = new AttributesManagerImpl(this);
      this.roleManager = new RoleManagerImpl(this);

   }





   public String getRealmName()
   {
      return realmName;
   }

   public void close() throws IdentityException
   {
      sessionContext.resolveStoreInvocationContext().getIdentityStoreSession().close();
   }

   public void save() throws IdentityException
   {
      sessionContext.resolveStoreInvocationContext().getIdentityStoreSession().save();
   }

   public void clear() throws IdentityException
   {
      sessionContext.resolveStoreInvocationContext().getIdentityStoreSession().clear();
   }

   public boolean isOpen()
   {

      return sessionContext.resolveStoreInvocationContext().getIdentityStoreSession().isOpen();
   }

   public Transaction beginTransaction()
   {
      Transaction transaction = new SimpleTransactionImpl(sessionContext.resolveStoreInvocationContext().getIdentityStoreSession());
      transaction.start();
      return transaction;
   }

   public Transaction getTransaction()
   {
      return new SimpleTransactionImpl(sessionContext.resolveStoreInvocationContext().getIdentityStoreSession());
   }

   public PersistenceManager getPersistenceManager()
   {
      return persistenceManager;
   }

   public RelationshipManager getRelationshipManager()
   {
      return relationshipManager;
   }

   public AttributesManager getAttributesManager()
   {
      return profileManager;
   }

   public RoleManager getRoleManager() throws FeatureNotSupportedException
   {
      if (!getSessionContext().getIdentityStoreRepository().getSupportedFeatures().isNamedRelationshipsSupported())
      {
         throw new FeatureNotSupportedException("Role management not supported by underlaying configured identity stores");
      }

      return roleManager;
   }

   public UserQuery createUserQuery()
   {
      return new UserQueryImpl(this);

   }

   public GroupQuery createGroupQuery()
   {
      return new GroupQueryImpl(this);

   }

   public RoleQuery createRoleQuery() throws FeatureNotSupportedException
   {
      if (!getSessionContext().getIdentityStoreRepository().getSupportedFeatures().isNamedRelationshipsSupported())
      {
         throw new FeatureNotSupportedException("Role management not supported by underlaying configured identity stores");
      }

      return new RoleQueryImpl(this);
   }
}
