/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.model.hibernate;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.HashMap;
import java.util.Collections;
import java.util.Collection;

import javax.persistence.Column;
import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.FetchType;
import javax.persistence.UniqueConstraint;
import javax.persistence.JoinColumn;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.JoinTable;

import org.jboss.identity.idm.exception.PolicyValidationException;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.model.IdentityObjectCredentialType;
import org.hibernate.annotations.Cascade;
import org.hibernate.annotations.CollectionOfElements;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
@Entity
@Table(name = "identity_obj", uniqueConstraints = {@UniqueConstraint(columnNames = {"NAME", "IDENTITY_TYPE", "REALM"})})
public class HibernateIdentityObject implements IdentityObject
{
   public static final String findIdentityObjectByNameAndType =
      "select o from HibernateIdentityObject o where o.realm like :realm and o.name like :name and o.identityType.name like :typeName";

   public static final String findIdentityObjectsByType =
      "select o from HibernateIdentityObject o where o.name like :nameFilter and o.realm like :realm and o.identityType.name like :typeName";

   public static final String findIdentityObjectsByTypeOrderedByNameAsc =
      "select o from HibernateIdentityObject o where o.name like :nameFilter and o.realm like :realm and o.identityType.name like :typeName " +
         "order by o.name asc";

   public static final String findIdentityObjectsByTypeOrderedByNameDesc =
      "select o from HibernateIdentityObject o where o.name like :nameFilter and o.realm like :realm and o.identityType.name like :typeName " +
         "order by o.name desc";

   public static final String countIdentityObjectsByType =
      "select count(o.id) from HibernateIdentityObject o where o.realm like :realm and o.identityType.name like :typeName";


   @Id
   @GeneratedValue
   private Long id;

   @Column(name = "NAME", nullable = false)
   private String name;

   @ManyToOne(fetch = FetchType.EAGER)
   @JoinColumn(name = "IDENTITY_TYPE", nullable = false)
   private HibernateIdentityObjectType identityType;

   @OneToMany(mappedBy = "fromIdentityObject")
   private Set<HibernateIdentityObjectRelationship> fromRelationships = new HashSet<HibernateIdentityObjectRelationship>();

   @OneToMany(mappedBy = "toIdentityObject")
   private Set<HibernateIdentityObjectRelationship> toRelationships = new HashSet<HibernateIdentityObjectRelationship>();

   @OneToMany(fetch = FetchType.EAGER, cascade = {CascadeType.ALL},
              mappedBy = "identityObject")
   @org.hibernate.annotations.Cascade(
      value = {org.hibernate.annotations.CascadeType.DELETE_ORPHAN}
   )
   private Set<HibernateIdentityObjectAttribute> attributes = new HashSet<HibernateIdentityObjectAttribute>();

   @CollectionOfElements
   @JoinTable(name = "identity_obj_properties", joinColumns = @JoinColumn(name = "IDENTITY_OBJ_ID"))
   @Column(name = "PROPERTY")
   private Map<String, String> properties = new HashMap<String, String>();

   @OneToMany(fetch = FetchType.EAGER, cascade = {CascadeType.ALL})
   @Cascade(value = org.hibernate.annotations.CascadeType.DELETE_ORPHAN)
   private Map<String, HibernateIdentityObjectCredential> credentials = new HashMap<String, HibernateIdentityObjectCredential>();

   @ManyToOne(fetch = FetchType.EAGER)
   @JoinColumn(nullable = false, unique = false, name="REALM")
   private HibernateRealm realm;

   public HibernateIdentityObject()
   {
   }

   public HibernateIdentityObject(String name, HibernateIdentityObjectType identityType, HibernateRealm realm)
   {
      this.name = name;
      this.identityType = identityType;
      this.realm = realm;
   }

   public String getId()
   {
      return id.toString();
   }

   public void setId(String id)
   {
      this.id = Long.valueOf(id);
   }

   public String getName()
   {
      return name;
   }

   public void setName(String name)
   {
      this.name = name;
   }

   public HibernateIdentityObjectType getIdentityType()
   {
      return identityType;
   }

   public void setIdentityType(HibernateIdentityObjectType identityType)
   {
      this.identityType = identityType;
   }

   public String getFQDN()
   {
      return null;
   }

   public Set<HibernateIdentityObjectAttribute> getAttributes()
   {
      return attributes;
   }

   public void setAttributes(Set<HibernateIdentityObjectAttribute> attributes)
   {
      this.attributes = attributes;
   }

   public void addAttribute(HibernateIdentityObjectAttribute attribute)
   {
      attribute.setIdentityObject(this);
      this.attributes.add(attribute);
   }

   public Map<String, Collection> getAttributesAsMap()
   {
      Map<String, Collection> map = new HashMap<String, Collection>();

      for (HibernateIdentityObjectAttribute attribute : attributes)
      {
         Collection values = attribute.getValues();
         map.put(attribute.getName(), values);
      }

      return Collections.unmodifiableMap(map);
   }

   public void addTextAttribute(String name, String[] values)
   {
      attributes.add(new HibernateIdentityObjectTextAttribute(this, name, values));
   }

   public void removeAttribute(String name)
   {
      HibernateIdentityObjectAttribute attributeToRemove = null;

      for (HibernateIdentityObjectAttribute attribute : attributes)
      {
         if (attribute.getName().equals(name))
         {
            attributeToRemove = attribute;
            break;
         }
      }

      if (attributeToRemove != null)
      {
         attributes.remove(attributeToRemove);
      }
   }

   public Set<HibernateIdentityObjectRelationship> getFromRelationships()
   {
      return fromRelationships;
   }

   public void setFromRelationships(Set<HibernateIdentityObjectRelationship> fromRelationships)
   {
      this.fromRelationships = fromRelationships;
   }

   public void addFromRelationship(HibernateIdentityObjectRelationship fromRelationship)
   {
      fromRelationship.setFromIdentityObject(this);
      fromRelationships.add(fromRelationship);
   }

   public Set<HibernateIdentityObjectRelationship> getToRelationships()
   {
      return toRelationships;
   }

   public void setToRelationships(Set<HibernateIdentityObjectRelationship> toRelationships)
   {
      this.toRelationships = toRelationships;
   }

   public void addToRelationship(HibernateIdentityObjectRelationship toRelationship)
   {
      toRelationship.setToIdentityObject(this);
      fromRelationships.add(toRelationship);
   }

   public HibernateRealm getRealm()
   {
      return realm;
   }

   public void setRealm(HibernateRealm realm)
   {
      this.realm = realm;
   }

   public Map<String, HibernateIdentityObjectCredential> getCredentials()
   {
      return credentials;
   }

   public void setCredentials(Map<String, HibernateIdentityObjectCredential> credentials)
   {
      this.credentials = credentials;
   }

   public void addCredential(HibernateIdentityObjectCredential credential)
   {
      credential.setIdentityObject(this);
      credentials.put(credential.getType().getName(), credential);
   }

   public boolean hasCredentials()
   {
      if (credentials != null && credentials.size() > 0)
      {
         return true;
      }
      return false;
   }

   public boolean hasCredential(IdentityObjectCredentialType type)
   {
      if (credentials != null && credentials.keySet().contains(type.getName()))
      {
         return true;
      }
      return false;
   }

   public Map<String, String> getProperties()
   {
      return properties;
   }

   public void setProperties(Map<String, String> properties)
   {
      this.properties = properties;
   }

   public void validatePolicy() throws PolicyValidationException
   {

   }

   @Override
   public String toString()
   {
      return "IdentityObject[id=" + getId() + "; name="  + getName() + "; type=" + getIdentityType().getName() + "]";
   }
}
