/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session.managers;

import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.Identity;
import org.jboss.identity.idm.api.Group;
import org.jboss.identity.idm.api.GroupType;
import org.jboss.identity.idm.api.IdentityType;
import org.jboss.identity.idm.api.IdentitySearchControl;
import org.jboss.identity.idm.api.Attribute;
import org.jboss.identity.idm.spi.repository.IdentityStoreRepository;
import org.jboss.identity.idm.spi.store.IdentityStoreInvocationContext;
import org.jboss.identity.idm.spi.searchcontrol.IdentityObjectSearchControl;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.model.IdentityObjectAttribute;
import org.jboss.identity.idm.impl.types.SimpleIdentityObject;
import org.jboss.identity.idm.impl.api.session.context.IdentitySessionContext;
import org.jboss.identity.idm.impl.api.session.IdentitySessionImpl;
import org.jboss.identity.idm.impl.api.model.SimpleIdentity;
import org.jboss.identity.idm.impl.api.model.SimpleGroup;
import org.jboss.identity.idm.impl.api.SimpleAttribute;

import java.util.List;
import java.util.LinkedList;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public abstract class AbstractManager
{
   protected final IdentitySession identitySession;

   protected AbstractManager(IdentitySession session)
   {
      this.identitySession = session;
   }

   public IdentitySession getIdentitySession()
   {
      return identitySession;
   }

   protected IdentitySessionContext getSessionContext()
   {
      if (identitySession instanceof IdentitySessionImpl)
      {
         return ((IdentitySessionImpl)identitySession).getSessionContext();
      }
      return null;
   }

   protected IdentityStoreRepository getRepository()
   {
      return getSessionContext().getIdentityStoreRepository();
   }

   protected IdentityStoreInvocationContext getInvocationContext()
   {
      return getSessionContext().resolveStoreInvocationContext();
   }

   protected Identity createIdentity(IdentityObject identityObject)
   {
      return new SimpleIdentity(identityObject.getName(), identityObject.getId());
   }

   protected Group createGroup(IdentityObject identityObject)
   {
      GroupType groupType = getSessionContext().getIdentityObjectTypeMapper().getGroupType(identityObject.getIdentityType());
      return new SimpleGroup(identityObject.getName(), identityObject.getId(), groupType);
   }

   protected IdentityObject createIdentityObject(Identity identity)
   {
      IdentityObjectType iot = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

      return new SimpleIdentityObject(identity.getName(), identity.getId(), iot);
   }

   protected IdentityObject createIdentityObject(Group group)
   {
      IdentityObjectType iot = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(group.getGroupType());

      return new SimpleIdentityObject(group.getName(), group.getId(), iot);
   }

   protected IdentityObject createIdentityObject(IdentityType identityType)
   {
      if (identityType instanceof Identity)
      {
         return createIdentityObject((Identity)identityType);
      }
      else if (identityType instanceof Group)
      {
         return createIdentityObject((Group)identityType);
      }

      throw new IllegalStateException("Not supported IdentityType extension: " + identityType.getClass());

   }

   protected IdentityObjectSearchControl[] convertSearchControls(IdentitySearchControl[] controls)
   {
      if (controls == null)
      {
         return null;
      }

      //TODO: simple check
      List<IdentityObjectSearchControl> result = new LinkedList<IdentityObjectSearchControl>();

      for (IdentitySearchControl control : controls)
      {
         if (control instanceof IdentityObjectSearchControl)
         {
            result.add((IdentityObjectSearchControl)control);
         }
      }

      if (result.size() == 0)
      {
         return null;
      }
      
      return result.toArray(new IdentityObjectSearchControl[result.size()]);

   }

   protected IdentityObjectType getIdentityObjectType()
   {
      return getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();
   }

   protected IdentityObjectType getIdentityObjectType(GroupType groupType)
   {
      return getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);
   }

   protected IdentityObjectAttribute[] convertAttributes(Attribute[] attributes)
   {
      IdentityObjectAttribute[] convertedAttributes = new IdentityObjectAttribute[attributes.length];

      for (int i = 0; i < attributes.length; i++)
      {
         convertedAttributes[i] = convertAttribute(attributes[i]);
      }
      return convertedAttributes;
   }

   protected Attribute[] convertAttributes(IdentityObjectAttribute[] attributes)
   {
      Attribute[] convertedAttributes = new Attribute[attributes.length];

      for (int i = 0; i < attributes.length; i++)
      {
         convertedAttributes[i] = convertAttribute(attributes[i]);
      }
      return convertedAttributes;
   }

   protected Attribute convertAttribute(IdentityObjectAttribute attribute)
   {
      if (attribute instanceof Attribute)
      {
         return (Attribute)attribute;
      }
      else
      {
         return new SimpleAttribute(attribute);
      }
   }

   protected IdentityObjectAttribute convertAttribute(Attribute attribute)
   {
      if (attribute instanceof IdentityObjectAttribute)
      {
         return (IdentityObjectAttribute)attribute;
      }
      else
      {
         return new SimpleAttribute(attribute);
      }
   }


   
}
