/*
 *  Copyright 2022 Red Hat
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.jboss.hal.client.accesscontrol;

import java.util.Arrays;

import javax.inject.Inject;

import org.jboss.hal.config.Environment;
import org.jboss.hal.core.accesscontrol.AccessControl;
import org.jboss.hal.core.finder.ColumnAction;
import org.jboss.hal.core.finder.ColumnActionFactory;
import org.jboss.hal.core.finder.Finder;
import org.jboss.hal.core.finder.PreviewContent;
import org.jboss.hal.core.finder.StaticItem;
import org.jboss.hal.core.finder.StaticItemColumn;
import org.jboss.hal.resources.Ids;
import org.jboss.hal.resources.Previews;
import org.jboss.hal.resources.Resources;
import org.jboss.hal.spi.AsyncColumn;

import com.google.gwt.resources.client.ExternalTextResource;

import elemental2.dom.HTMLElement;

import static org.jboss.elemento.Elements.div;
import static org.jboss.hal.resources.CSS.fontAwesome;

@AsyncColumn(Ids.ACCESS_CONTROL_BROWSE_BY)
public class BrowseByColumn extends StaticItemColumn {

    private static class TopLevelPreview extends PreviewContent<StaticItem> {

        TopLevelPreview(String header, ExternalTextResource resource,
                AccessControl accessControl, Resources resources) {
            super(header);

            AccessControlWarnings warnings = new AccessControlWarnings(accessControl, resources);
            previewBuilder().add(warnings.providerWarning);
            previewBuilder().add(warnings.ssoWarning);

            HTMLElement content = div().element();
            Previews.innerHtml(content, resource);
            previewBuilder().add(content);
        }
    }

    @Inject
    public BrowseByColumn(Finder finder, AccessControl accessControl, Environment environment,
            ColumnActionFactory columnActionFactory, Resources resources) {
        super(finder, Ids.ACCESS_CONTROL_BROWSE_BY, resources.constants().browseBy(),
                Arrays.asList(
                        new StaticItem.Builder(resources.constants().users())
                                .id(Ids.ACCESS_CONTROL_BROWSE_BY_USERS)
                                .onPreview(new TopLevelPreview(resources.constants().users(),
                                        resources.previews().rbacUsers(), accessControl,
                                        resources))
                                .nextColumn(Ids.USER)
                                .build(),
                        new StaticItem.Builder(resources.constants().groups())
                                .id(Ids.ACCESS_CONTROL_BROWSE_BY_GROUPS)
                                .onPreview(new TopLevelPreview(resources.constants().groups(),
                                        resources.previews().rbacGroups(), accessControl,
                                        resources))
                                .nextColumn(Ids.GROUP)
                                .build(),
                        new StaticItem.Builder(resources.constants().roles())
                                .id(Ids.ACCESS_CONTROL_BROWSE_BY_ROLES)
                                .onPreview(new TopLevelPreview(resources.constants().roles(),
                                        environment.isStandalone() ? resources.previews()
                                                .rbacRolesStandalone() : resources.previews().rbacRolesDomain(),
                                        accessControl, resources))
                                .nextColumn(Ids.ROLE)
                                .build()));

        addColumnAction(new ColumnAction.Builder<StaticItem>(Ids.ACCESS_CONTROL_SWITCH_PROVIDER)
                .element(columnActionFactory.addButton(resources.constants().switchProvider(), fontAwesome("shield")))
                .handler(column -> accessControl.switchProvider())
                .build());
    }
}
