/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.timerservice.deployer;

import org.jboss.beans.metadata.plugins.AbstractInjectionValueMetaData;
import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.ejb3.EJBContainer;
import org.jboss.ejb3.common.deployers.spi.AttachmentNames;
import org.jboss.logging.Logger;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeanMetaData;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeansMetaData;
import org.jboss.metadata.ejb.jboss.JBossMetaData;
import org.jboss.metadata.ejb.jboss.JBossSessionBeanMetaData;

/**
 * A deployer which processing EJB3.1 bean metadata and creating {@link AutoTimerInitializer}
 * for appropriate EJBs.
 * <p>
 *  Each such {@link AutoTimerInitializer} will be attached to the {@link DeploymentUnit} as a
 *  {@link BeanMetaData} so that it's deployed as a MC bean.
 * </p>
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class TimerServiceDeployer extends AbstractDeployer
{

   /**
    * Logger
    */
   private Logger logger = Logger.getLogger(TimerServiceDeployer.class);

   /**
    * MC bean name prefix
    */
   private static final String AUTO_TIMER_MC_BEAN_PREFIX = "auto-timer-initializer:";

   /**
    * Setup the deployer
    */
   public TimerServiceDeployer()
   {
      // run in REAL stage
      setStage(DeploymentStages.REAL);
      // we need JBossMetaData
      setInput(JBossMetaData.class);
      // ordering (i.e. this deployer should run after the deployer which attaches PROCESSED_METADATA to the unit) 
      addInput(AttachmentNames.PROCESSED_METADATA);
      // we deploy MC bean(s)
      addOutput(BeanMetaData.class);
   }

   /**
    * For each EJB3.1 bean in this <code>unit</code>, this method creates a {@link AutoTimerInitializer} and attaches it
    * as {@link BeanMetaData} to the <code>unit</code> 
    */
   @Override
   public void deploy(DeploymentUnit unit) throws DeploymentException
   {
      // get processed metadata
      JBossMetaData metaData = unit.getAttachment(AttachmentNames.PROCESSED_METADATA, JBossMetaData.class);
      if (metaData == null)
      {
         return;
      }
      // Auto timers are only since EJB3.1
      // TODO: This check will fail when EJB3.2 or any newer versions are introduced.
      // A better check would be to look for any version greater than EJB3.1
      if (metaData.isEJB31() == false)
      {
         return;
      }
      // work on the ejbs
      JBossEnterpriseBeansMetaData beans = metaData.getEnterpriseBeans();
      for (JBossEnterpriseBeanMetaData bean : beans)
      {
         if (bean.isSession())
         {
            JBossSessionBeanMetaData sessionBean = (JBossSessionBeanMetaData) bean;
            // stateful beans don't have timerservice/timers
            if (sessionBean.isStateful())
            {
               continue;
            }
         }
         // ignore entity and (JBoss specific) @Service beans
         else if (bean.isEntity() || bean.isService())
         {
            continue;
         }
         // process
         String mcBeanName = AUTO_TIMER_MC_BEAN_PREFIX + unit.getName() + "$" + bean.getEjbName();
         // create the BeanMetadata for the AutoTimerInitializer
         BeanMetaData bmd = this.createAutoTimerInitializer(mcBeanName, bean);
         // add it as attachment
         unit.addAttachment(BeanMetaData.class + ":" + mcBeanName, bmd);
      }
   }

   /**
    * Returns {@link BeanMetaData} for a {@link AutoTimerInitializer} instance
    * <p>
    *   The MC bean for {@link AutoTimerInitializer} created by this method will be configured
    *   to depend on the {@link ControllerState#INSTALLED} state of the MC bean corresponding
    *   to the {@link EJBContainer}. The EJB container MC bean on which the {@link AutoTimerInitializer}
    *   will depend, is obtained through {@link JBossEnterpriseBeanMetaData#getContainerName()}. 
    * </p>  
    * <p>
    *   Furthermore, the MC bean for {@link AutoTimerInitializer} will be configured to inject
    *   the {@link EJBContainer} into the {@link AutoTimerInitializer} instance
    * </p>
    * @param mcBeanName The name of the MC bean being created
    * @param bean Bean metadata
    * @return
    */
   private BeanMetaData createAutoTimerInitializer(String mcBeanName, JBossEnterpriseBeanMetaData bean)
   {
      AutoTimerInitializer autoTimerInitializer = new AutoTimerInitializer();

      BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(mcBeanName, autoTimerInitializer.getClass()
            .getName());
      builder.setConstructorValue(autoTimerInitializer);

      // add dependency - the AutoTimerInitializer will depend on a INSTALLED state of EJBContainer
      AbstractInjectionValueMetaData injectMetaData = new AbstractInjectionValueMetaData(bean.getContainerName());
      injectMetaData.setDependentState(ControllerState.INSTALLED);

      // TODO: Too bad we have to know the field name. Need to do more research on MC to see if we can
      // add property metadata based on type instead of field name.
      // Add the injection point in AutoTimerInitializer to inject the EJBContainer
      builder.addPropertyMetaData("container", injectMetaData);

      // return the BeanMetaData
      return builder.getBeanMetaData();
   }
}
