/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.web.tomcat.service.session.distributedcache.jbc;

import org.jboss.cache.Cache;
import org.jboss.metadata.web.jboss.ReplicationGranularity;
import org.jboss.web.tomcat.service.session.distributedcache.spi.ClusteringNotSupportedException;
import org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManagerFactory;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;

/**
 * @author Brian Stansberry
 *
 */
public class DistributedCacheManagerFactoryImpl implements DistributedCacheManagerFactory
{
   public static final String DEFAULT_CLUSTER_NAME = "Tomcat-Cluster";

   /** TreeCache's isolation level */
   public static final String DEFAULT_ISOLATION_LEVEL = "REPEATABLE_READ";
   
   /** TreeCache's cache mode */
   public static final String DEFAULT_CACHE_MODE = "REPL_ASYNC";
   
   /** TreeCache's lock aquisition timeout */
   public static final long DEFAULT_LOCK_TIMEOUT = 15000;
   
   /** TransactionManagerLookup implementation that the TreeCache should use. */
   @SuppressWarnings("deprecation")
   public static final String DEFAULT_TM_LOOKUP = 
      org.jboss.cache.transaction.BatchModeTransactionManagerLookup.class.getName();
   
   private Cache<Object, Object> plainCache;
   
   @SuppressWarnings("unchecked")
   public DistributedCacheManager getDistributedCacheManager(LocalDistributableSessionManager localManager)
         throws ClusteringNotSupportedException
   {
      ReplicationGranularity granularity = Util.getReplicationGranularity(localManager);
      switch(granularity)
      {
         case SESSION:
            return plainCache == null? new SessionBasedJBossCacheService(localManager) : new SessionBasedJBossCacheService(localManager, plainCache);
         case ATTRIBUTE:
            return plainCache == null? new AttributeBasedJBossCacheService(localManager) : new AttributeBasedJBossCacheService(localManager, plainCache);
         default:
            throw new IllegalStateException("Unknown ReplicationGranularity " + granularity);
      }
   }

   public void stop() throws Exception
   {
      if (plainCache != null)
      {
         plainCache.stop();
         plainCache.destroy();
      }
   }

   public Cache<Object, Object> getPlainCache()
   {
      return plainCache;
   }

   /** 
    * Hook for test fixtures to inject a Cache, which if present will
    * be used to create the DistributedCacheManager in preference to any
    * passed <code>cacheConfigName</code>.
    */
   public void setPlainCache(Cache<Object, Object> plainCache)
   {
      this.plainCache = plainCache;
   }
   
   /**
    * Convenience method for test fixtures to clear any injected cache.
    */
   public void clearCaches()
   {
      this.plainCache = null;
   }
}
