/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.ha.cachemanager;

import java.security.AccessController;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jboss.cache.Cache;
import org.jboss.cache.CacheException;
import org.jboss.cache.CacheStatus;
import org.jboss.cache.Fqn;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.Node;
import org.jboss.cache.NodeNotExistsException;
import org.jboss.cache.Region;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.interceptors.base.CommandInterceptor;
import org.jboss.logging.Logger;
import org.jboss.util.loading.ContextClassLoaderSwitcher;
import org.jgroups.Address;

/**
 * Wrapper around a cache that 1) ensures the calling thread's TCCL doesn't
 * leak to cache threads via calls to create/start; and 2) logs WARNS about calls
 * to stop/destroy as these should be handled by the CacheManager.
 * 
 * TODO disable calls to stop/destroy once testsuite cleanup code is fixed
 * to not call those methods.
 * 
 * @author Brian Stansberry
 */
class CacheManagerManagedCache implements Cache
{
   private static final Logger log = Logger.getLogger(CacheManagerManagedCache.class);
   
   private final Cache delegate;
   private final ContextClassLoaderSwitcher switcher;
   
   CacheManagerManagedCache(Cache delegate)
   {
      assert delegate != null : "delegate is null";
      this.delegate = delegate;
      this.switcher = (ContextClassLoaderSwitcher) AccessController.doPrivileged(ContextClassLoaderSwitcher.INSTANTIATOR);
   }

   /**
    * Switches the TCCL to the delegate's classloader before calling create()
    * on the delegate.
    */
   public void create() throws CacheException
   {
      ContextClassLoaderSwitcher.SwitchContext switchContext = switcher.getSwitchContext();
      try
      {
         switchContext.setClassLoader(delegate.getClass().getClassLoader());
         delegate.create();
      }
      finally
      {
         switchContext.reset();
      }
   }

   /**
    * Switches the TCCL to the delegate's classloader before calling start()
    * on the delegate.
    */
   public void start() throws CacheException
   {
      ContextClassLoaderSwitcher.SwitchContext switchContext = switcher.getSwitchContext();
      try
      {
         switchContext.setClassLoader(delegate.getClass().getClassLoader());
         delegate.start();
      }
      finally
      {
         switchContext.reset();
      }
   }

   /**
    * TODO: Log a WARN and do nothing else; currently logs and then calls 
    * through to delegate.
    */
   public void stop()
   {
      log.warn("stop() should not be directly called on caches obtained from a CacheManager -- use CacheManager.releaseCache()", 
            new UnsupportedOperationException("stop() is not supported"));
      delegate.stop();
   }

   /**
    * TODO: Log a WARN and do nothing else; currently logs and then calls 
    * through to delegate.
    */
   public void destroy()
   {
      log.warn("destroy() should not be directly called on caches obtained from a CacheManager -- use CacheManager.releaseCache()", 
            new UnsupportedOperationException("destroy() is not supported"));
      delegate.destroy();
   }
   
   // ------------------------------------------------------------------ Cache
   
   public void addCacheListener(Object arg0)
   {
      delegate.addCacheListener(arg0);
   }

   public void addInterceptor(CommandInterceptor arg0, Class arg1)
   {
      delegate.addInterceptor(arg0, arg1);      
   }

   public void addInterceptor(CommandInterceptor arg0, int arg1)
   {
      delegate.addInterceptor(arg0, arg1); 
   }

   public void removeInterceptor(Class arg0)
   {
      delegate.removeInterceptor(arg0);
   }

   public void removeInterceptor(int arg0)
   {
      delegate.removeInterceptor(arg0);
   }

   public void startBatch()
   {
      delegate.startBatch();      
   }

   public void endBatch(boolean arg0)
   {
      delegate.endBatch(arg0);
   }

   public void clearData(String arg0)
   {
      delegate.clearData(arg0);
   }

   public void clearData(Fqn arg0)
   {
      delegate.clearData(arg0);
   }

   public void evict(Fqn arg0)
   {
      delegate.evict(arg0);
   }

   public void evict(Fqn arg0, boolean arg1)
   {
      delegate.evict(arg0, arg1);
   }

   public Object get(Fqn arg0, Object arg1)
   {
      return delegate.get(arg0, arg1);
   }

   public Object get(String arg0, Object arg1)
   {
      return delegate.get(arg0, arg1);
   }

   public Set getCacheListeners()
   {
      return delegate.getCacheListeners();
   }

   public CacheStatus getCacheStatus()
   {
      return delegate.getCacheStatus();
   }

   public Configuration getConfiguration()
   {
      return delegate.getConfiguration();
   }

   public Map getData(Fqn arg0)
   {
      return delegate.getData(arg0);
   }

   public InvocationContext getInvocationContext()
   {
      return delegate.getInvocationContext();
   }

   public Set getKeys(String arg0)
   {
      return getKeys(arg0);
   }

   public Set getKeys(Fqn arg0)
   {
      return getKeys(arg0);
   }

   public Address getLocalAddress()
   {
      return delegate.getLocalAddress();
   }

   public List getMembers()
   {
      return delegate.getMembers();
   }

   public Node getNode(Fqn arg0)
   {
      return delegate.getNode(arg0);
   }

   public Node getNode(String arg0)
   {
      return delegate.getNode(arg0);
   }

   public Region getRegion(Fqn arg0, boolean arg1)
   {
      return delegate.getRegion(arg0, arg1);
   }

   public Node getRoot()
   {
      return delegate.getRoot();
   }

   public String getVersion()
   {
      return delegate.getVersion();
   }

   public void move(Fqn arg0, Fqn arg1) throws NodeNotExistsException
   {
      delegate.move(arg0, arg1);
   }

   public void move(String arg0, String arg1) throws NodeNotExistsException
   {
      delegate.move(arg0, arg1);
   }

   public void put(Fqn arg0, Map arg1)
   {
      delegate.put(arg0, arg1);
   }

   public void put(String arg0, Map arg1)
   {
      delegate.put(arg0, arg1);
   }

   public Object put(Fqn arg0, Object arg1, Object arg2)
   {
      return delegate.put(arg0, arg1, arg2);
   }

   public Object put(String arg0, Object arg1, Object arg2)
   {
      return delegate.put(arg0, arg1, arg2);
   }

   public void putForExternalRead(Fqn arg0, Object arg1, Object arg2)
   {
      delegate.putForExternalRead(arg0, arg1, arg2);
   }

   public Object remove(Fqn arg0, Object arg1)
   {
      return delegate.remove(arg0, arg1);
   }

   public Object remove(String arg0, Object arg1)
   {
      return delegate.remove(arg0, arg1);
   }

   public void removeCacheListener(Object arg0)
   {
      delegate.removeCacheListener(arg0);
   }

   public boolean removeNode(Fqn arg0)
   {
       return delegate.removeNode(arg0);
   }

   public boolean removeNode(String arg0)
   {
      return delegate.removeNode(arg0);
   }

   public boolean removeRegion(Fqn arg0)
   {
      return delegate.removeRegion(arg0);
   }

   public void setInvocationContext(InvocationContext arg0)
   {
      delegate.setInvocationContext(arg0);
   }

   // --------------------------------------------------------------  Overrides

   @Override
   public boolean equals(Object obj)
   {
      if (obj instanceof CacheManagerManagedCache)
      {
         CacheManagerManagedCache other = (CacheManagerManagedCache) obj;
         return delegate.equals(other.delegate);
      }
      return false;
   }

   @Override
   public int hashCode()
   {
      return delegate.hashCode();
   }
   
   
}
