/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.ha.cachemanager;

import java.security.AccessController;
import java.util.Collection;
import java.util.Map;
import java.util.regex.Pattern;

import org.jboss.cache.Cache;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.pojo.PojoCache;
import org.jboss.cache.pojo.PojoCacheException;
import org.jboss.cache.pojo.PojoCacheThreadContext;
import org.jboss.logging.Logger;
import org.jboss.util.loading.ContextClassLoaderSwitcher;

/**
 * Wrapper around a PojoCache that 1) ensures the calling thread's TCCL doesn't
 * leak to cache threads via calls to create/start; and 2) logs WARNS about calls
 * to stop/destroy as these should be handled by the PojoCacheManager.
 * 
 * TODO disable calls to stop/destroy once testsuite cleanup code is fixed
 * to not call those methods.
 * 
 * @author Brian Stansberry
 *
 */
class PojoCacheManagerManagedPojoCache implements PojoCache
{
   private static final Logger log = Logger.getLogger(PojoCacheManagerManagedPojoCache.class);
   
   private final PojoCache delegate;
   private final ContextClassLoaderSwitcher switcher;
   
   PojoCacheManagerManagedPojoCache(PojoCache delegate)
   {
      assert delegate != null : "delegate is null";
      this.delegate = delegate;
      this.switcher = (ContextClassLoaderSwitcher) AccessController.doPrivileged(ContextClassLoaderSwitcher.INSTANTIATOR);
   }

   /**
    * Switches the TCCL to the delegate's classloader before calling create()
    * on the delegate.
    */
   public void create() throws CacheException
   {
      ContextClassLoaderSwitcher.SwitchContext switchContext = switcher.getSwitchContext();
      try
      {
         switchContext.setClassLoader(delegate.getClass().getClassLoader());
         delegate.create();
      }
      finally
      {
         switchContext.reset();
      }
   }

   /**
    * Switches the TCCL to the delegate's classloader before calling start()
    * on the delegate.
    */
   public void start() throws CacheException
   {
      ContextClassLoaderSwitcher.SwitchContext switchContext = switcher.getSwitchContext();
      try
      {
         switchContext.setClassLoader(delegate.getClass().getClassLoader());
         delegate.start();
      }
      finally
      {
         switchContext.reset();
      }
   }

   /**
    * TODO: Log a WARN and do nothing else; currently logs and then calls 
    * through to delegate.
    */
   public void stop() throws PojoCacheException
   {
      log.warn("stop() should not be directly called on caches obtained from a PojoCacheManager -- use CacheManager.releaseCache()", 
            new UnsupportedOperationException("stop() is not supported"));
      delegate.stop();
   }

   /**
    * TODO: Log a WARN and do nothing else; currently logs and then calls 
    * through to delegate.
    */
   public void destroy() throws PojoCacheException
   {
      log.warn("destroy() should not be directly called on caches obtained from a PojoCacheManager -- use CacheManager.releaseCache()", 
            new UnsupportedOperationException("destroy() is not supported"));
      delegate.destroy();
   }
   
   public void addListener(Object arg0)
   {
      delegate.addListener(arg0);
   }

   public void addListener(Object arg0, Pattern arg1)
   {
      delegate.addListener(arg0, arg1);
   }

   public Object attach(String arg0, Object arg1) throws PojoCacheException
   {
      return delegate.attach(arg0, arg1);
   }

   public Object attach(Fqn<?> arg0, Object arg1) throws PojoCacheException
   {
      return delegate.attach(arg0, arg1);
   }

   public Object detach(String arg0) throws PojoCacheException
   {
      return delegate.detach(arg0);
   }

   public Object detach(Fqn<?> arg0) throws PojoCacheException
   {
      return delegate.detach(arg0);
   }

   public boolean exists(Fqn<?> arg0)
   {
      return delegate.exists(arg0);
   }

   public Object find(String arg0) throws PojoCacheException
   {
      return delegate.find(arg0);
   }

   public Object find(Fqn<?> arg0) throws PojoCacheException
   {
      return delegate.find(arg0);
   }

   public Map<Fqn<?>, Object> findAll(String arg0) throws PojoCacheException
   {
      return delegate.findAll(arg0);
   }

   public Map<Fqn<?>, Object> findAll(Fqn<?> arg0) throws PojoCacheException
   {
      return delegate.findAll(arg0);
   }

   public Cache<Object, Object> getCache()
   {
      Cache c = delegate.getCache();
      return c == null ? null : new CacheManagerManagedCache(c);
   }

   public Collection<Object> getListeners()
   {
      return delegate.getListeners();
   }

   public String getPojoID(Object arg0)
   {
      return delegate.getPojoID(arg0);
   }

   public PojoCacheThreadContext getThreadContext()
   {
      return delegate.getThreadContext();
   }

   public void removeListener(Object arg0)
   {
      delegate.removeListener(arg0);
   }

   // --------------------------------------------------------------  Overrides

   @Override
   public boolean equals(Object obj)
   {
      if (obj instanceof PojoCacheManagerManagedPojoCache)
      {
         PojoCacheManagerManagedPojoCache other = (PojoCacheManagerManagedPojoCache) obj;
         return delegate.equals(other.delegate);
      }
      return false;
   }

   @Override
   public int hashCode()
   {
      return delegate.hashCode();
   }

}
