/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.tomcat.service.session.distributedcache.ispn;

import java.util.HashMap;
import java.util.Map;

import org.infinispan.Cache;
import org.infinispan.config.Configuration;
import org.infinispan.config.Configuration.CacheMode;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.remoting.transport.jgroups.JGroupsTransport;
import org.jboss.ha.core.framework.server.CoreGroupCommunicationService;
import org.jboss.ha.framework.server.lock.SharedLocalYieldingClusterLockManager;
import org.jboss.ha.ispn.CacheContainerRegistry;
import org.jboss.metadata.web.jboss.ReplicationConfig;
import org.jboss.metadata.web.jboss.ReplicationMode;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;
import org.jgroups.Channel;

/**
 * @author Paul Ferraro
 *
 */
public class DefaultCacheSource implements CacheSource, LockManagerSource
{
   /** The scope assigned to our group communication service */
   public static final Short SCOPE_ID = Short.valueOf((short) 222);
   /** The service name of the group communication service */
   public static final String SERVICE_NAME = "HTTPSESSIONOWNER";
   
   public static final String DEFAULT_CACHE_CONTAINER = "session";
   
   private final CacheContainerRegistry registry;
   
   private String defaultContainerName = DEFAULT_CACHE_CONTAINER;
   
   private final Map<String,SharedLocalYieldingClusterLockManager> lockManagers = new HashMap<String, SharedLocalYieldingClusterLockManager>();
   
   public DefaultCacheSource(CacheContainerRegistry registry)
   {
      this.registry = registry;
   }
   
   @Override
   public SharedLocalYieldingClusterLockManager getLockManager(LocalDistributableSessionManager manager)
   {
      String containerName = getContainerName(manager.getReplicationConfig());

      synchronized (lockManagers)
      {
         SharedLocalYieldingClusterLockManager lockManager = lockManagers.get(containerName);
         
         if (lockManager == null)
         {
            EmbeddedCacheManager container = this.registry.getCacheContainer(containerName);            
            String cacheName = getCacheName(manager); 
            Cache<Object, Object> cache = container.getCache(cacheName);
            JGroupsTransport transport = (JGroupsTransport) cache.getAdvancedCache().getRpcManager().getTransport();
            Channel channel = transport.getChannel();
            
            CoreGroupCommunicationService gcs = new CoreGroupCommunicationService();
            gcs.setChannel(channel);
            gcs.setScopeId(SCOPE_ID);
            
            try
            {
               gcs.start();
            }
            catch (Exception e)
            {
               throw new IllegalStateException("Unexpected exception while starting group communication service for " + containerName);
            }
            
            lockManager = new SharedLocalYieldingClusterLockManager(SERVICE_NAME, gcs, gcs);
            
            try
            {
               lockManager.start();
               
               this.lockManagers.put(containerName, lockManager);
            }
            catch (Exception e)
            {
               gcs.stop();
               throw new IllegalStateException("Unexpected exception while starting lock manager for " + containerName);
            }
         }
         
         return lockManager;
      }
   }
   

   @Override
   public <K, V> Cache<K, V> getCache(LocalDistributableSessionManager manager)
   {
      ReplicationConfig replConfig = manager.getReplicationConfig();
      String containerName = getContainerName(replConfig);
      
      EmbeddedCacheManager container = this.registry.getCacheContainer(containerName);
      String cacheName = getCacheName(manager);
      
      Cache<?, ?> templateCache = container.getCache();
      Configuration configuration = templateCache.getConfiguration().clone();
      CacheMode mode = getCacheMode(replConfig, configuration);
      configuration.setCacheMode(mode);
      container.defineConfiguration(cacheName, configuration);
      return container.getCache(cacheName);
   }
   
   private String getContainerName(ReplicationConfig replConfig)
   {
      String templateCacheName = replConfig.getCacheName();
      String containerName = this.defaultContainerName;
      
      if (templateCacheName != null)
      {
         String[] parts = templateCacheName.split(":");
         
         if (parts.length == 2)
         {
            containerName = parts[0];
         }
      }
      
      return containerName;
   }

   private CacheMode getCacheMode(ReplicationConfig replConfig, Configuration configuration)
   {
      Integer backups = replConfig.getBackups();
      ReplicationMode replMode = replConfig.getReplicationMode();
      
      CacheMode mode = configuration.getCacheMode();
      
      if (backups != null)
      {
         int value = backups.intValue();
         
         configuration.setNumOwners(value);
         
         if (value == 0)
         {
            mode = CacheMode.LOCAL;
         }
         else
         {
            boolean synchronous = mode.isSynchronous();
            if (value > 0)
            {
               mode = synchronous ? CacheMode.DIST_SYNC : CacheMode.DIST_ASYNC;
            }
            else // Negative backups means total replication
            {
               mode = synchronous ? CacheMode.REPL_SYNC : CacheMode.REPL_ASYNC;
            }
         }
      }
      
      if (replMode != null)
      {
         switch (replMode)
         {
            case SYNCHRONOUS:
            {
               mode = mode.toSync();
               break;
            }
            case ASYNCHRONOUS:
            {
               mode = mode.toAsync();
               break;
            }
         }
      }
      return mode;
   }
   
   private String getCacheName(LocalDistributableSessionManager manager)
   {
      String hostName = manager.getHostName();
      String host = (hostName == null) || hostName.isEmpty() ? "localhost" : hostName;
      
      String context = manager.getContextName();
      String path = context.isEmpty() || context.equals("/") ? "ROOT" : context.startsWith("/") ? context.substring(1) : context;

      return host + "/" + path;
   }   
   
   public void setDefaultContainerName(String defaultContainerName)
   {
      this.defaultContainerName = defaultContainerName;
   }
}
