/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.api.test;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.management.ObjectName;

import org.jboss.bpm.api.ProcessEngineException;
import org.jboss.bpm.api.model.Message;
import org.jboss.bpm.api.model.MessageListener;
import org.jboss.bpm.api.model.ProcessDefinition;
import org.jboss.bpm.api.model.Signal;
import org.jboss.bpm.api.model.SignalListener;
import org.jboss.bpm.api.model.Signal.SignalType;
import org.jboss.bpm.api.model.builder.SignalBuilder;
import org.jboss.bpm.api.service.DialectHandler;
import org.jboss.bpm.api.service.DialectHandlerService;
import org.jboss.bpm.api.service.MessageService;
import org.jboss.bpm.api.service.ProcessDefinitionService;
import org.jboss.bpm.api.service.ProcessEngine;
import org.jboss.bpm.api.service.SignalBuilderService;
import org.jboss.bpm.api.service.SignalService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A CTS test case
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 25-Sep-2008
 */
public class CTSTestCase extends APITestCase
{
  // Provide logging
  private static final Logger log = LoggerFactory.getLogger(CTSTestCase.class);

  // The embedded SignalListener
  private SignalListener signalListener;
  // The signals caught by this test case
  private List<Signal> signals = new ArrayList<Signal>();
  // The embedded MessageListener
  private MessageListener messageListener;
  // The messages caught by this test case
  private List<Message> messages = new ArrayList<Message>();
  // Unregister this process definition on tearDown
  private ProcessDefinition procDef;

  protected ProcessDefinition unregisterOnTearDown(ProcessDefinition procDef)
  {
    this.procDef = procDef;
    ProcessDefinitionService pdService = getProcessEngine().getService(ProcessDefinitionService.class);
    return pdService.registerProcessDefinition(procDef);
  }

  @Override
  protected void setUp() throws Exception
  {
    super.setUp();

    ProcessEngine engine = getProcessEngine();

    // Setup the SignalListener
    SignalService sigService = engine.getService(SignalService.class);
    if (sigService != null)
    {
      clearAllSignalListeners(sigService);
      sigService.addSignalListener(getSignalListener());
      synchronized (signals)
      {
        signals.clear();
      }
    }
    
    // Setup the MessageListener
    MessageService msgService = engine.getService(MessageService.class);
    if (msgService != null)
    {
      clearAllMessageListeners(msgService);
      msgService.addMessageListener(getMessageListener());
    }
  }

  @Override
  protected void tearDown() throws Exception
  {
    ProcessEngine engine = getProcessEngine();

    // Auto unregister
    ProcessDefinitionService procDefService = engine.getService(ProcessDefinitionService.class);
    if (procDef != null)
    {
      procDefService.unregisterProcessDefinition(procDef.getKey());
      procDef = null;
    }

    // Check that there are no registered processes left
    Set<ObjectName> procs = procDefService.getProcessDefinitions();
    if (procs.size() > 0)
    {
      String logMsg = "Registered processes on tear down of " + getName() + ": " + procs;
      System.out.println(logMsg);
      log.error(logMsg);
    }

    // Tear down the SignalListener
    SignalService sigService = engine.getService(SignalService.class);
    if (sigService != null)
    {
      sigService.removeSignalListener(getSignalListener());
      
      // Check that there are no registered signal listeners left
      Set<SignalListener> sigListeners = sigService.getSignalListeners();
      if (sigListeners.size() > 0)
      {
        String logMsg = "Registered signal listeners on tear down of " + getName() + ": " + sigListeners;
        System.out.println(logMsg);
        log.error(logMsg);
      }
    }
    
    // Tear down the MessageListener
    MessageService msgService = engine.getService(MessageService.class);
    if (msgService != null)
    {
      msgService.removeMessageListener(getTestID());
      
      // Check that there are no registered message listeners left
      Set<MessageListener> msgListeners = msgService.getMessageListeners();
      if (msgListeners.size() > 0)
      {
        String logMsg = "Registered message listeners on tear down of " + getName() + ": " + msgListeners;
        System.out.println(logMsg);
        log.error(logMsg);
      }
    }
    
    super.tearDown();
  }

  public List<Signal> getSignals()
  {
    synchronized (signals)
    {
      return Collections.unmodifiableList(signals);
    }
  }

  public List<Message> getMessages()
  {
    synchronized (messages)
    {
      return Collections.unmodifiableList(messages);
    }
  }

  protected Signal newSignal(ObjectName fromRef, SignalType signalType, String message)
  {
    SignalBuilder sigBuilder = getProcessEngine().getService(SignalBuilderService.class).getSignalBuilder();
    return sigBuilder.newSignal(signalType, fromRef, message);
  }

  public List<Signal> getSignals(Signal.SignalType type)
  {
    synchronized (signals)
    {
      List<Signal> retSignals = new ArrayList<Signal>();
      for (Signal sig : signals)
      {
        if (sig.getSignalType() == type)
          retSignals.add(sig);
      }
      return Collections.unmodifiableList(retSignals);
    }
  }

  private void clearAllSignalListeners(SignalService sigService)
  {
    Set<SignalListener> sigListeners = sigService.getSignalListeners();
    for (SignalListener sigListener : sigListeners)
    {
      sigService.removeSignalListener(sigListener);
    }
  }

  private void clearAllMessageListeners(MessageService msgService)
  {
    Set<MessageListener> msgListeners = msgService.getMessageListeners();
    for (MessageListener msgListener : msgListeners)
    {
      msgService.removeMessageListener(msgListener.getKey());
    }
  }

  public SignalListener getSignalListener()
  {
    if (signalListener == null)
    {
      signalListener = new SignalListener()
      {
        public boolean acceptSignal(Signal signal)
        {
          return true;
        }

        public void catchSignal(Signal signal)
        {
          synchronized (signals)
          {
            signals.add(signal);
          }
        }

        public String toString()
        {
          return "SignalListener[" + getShortName() + "]";
        }
      };
    }
    return signalListener;
  }

  public MessageListener getMessageListener()
  {
    if (messageListener == null)
    {
      messageListener = new MessageListener()
      {
        public ObjectName getKey()
        {
          return getTestID();
        }

        public void catchMessage(Message message)
        {
          synchronized (messages)
          {
            log.debug("catchMessage: " + message);
            messages.add(message);
          }
        }
      };
    }
    return messageListener;
  }

  /**
   * Marshall the given process
   * 
   * @param out if null, the proces is marshalled to a file
   */
  protected String marshallProcess(ProcessDefinition procDef, Writer out)
  {
    try
    {
      if (out == null)
      {
        File file = new File("target/" + getName() + "-" + getDialect() + ".xml");
        out = new FileWriter(file);
        System.out.println("Marshall process to: " + file.getCanonicalPath());
      }

      String procXML = marshallProcess(procDef);
      out.write(procXML);
      out.close();

      return procXML;
    }
    catch (IOException ex)
    {
      throw new ProcessEngineException("Cannot marshall process", ex);
    }
  }

  /**
   * Marshall the given process definition
   */
  public String marshallProcess(ProcessDefinition procDef) throws IOException
  {
    DialectHandlerService dhService = getProcessEngine().getService(DialectHandlerService.class);
    DialectHandler dialectHandler = dhService.getDialectHandler(getDialectURI());

    StringWriter strwr = new StringWriter();
    dialectHandler.marshallProcessDefinition(procDef, strwr);
    return strwr.toString();
  }
}
