/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.api.test;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import javax.management.ObjectName;

import junit.framework.TestCase;

import org.jboss.bpm.api.Constants;
import org.jboss.bpm.api.config.Configuration;
import org.jboss.bpm.api.config.ConfigurationProvider;
import org.jboss.bpm.api.model.ObjectNameFactory;
import org.jboss.bpm.api.service.ProcessEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An API test case
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 25-Sep-2008
 */
public class APITestCase extends TestCase
{
  // Provide logging
  final Logger log = LoggerFactory.getLogger(APITestCase.class);

  private APITestHelper delegate = new APITestHelper();
  private static Map<String,ProcessEngine> engineRegistry = new HashMap<String,ProcessEngine>();

  @Override
  protected void setUp() throws Exception
  {
    log.debug("### START " + getLongName());
    super.setUp();
  }

  @Override
  protected void tearDown() throws Exception
  {
    super.tearDown();
    log.debug("### END " + getLongName());
  }

  protected String getShortName()
  {
    String shortName = getClass().getName();
    shortName = shortName.substring(shortName.lastIndexOf(".") + 1);
    return shortName;
  }

  protected String getLongName()
  {
    return getClass().getName() + "." + getName();
  }

  public ObjectName getTestID()
  {
    String shortName = getShortName();
    shortName = shortName.replace("DescriptorTest", "Test");
    shortName = shortName.replace("MarshallerTest", "Test");
    return ObjectNameFactory.create(Constants.ID_DOMAIN, "test", shortName);
  }

  protected String getDialect()
  {
    String propName = "bpm.dialect";
    String dialect = System.getProperty(propName);
    if (dialect == null)
      throw new IllegalStateException("Cannot obtain dialect from system property: " + propName);
    return dialect;
  }

  /**
   * Get the BPM descriptor dialect ID
   */
  protected URI getDialectURI()
  {
    String dialect = System.getProperty("bpm.dialect.uri");
    return URI.create(dialect);
  }
  
  protected ProcessEngine getProcessEngine()
  {
    String dialect = getDialect();
    ProcessEngine engine = engineRegistry.get(dialect); 
    if (engine == null)
    {
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      
      String cfgResource = "bpm-spec-" + dialect + "-beans.xml";
      URL cfgURL = ctxLoader.getResource(cfgResource);
      if (cfgURL == null)
        throw new IllegalStateException("Cannot obtain resource: " + cfgResource);

      ConfigurationProvider provider = Configuration.getConfigurationProvider();
      provider.configure(cfgURL);
      
      engine = provider.getProcessEngine();
      engineRegistry.put(dialect, engine);
    }
    return engine;
  }

  protected URL getResourceURL(String resource)
  {
    return delegate.getResourceURL(resource);
  }

  protected File getResourceFile(String resource)
  {
    return delegate.getResourceFile(resource);
  }

  public File getTestArchiveFile(String archive)
  {
    return delegate.getTestArchiveFile(archive);
  }

  public URL getTestArchiveURL(String archive) throws MalformedURLException
  {
    return delegate.getTestArchiveFile(archive).toURI().toURL();
  }
}
