/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.api.service.internal;

//$Id$

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.jboss.bpm.api.model.ProcessDefinition;
import org.jboss.bpm.api.service.DialectHandler;
import org.jboss.bpm.api.service.DialectHandlerService;
import org.jboss.bpm.api.service.ProcessDefinitionService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * The ProcessDefinitionService is the entry point to create, find and otherwise manage process definitions.
 * 
 * @author thomas.diesler@jboss.com
 * @since 25-Sep-2008
 */
public abstract class AbstractProcessDefinitionService extends AbstractService implements ProcessDefinitionService
{
  // Provide logging
  final static Logger log = LoggerFactory.getLogger(AbstractProcessDefinitionService.class);

  // Hide public constructor
  protected AbstractProcessDefinitionService()
  {
  }

  @Override
  public final ProcessDefinition parseProcessDefinition(String pXML)
  {
    URI nsURI = getNamespaceURI(new ByteArrayInputStream(pXML.getBytes()));
    DialectHandlerService dhService = getProcessEngine().getService(DialectHandlerService.class);
    DialectHandler dialectHandler = dhService.getDialectHandler(nsURI);
    if (dialectHandler == null)
      throw new IllegalStateException("Cannot obtain DialectHandler for: " + nsURI);

    ProcessDefinition procDef = dialectHandler.parseProcessDefinition(pXML);
    return procDef;
  }

  @Override
  public final ProcessDefinition parseProcessDefinition(URL pURL) throws IOException
  {
    URI nsURI = getNamespaceURI(pURL.openStream());
    DialectHandlerService dhService = getProcessEngine().getService(DialectHandlerService.class);
    DialectHandler dialectHandler = dhService.getDialectHandler(nsURI);
    if (dialectHandler == null)
      throw new IllegalStateException("Cannot obtain DialectHandler for: " + nsURI);

    ProcessDefinition procDef = dialectHandler.parseProcessDefinition(pURL);
    return procDef;
  }

  private URI getNamespaceURI(InputStream procXML)
  {
    DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
    dbf.setNamespaceAware(true);
    Document doc;
    try
    {
      DocumentBuilder db = dbf.newDocumentBuilder();
      doc = db.parse(procXML);
    }
    catch (Exception ex)
    {
      throw new IllegalStateException("Cannot parse process descriptor", ex);
    }

    Element root = doc.getDocumentElement();
    String nsURI = root.getNamespaceURI();
    if (nsURI == null)
      throw new IllegalStateException("Cannot get namespace URI from root element");

    return URI.create(nsURI);
  }
}