/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.api.service;

//$Id: ProcessEngine.java 3234 2008-12-05 17:49:30Z thomas.diesler@jboss.com $

import java.util.HashSet;
import java.util.Set;

import org.jboss.bpm.api.config.Configuration;
import org.jboss.bpm.api.config.internal.MicrocontainerConfigurationProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The process engine is an agregator of various service objects used by the BPM engine. 
 * <p/> 
 * A ProcessEngine is always created via one of the available {@link Configuration} objects. Every implementation of the API has one default
 * {@link Configuration} which is configured via a resource file with the fully qualified class name of {@link Configuration}. 
 * <p/> 
 * The API natively supports {@link MicrocontainerConfigurationProvider}. To create and register a ProcessEngine explicitly, you would do
 * 
 * <pre>
 *      MicrocontainerConfiguration config = new MicrocontainerConfiguration();
 *      ProcessEngine engine = config.deployBeansConfig(cfgURL).getProcessEngine("EngineBeanName");
 *      ProcessEngineRegistry.registerProcessEngine(engine);
 * </pre>
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public class ProcessEngine
{
  // Provide logging
  final static Logger log = LoggerFactory.getLogger(ProcessEngine.class);

  // The map of registered services
  protected Set<Service> services = new HashSet<Service>();
  // The name of this engine
  private String name;

  // Hide public constructor
  protected ProcessEngine()
  {
  }

  /**
   * Get the instance of the registered service
   * 
   * @return null if there is no service registered for the given class
   */
  @SuppressWarnings("unchecked")
  public <T> T getService(Class<T> clazz)
  {
    Service service = null;
    for (Service aux : services)
    {
      if (clazz.isAssignableFrom(aux.getClass()))
      {
        service = aux;
        break;
      }
    }
    return (T)service;
  }

  /**
   * Get the name of this engine
   */
  public String getName()
  {
    return name;
  }

  // Set the name of this engine
  protected void setName(String name)
  {
    this.name = name;
  }

  @Override
  public String toString()
  {
    return "ProcessEngine[name=" + name + ",class=" + getClass().getName() + "]";
  }
}