/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software{} you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation{} either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY{} without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software{} if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.api.service;

//$Id: MessageService.java 3484 2008-12-20 14:32:41Z thomas.diesler@jboss.com $

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;

import org.jboss.bpm.api.config.Configuration;
import org.jboss.bpm.api.model.Message;
import org.jboss.bpm.api.model.MessageListener;
import org.jboss.bpm.api.model.Node;
import org.jboss.bpm.api.model.Participant;
import org.jboss.bpm.api.model.ProcessInstance;
import org.jboss.bpm.api.service.internal.AbstractService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The ProcessEngine sends messages through the MessageService. 
 * <p/> 
 * A {@link Message} has an ID and is targeted to a
 * specific Participant. A component can register a {@link MessageListener} with the MessageService.
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public abstract class MessageService extends AbstractService
{
  // Provide logging
  final Logger log = LoggerFactory.getLogger(MessageService.class);

  // The set of registered message listeners
  private Map<ObjectName, MessageListener> listeners = new HashMap<ObjectName, MessageListener>();

  /**
   * Locate the default MessageService
   */
  public static MessageService locateMessageService()
  {
    ProcessEngine engine = Configuration.getProcessEngine();
    return engine.getService(MessageService.class);
  }

  /**
   * Add a MessageListener
   */
  public void addMessageListener(MessageListener listener)
  {
    synchronized (listeners)
    {
      if (hasMessageListener(listener.getKey()))
        throw new IllegalStateException("Listener already registered: " + listener.getKey());

      log.debug("addMessageListener: " + listener.getKey());
      listeners.put(listener.getKey(), listener);
    }
  }

  /**
   * Get the set of registered MessageListeners 
   */
  public Set<MessageListener> getMessageListeners()
  {
    synchronized (listeners)
    {
      HashSet<MessageListener> set = new HashSet<MessageListener>(listeners.values());
      return Collections.unmodifiableSet(set);
    }
  }

  /**
   * Get a MessageListener for a given ID
   * 
   * @return null if there is none
   */
  public MessageListener getMessageListener(ObjectName listenerID)
  {
    synchronized (listeners)
    {
      return listeners.get(listenerID);
    }
  }

  /**
   * True if there is a MessageListener for a given ID
   */
  public boolean hasMessageListener(ObjectName listenerID)
  {
    return getMessageListener(listenerID) != null;
  }

  /**
   * Remove an MessageListener
   */
  public void removeMessageListener(ObjectName listenerID)
  {
    synchronized (listeners)
    {
      log.debug("removeMessageListener: " + listenerID);
      listeners.remove(listenerID);
    }
  }

  /**
   * Send a message to a given {@link Task} or {@link Event}
   */
  public void sendMessage(ObjectName procID, String targetName, Message msg)
  {
    ProcessInstanceService procService = getProcessEngine().getService(ProcessInstanceService.class);
    ProcessInstance proc = procService.getInstance(procID);
    if (proc == null)
      throw new IllegalStateException("Cannot obtain registered process: " + procID);

    Node targetNode = proc.getNode(targetName);
    if (targetNode == null)
      throw new IllegalArgumentException("Cannot find message target: " + targetName);
    if (targetNode instanceof MessageListener == false)
      throw new IllegalArgumentException("Message target is not a valid message receiver: " + targetName);

    log.debug("sendMessage to " + targetNode + " => " + msg);
    MessageListener msgListener = (MessageListener)targetNode;
    msgListener.catchMessage(msg);
  }
  
  /**
   * Send a message to a given {@link MessageListener}
   */
  public void sendMessage(Message msg)
  {
    Participant toRef = msg.getToRef();
    if (toRef == null)
      throw new IllegalArgumentException("Target entity cannot be null");
    
    MessageListener msgListener = getMessageListener(toRef.getName());
    if (msgListener == null)
      throw new IllegalStateException("No message listener registered for: " + toRef);

    log.debug("sendMessage to '" + toRef + "' => " + msg);
    msgListener.catchMessage(msg);
  }
}