/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.model;

import java.util.Collections;
import java.util.List;

/**
 * Update a {@link ServerElement} within a {@link HostModel}
 *
 * @author Emanuel Muckenhuber
 */
public class HostServerUpdate<R> extends AbstractHostModelUpdate<R> {

    private static final long serialVersionUID = 1656392748485415899L;
    private final String serverName;
    private final AbstractModelUpdate<ServerElement, R> serverUpdate;

    public HostServerUpdate(String serverName, AbstractModelUpdate<ServerElement, R> serverUpdate) {
        this.serverName = serverName;
        this.serverUpdate = serverUpdate;
    }

    public static <T> HostServerUpdate<T> create(final String serverName, final AbstractModelUpdate<ServerElement, T> serverUpdate) {
        return new HostServerUpdate<T>(serverName, serverUpdate);
    }

    /** {@inheritDoc} */
    @Override
    protected void applyUpdate(HostModel element) throws UpdateFailedException {
        final ServerElement server = element.getServer(serverName);
        if(server == null) {
            throw new UpdateFailedException(String.format("server (%s) does not exist", serverName));
        }
        serverUpdate.applyUpdate(server);
    }

    /** {@inheritDoc} */
    @Override
    public HostServerUpdate<?> getCompensatingUpdate(HostModel original) {
        final ServerElement server = original.getServer(serverName);
        if(server == null) {
            return null;
        }
        return HostServerUpdate.create(serverName, serverUpdate.getCompensatingUpdate(server));
    }

    /** {@inheritDoc} */
    @Override
    public AbstractServerModelUpdate<R> getServerModelUpdate() {
        return serverUpdate.getServerModelUpdate();
    }

    @Override
    public List<String> getAffectedServers(HostModel hostModel) {
        if (getServerModelUpdate() == null) {
            return Collections.emptyList();
        }
        else {
            return Collections.singletonList(serverName);
        }
    }

}
