/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.persistence.sifs;

import java.io.IOException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.infinispan.commons.marshall.Marshaller;
import org.infinispan.commons.time.TimeService;
import org.infinispan.persistence.sifs.EntryHeader;
import org.infinispan.persistence.sifs.EntryInfo;
import org.infinispan.persistence.sifs.EntryPosition;
import org.infinispan.persistence.sifs.EntryRecord;
import org.infinispan.persistence.sifs.FileProvider;
import org.infinispan.persistence.sifs.Index;
import org.infinispan.persistence.sifs.IndexRequest;
import org.infinispan.persistence.sifs.Log;
import org.infinispan.persistence.sifs.TemporaryTable;
import org.infinispan.util.logging.LogFactory;

class Compactor
extends Thread {
    private static final Log log = (Log)LogFactory.getLog(Compactor.class, Log.class);
    private static final boolean trace = log.isTraceEnabled();
    private final ConcurrentMap<Integer, Stats> fileStats = new ConcurrentHashMap<Integer, Stats>();
    private final BlockingQueue<Integer> scheduledCompaction = new LinkedBlockingQueue<Integer>();
    private final BlockingQueue<IndexRequest> indexQueue;
    private final FileProvider fileProvider;
    private final TemporaryTable temporaryTable;
    private final Marshaller marshaller;
    private final TimeService timeService;
    private final int maxFileSize;
    private final double compactionThreshold;
    private Index index;
    private volatile boolean clearSignal = false;
    private volatile boolean terminateSignal = false;
    private volatile CountDownLatch compactorResume;
    private volatile CountDownLatch compactorStop;

    public Compactor(FileProvider fileProvider, TemporaryTable temporaryTable, BlockingQueue<IndexRequest> indexQueue, Marshaller marshaller, TimeService timeService, int maxFileSize, double compactionThreshold) {
        super("BCS-Compactor");
        this.fileProvider = fileProvider;
        this.temporaryTable = temporaryTable;
        this.indexQueue = indexQueue;
        this.marshaller = marshaller;
        this.timeService = timeService;
        this.maxFileSize = maxFileSize;
        this.compactionThreshold = compactionThreshold;
        this.start();
    }

    public void setIndex(Index index) {
        this.index = index;
    }

    public void releaseStats(int file) {
        this.fileStats.remove(file);
    }

    public void free(int file, int size) {
        if (file < 0) {
            return;
        }
        this.recordFreeSpace(this.getStats(file), file, size);
    }

    public void completeFile(int file) {
        Stats stats = this.getStats(file);
        stats.setCompleted();
        if (stats.readyToBeScheduled(this.compactionThreshold, stats.getFree())) {
            this.schedule(file, stats);
        }
    }

    private Stats getStats(int file) {
        int fileSize;
        Stats stats = (Stats)this.fileStats.get(file);
        if (stats == null) {
            fileSize = (int)this.fileProvider.getFileSize(file);
            stats = new Stats(fileSize, 0);
            Stats other = this.fileStats.putIfAbsent(file, stats);
            if (other != null) {
                if (fileSize > other.getTotal()) {
                    other.setTotal(fileSize);
                }
                return other;
            }
        }
        if (stats.getTotal() < 0 && (fileSize = (int)this.fileProvider.getFileSize(file)) >= 0) {
            stats.setTotal(fileSize);
        }
        return stats;
    }

    private void recordFreeSpace(Stats stats, int file, int size) {
        if (stats.addFree(size, this.compactionThreshold)) {
            this.schedule(file, stats);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void schedule(int file, Stats stats) {
        try {
            Stats stats2 = stats;
            synchronized (stats2) {
                if (!stats.isScheduled()) {
                    log.debug(String.format("Scheduling file %d for compaction: %d/%d free", file, stats.free.get(), stats.total));
                    stats.setScheduled();
                    this.scheduledCompaction.put(file);
                }
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        try {
            FileProvider.Log logFile = null;
            int currentOffset = 0;
            while (true) {
                Integer scheduledFile = null;
                try {
                    scheduledFile = this.scheduledCompaction.poll(1L, TimeUnit.MINUTES);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                if (this.terminateSignal) {
                    if (logFile != null) {
                        logFile.close();
                        this.completeFile(logFile.fileId);
                    }
                    break;
                }
                if (this.clearSignal) {
                    this.pauseCompactor(logFile);
                    logFile = null;
                    continue;
                }
                if (scheduledFile == null) {
                    if (logFile == null) continue;
                    logFile.close();
                    this.completeFile(logFile.fileId);
                    logFile = null;
                    continue;
                }
                log.debugf("Compacting file %d", scheduledFile);
                int scheduledOffset = 0;
                FileProvider.Handle handle = this.fileProvider.getFile(scheduledFile);
                if (handle == null) {
                    throw new IllegalStateException("Compactor should not get deleted file for compaction!");
                }
                try {
                    while (!this.clearSignal && !this.terminateSignal) {
                        EntryHeader header = EntryRecord.readEntryHeader(handle, scheduledOffset);
                        if (header == null) {
                            break;
                        }
                        byte[] serializedKey = EntryRecord.readKey(handle, header, scheduledOffset);
                        if (serializedKey == null) {
                            throw new IllegalStateException("End of file reached when reading key on " + handle.getFileId() + ":" + scheduledOffset);
                        }
                        Object key = this.marshaller.objectFromByteBuffer(serializedKey);
                        int indexedOffset = header.valueLength() > 0 ? scheduledOffset : ~scheduledOffset;
                        boolean drop = true;
                        boolean truncate = false;
                        EntryPosition entry = this.temporaryTable.get(key);
                        if (entry != null) {
                            EntryPosition entryPosition = entry;
                            synchronized (entryPosition) {
                                if (trace) {
                                    log.tracef("Key for %d:%d was found in temporary table on %d:%d", new Object[]{scheduledFile, scheduledOffset, entry.file, entry.offset});
                                }
                                if (entry.file == scheduledFile && entry.offset == indexedOffset) {
                                    if (header.expiryTime() >= 0L && header.expiryTime() <= this.timeService.wallClockTime()) {
                                        truncate = true;
                                    }
                                } else {
                                    truncate = true;
                                }
                            }
                            drop = false;
                        } else {
                            EntryInfo info = this.index.getInfo(key, serializedKey);
                            assert (info != null) : String.format("Index does not recognize entry on %d:%d", new Object[0]);
                            assert (info.numRecords > 0);
                            if (info.file == scheduledFile && info.offset == scheduledOffset) {
                                assert (header.valueLength() > 0);
                                boolean bl = truncate = header.expiryTime() >= 0L && header.expiryTime() <= this.timeService.wallClockTime();
                                if (trace) {
                                    log.tracef("Is %d:%d expired? %s, numRecords? %d", new Object[]{scheduledFile, scheduledOffset, truncate, info.numRecords});
                                }
                                if (!truncate || info.numRecords > 1) {
                                    drop = false;
                                }
                            } else if (info.file == scheduledFile && info.offset == ~scheduledOffset && info.numRecords > 1) {
                                drop = false;
                            } else if (trace) {
                                log.tracef("Key for %d:%d was found in index on %d:%d, %d record => drop", new Object[]{scheduledFile, scheduledOffset, info.file, info.offset, info.numRecords});
                            }
                        }
                        if (drop) {
                            if (trace) {
                                log.tracef("Drop %d:%d (%s)", scheduledFile, scheduledOffset, header.valueLength() > 0 ? "record" : "tombstone");
                            }
                            this.indexQueue.put(IndexRequest.dropped(key, serializedKey, scheduledFile, scheduledOffset));
                        } else {
                            int writtenLength;
                            int entryOffset;
                            if (logFile == null || currentOffset + header.totalLength() > this.maxFileSize) {
                                if (logFile != null) {
                                    logFile.close();
                                    this.completeFile(logFile.fileId);
                                }
                                currentOffset = 0;
                                logFile = this.fileProvider.getFileForLog();
                                log.debugf("Compacting to %d", logFile.fileId);
                            }
                            byte[] serializedValue = null;
                            byte[] serializedMetadata = null;
                            if (header.valueLength() > 0 && !truncate) {
                                if (header.metadataLength() > 0) {
                                    serializedMetadata = EntryRecord.readMetadata(handle, header, scheduledOffset);
                                }
                                serializedValue = EntryRecord.readValue(handle, header, scheduledOffset);
                                entryOffset = currentOffset;
                                writtenLength = header.totalLength();
                            } else {
                                entryOffset = ~currentOffset;
                                writtenLength = 24 + header.keyLength();
                            }
                            EntryRecord.writeEntry(logFile.fileChannel, serializedKey, serializedMetadata, serializedValue, header.seqId(), header.expiryTime());
                            TemporaryTable.LockedEntry lockedEntry = this.temporaryTable.replaceOrLock(key, logFile.fileId, entryOffset, scheduledFile, indexedOffset);
                            if (lockedEntry == null) {
                                if (trace) {
                                    log.trace("Found entry in temporary table");
                                }
                            } else {
                                boolean update = false;
                                try {
                                    EntryInfo info = this.index.getInfo(key, serializedKey);
                                    if (info == null) {
                                        throw new IllegalStateException(String.format("%s was not found in index but it was not in temporary table and there's entry on %d:%d", key, scheduledFile, indexedOffset));
                                    }
                                    boolean bl = update = info.file == scheduledFile && info.offset == indexedOffset;
                                    if (trace) {
                                        log.tracef("In index the key is on %d:%d (%s)", info.file, info.offset, String.valueOf(update));
                                    }
                                }
                                finally {
                                    if (update) {
                                        this.temporaryTable.updateAndUnlock(lockedEntry, logFile.fileId, entryOffset);
                                    } else {
                                        this.temporaryTable.removeAndUnlock(lockedEntry, key);
                                    }
                                }
                            }
                            if (trace) {
                                log.tracef("Update %d:%d -> %d:%d | %d,%d", new Object[]{scheduledFile, indexedOffset, logFile.fileId, entryOffset, logFile.fileChannel.position(), logFile.fileChannel.size()});
                            }
                            this.indexQueue.put(IndexRequest.moved(key, serializedKey, logFile.fileId, entryOffset, writtenLength, scheduledFile, indexedOffset));
                            currentOffset += writtenLength;
                        }
                        scheduledOffset += header.totalLength();
                    }
                }
                finally {
                    handle.close();
                }
                if (this.terminateSignal) {
                    if (logFile != null) {
                        logFile.close();
                        this.completeFile(logFile.fileId);
                    }
                    return;
                }
                if (this.clearSignal) {
                    this.pauseCompactor(logFile);
                    logFile = null;
                    continue;
                }
                log.debugf("Finished compacting %d, scheduling delete", scheduledFile);
                this.indexQueue.put(IndexRequest.deleteFileRequest(scheduledFile));
            }
        }
        catch (Exception e) {
            log.compactorFailed(e);
        }
    }

    private void pauseCompactor(FileProvider.Log logFile) throws IOException, InterruptedException {
        if (logFile != null) {
            logFile.close();
            this.completeFile(logFile.fileId);
        }
        this.compactorStop.countDown();
        this.compactorResume.await();
    }

    public void clearAndPause() throws InterruptedException {
        this.compactorResume = new CountDownLatch(1);
        this.compactorStop = new CountDownLatch(1);
        this.clearSignal = true;
        this.scheduledCompaction.put(-1);
        this.compactorStop.await();
        this.scheduledCompaction.clear();
        this.fileStats.clear();
    }

    public void resumeAfterPause() {
        this.clearSignal = false;
        this.compactorResume.countDown();
    }

    public void stopOperations() throws InterruptedException {
        this.terminateSignal = true;
        this.scheduledCompaction.put(-1);
        this.join();
    }

    private static class Stats {
        private final AtomicInteger free;
        private volatile int total;
        private volatile boolean completed = false;
        private volatile boolean scheduled = false;

        private Stats(int total, int free) {
            this.free = new AtomicInteger(free);
            this.total = total;
        }

        public int getTotal() {
            return this.total;
        }

        public void setTotal(int total) {
            this.total = total;
        }

        public boolean addFree(int size, double compactionThreshold) {
            int free = this.free.addAndGet(size);
            return this.readyToBeScheduled(compactionThreshold, free);
        }

        public int getFree() {
            return this.free.get();
        }

        public boolean readyToBeScheduled(double compactionThreshold, int free) {
            int total = this.total;
            return this.completed && !this.scheduled && total >= 0 && (double)free > (double)total * compactionThreshold;
        }

        public boolean isScheduled() {
            return this.scheduled;
        }

        public void setScheduled() {
            this.scheduled = true;
        }

        public boolean isCompleted() {
            return this.completed;
        }

        public void setCompleted() {
            this.completed = true;
        }
    }
}

