package org.immutables.value.processor.meta;

import com.google.common.primitives.Booleans;
import java.lang.annotation.Annotation;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
/**
 * TypeAdaptersMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.gson.Gson.TypeAdapters}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.GsonMirrors.TypeAdapters} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class TypeAdaptersMirror implements GsonMirrors.TypeAdapters {
  public static final String QUALIFIED_NAME = "org.immutables.gson.Gson.TypeAdapters";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.GsonMirrors.TypeAdapters";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "TypeAdapters";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code TypeAdaptersMirror}, present if this annotation found
   */
  public static Optional<TypeAdaptersMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code TypeAdaptersMirror}, present if this annotation found
   */
  public static Optional<TypeAdaptersMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new TypeAdaptersMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code TypeAdaptersMirror}s
   */
  public static ImmutableList<TypeAdaptersMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<TypeAdaptersMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new TypeAdaptersMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code TypeAdaptersMirror}
   */
  public static TypeAdaptersMirror from(TypeElement element) {
    return new TypeAdaptersMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code TypeAdaptersMirror}, present if mirror matched this annotation type
   */
  public static Optional<TypeAdaptersMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final boolean metainfService;
  private final boolean fieldNamingStrategy;
  private final boolean emptyAsNulls;
  private final boolean nullAsDefault;

  private TypeAdaptersMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    boolean metainfService = false;
    boolean fieldNamingStrategy = false;
    boolean emptyAsNulls = false;
    boolean nullAsDefault = false;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("metainfService".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @TypeAdapters");
        }
        MetainfServiceExtractor metainfServiceExtractor$ = new MetainfServiceExtractor();
        annotationValue$.accept(metainfServiceExtractor$, null);

        metainfService = metainfServiceExtractor$.get();
        continue;
      }
      if ("fieldNamingStrategy".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @TypeAdapters");
        }
        FieldNamingStrategyExtractor fieldNamingStrategyExtractor$ = new FieldNamingStrategyExtractor();
        annotationValue$.accept(fieldNamingStrategyExtractor$, null);

        fieldNamingStrategy = fieldNamingStrategyExtractor$.get();
        continue;
      }
      if ("emptyAsNulls".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @TypeAdapters");
        }
        EmptyAsNullsExtractor emptyAsNullsExtractor$ = new EmptyAsNullsExtractor();
        annotationValue$.accept(emptyAsNullsExtractor$, null);

        emptyAsNulls = emptyAsNullsExtractor$.get();
        continue;
      }
      if ("nullAsDefault".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @TypeAdapters");
        }
        NullAsDefaultExtractor nullAsDefaultExtractor$ = new NullAsDefaultExtractor();
        annotationValue$.accept(nullAsDefaultExtractor$, null);

        nullAsDefault = nullAsDefaultExtractor$.get();
        continue;
      }
    }
    this.metainfService = metainfService;
    this.fieldNamingStrategy = fieldNamingStrategy;
    this.emptyAsNulls = emptyAsNulls;
    this.nullAsDefault = nullAsDefault;
  }

  private TypeAdaptersMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    boolean metainfService = false;
    boolean fieldNamingStrategy = false;
    boolean emptyAsNulls = false;
    boolean nullAsDefault = false;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("metainfService".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'metainfService' attribute of @TypeAdapters");
        }
        MetainfServiceExtractor metainfServiceExtractor$ = new MetainfServiceExtractor();
        annotationValue$.accept(metainfServiceExtractor$, null);

        metainfService = metainfServiceExtractor$.get();
        continue;
      }
      if ("fieldNamingStrategy".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'fieldNamingStrategy' attribute of @TypeAdapters");
        }
        FieldNamingStrategyExtractor fieldNamingStrategyExtractor$ = new FieldNamingStrategyExtractor();
        annotationValue$.accept(fieldNamingStrategyExtractor$, null);

        fieldNamingStrategy = fieldNamingStrategyExtractor$.get();
        continue;
      }
      if ("emptyAsNulls".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'emptyAsNulls' attribute of @TypeAdapters");
        }
        EmptyAsNullsExtractor emptyAsNullsExtractor$ = new EmptyAsNullsExtractor();
        annotationValue$.accept(emptyAsNullsExtractor$, null);

        emptyAsNulls = emptyAsNullsExtractor$.get();
        continue;
      }
      if ("nullAsDefault".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'nullAsDefault' attribute of @TypeAdapters");
        }
        NullAsDefaultExtractor nullAsDefaultExtractor$ = new NullAsDefaultExtractor();
        annotationValue$.accept(nullAsDefaultExtractor$, null);

        nullAsDefault = nullAsDefaultExtractor$.get();
        continue;
      }
    }
    this.metainfService = metainfService;
    this.fieldNamingStrategy = fieldNamingStrategy;
    this.emptyAsNulls = emptyAsNulls;
    this.nullAsDefault = nullAsDefault;
  }

  /**
   * @return value of attribute {@code metainfService}
   */
  @Override
  public boolean metainfService() {
    return metainfService;
  }

  /**
   * @return value of attribute {@code fieldNamingStrategy}
   */
  @Override
  public boolean fieldNamingStrategy() {
    return fieldNamingStrategy;
  }

  /**
   * @return value of attribute {@code emptyAsNulls}
   */
  @Override
  public boolean emptyAsNulls() {
    return emptyAsNulls;
  }

  /**
   * @return value of attribute {@code nullAsDefault}
   */
  @Override
  public boolean nullAsDefault() {
    return nullAsDefault;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code TypeAdapters.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return GsonMirrors.TypeAdapters.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "metainfService".hashCode() ^ Booleans.hashCode(metainfService);
    h += 127 * "fieldNamingStrategy".hashCode() ^ Booleans.hashCode(fieldNamingStrategy);
    h += 127 * "emptyAsNulls".hashCode() ^ Booleans.hashCode(emptyAsNulls);
    h += 127 * "nullAsDefault".hashCode() ^ Booleans.hashCode(nullAsDefault);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof TypeAdaptersMirror) {
      TypeAdaptersMirror otherMirror = (TypeAdaptersMirror) other;
      return metainfService == otherMirror.metainfService
          && fieldNamingStrategy == otherMirror.fieldNamingStrategy
          && emptyAsNulls == otherMirror.emptyAsNulls
          && nullAsDefault == otherMirror.nullAsDefault;
    }
    return false;
  }

  @Override
  public String toString() {
    return "TypeAdaptersMirror:" + annotationMirror;
  }

  private static class MetainfServiceExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'metainfService' in @" + QUALIFIED_NAME);
    }
  }

  private static class FieldNamingStrategyExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'fieldNamingStrategy' in @" + QUALIFIED_NAME);
    }
  }

  private static class EmptyAsNullsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'emptyAsNulls' in @" + QUALIFIED_NAME);
    }
  }

  private static class NullAsDefaultExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'nullAsDefault' in @" + QUALIFIED_NAME);
    }
  }
}
