package org.immutables.value.processor.meta;

import com.google.common.primitives.Booleans;
import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
/**
 * InjectAnnotationMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.annotate.InjectAnnotation}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.AnnotationInjections.InjectAnnotation} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class InjectAnnotationMirror implements AnnotationInjections.InjectAnnotation {
  public static final String QUALIFIED_NAME = "org.immutables.annotate.InjectAnnotation";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.AnnotationInjections.InjectAnnotation";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "InjectAnnotation";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code InjectAnnotationMirror}, present if this annotation found
   */
  public static Optional<InjectAnnotationMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code InjectAnnotationMirror}, present if this annotation found
   */
  public static Optional<InjectAnnotationMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new InjectAnnotationMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code InjectAnnotationMirror}s
   */
  public static ImmutableList<InjectAnnotationMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<InjectAnnotationMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new InjectAnnotationMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code InjectAnnotationMirror}
   */
  public static InjectAnnotationMirror from(TypeElement element) {
    return new InjectAnnotationMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code InjectAnnotationMirror}, present if mirror matched this annotation type
   */
  public static Optional<InjectAnnotationMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final String code;
  private final TypeMirror type;
  private final String typeName;
  private final boolean ifPresent;
  private final AnnotationInjections.InjectAnnotation.Where[] target;
  private final String deduplicationKey;

  private InjectAnnotationMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    String code = null;
    TypeMirror type = null;
    String typeName = null;
    boolean ifPresent = false;
    AnnotationInjections.InjectAnnotation.Where[] target = null;
    String deduplicationKey = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("code".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @InjectAnnotation");
        }
        CodeExtractor codeExtractor$ = new CodeExtractor();
        annotationValue$.accept(codeExtractor$, null);

        code = codeExtractor$.get();
        continue;
      }
      if ("type".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @InjectAnnotation");
        }
        TypeExtractor typeExtractor$ = new TypeExtractor();
        annotationValue$.accept(typeExtractor$, null);

        type = typeExtractor$.get();
        typeName = typeExtractor$.name();
        continue;
      }
      if ("ifPresent".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @InjectAnnotation");
        }
        IfPresentExtractor ifPresentExtractor$ = new IfPresentExtractor();
        annotationValue$.accept(ifPresentExtractor$, null);

        ifPresent = ifPresentExtractor$.get();
        continue;
      }
      if ("target".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @InjectAnnotation");
        }
        TargetExtractor targetExtractor$ = new TargetExtractor();
        annotationValue$.accept(targetExtractor$, null);

        target = targetExtractor$.get();
        continue;
      }
      if ("deduplicationKey".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @InjectAnnotation");
        }
        DeduplicationKeyExtractor deduplicationKeyExtractor$ = new DeduplicationKeyExtractor();
        annotationValue$.accept(deduplicationKeyExtractor$, null);

        deduplicationKey = deduplicationKeyExtractor$.get();
        continue;
      }
    }
    this.code = Preconditions.checkNotNull(code, "default attribute 'code'");
    this.type = Preconditions.checkNotNull(type, "default attribute 'type'");
    this.typeName = Preconditions.checkNotNull(typeName, "default attribute 'type'");
    this.ifPresent = ifPresent;
    this.target = Preconditions.checkNotNull(target, "default attribute 'target'");
    this.deduplicationKey = Preconditions.checkNotNull(deduplicationKey, "default attribute 'deduplicationKey'");
  }

  private InjectAnnotationMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    String code = null;
    TypeMirror type = null;
    String typeName = null;
    boolean ifPresent = false;
    AnnotationInjections.InjectAnnotation.Where[] target = null;
    String deduplicationKey = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("code".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'code' attribute of @InjectAnnotation");
        }
        CodeExtractor codeExtractor$ = new CodeExtractor();
        annotationValue$.accept(codeExtractor$, null);

        code = codeExtractor$.get();
        continue;
      }
      if ("type".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'type' attribute of @InjectAnnotation");
        }
        TypeExtractor typeExtractor$ = new TypeExtractor();
        annotationValue$.accept(typeExtractor$, null);

        type = typeExtractor$.get();
        typeName = typeExtractor$.name();
        continue;
      }
      if ("ifPresent".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'ifPresent' attribute of @InjectAnnotation");
        }
        IfPresentExtractor ifPresentExtractor$ = new IfPresentExtractor();
        annotationValue$.accept(ifPresentExtractor$, null);

        ifPresent = ifPresentExtractor$.get();
        continue;
      }
      if ("target".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'target' attribute of @InjectAnnotation");
        }
        TargetExtractor targetExtractor$ = new TargetExtractor();
        annotationValue$.accept(targetExtractor$, null);

        target = targetExtractor$.get();
        continue;
      }
      if ("deduplicationKey".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'deduplicationKey' attribute of @InjectAnnotation");
        }
        DeduplicationKeyExtractor deduplicationKeyExtractor$ = new DeduplicationKeyExtractor();
        annotationValue$.accept(deduplicationKeyExtractor$, null);

        deduplicationKey = deduplicationKeyExtractor$.get();
        continue;
      }
    }
    this.code = Preconditions.checkNotNull(code, "value for 'code'");
    this.type = Preconditions.checkNotNull(type, "value for 'type'");
    this.typeName = Preconditions.checkNotNull(typeName, "Value for 'type'");
    this.ifPresent = ifPresent;
    this.target = Preconditions.checkNotNull(target, "value for 'target'");
    this.deduplicationKey = Preconditions.checkNotNull(deduplicationKey, "value for 'deduplicationKey'");
  }

  /**
   * @return value of attribute {@code code}
   */
  @Override
  public String code() {
    return code;
  }

  /**
   * @return type name for value of attribute {@code type}
   */
  public String typeName() {
    return typeName;
  }

  /**
   * @return type mirror for value of attribute {@code type}
   */
  public TypeMirror typeMirror() {
    return type;
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #typeMirror} or {@link typeName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<? extends Annotation> type() {
    throw new UnsupportedOperationException("Use 'typeMirror()' or 'typeName()'");
  }

  /**
   * @return value of attribute {@code ifPresent}
   */
  @Override
  public boolean ifPresent() {
    return ifPresent;
  }

  /**
   * @return value of attribute {@code target}
   */
  @Override
  public AnnotationInjections.InjectAnnotation.Where[] target() {
    return target.clone();
  }

  /**
   * @return value of attribute {@code deduplicationKey}
   */
  @Override
  public String deduplicationKey() {
    return deduplicationKey;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code InjectAnnotation.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return AnnotationInjections.InjectAnnotation.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "code".hashCode() ^ code.hashCode();
    h += 127 * "type".hashCode() ^ typeName.hashCode();
    h += 127 * "ifPresent".hashCode() ^ Booleans.hashCode(ifPresent);
    h += 127 * "target".hashCode() ^ Arrays.hashCode(target);
    h += 127 * "deduplicationKey".hashCode() ^ deduplicationKey.hashCode();
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof InjectAnnotationMirror) {
      InjectAnnotationMirror otherMirror = (InjectAnnotationMirror) other;
      return code.equals(otherMirror.code)
          && typeName.equals(otherMirror.typeName)
          && ifPresent == otherMirror.ifPresent
          && Arrays.equals(target, otherMirror.target)
          && deduplicationKey.equals(otherMirror.deduplicationKey);
    }
    return false;
  }

  @Override
  public String toString() {
    return "InjectAnnotationMirror:" + annotationMirror;
  }

  private static class CodeExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'code' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror value;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    TypeMirror get() {
      return value;
    }

    public String name() {
      return value.toString();
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'type' in @" + QUALIFIED_NAME);
    }
  }

  private static class IfPresentExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'ifPresent' in @" + QUALIFIED_NAME);
    }
  }

  private static class TargetExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    AnnotationInjections.InjectAnnotation.Where[] values;
    int position;

    @Override
    public Void visitEnumConstant(VariableElement value, Void p) {
      this.values[position++] = AnnotationInjections.InjectAnnotation.Where.valueOf(value.getSimpleName().toString());
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new AnnotationInjections.InjectAnnotation.Where[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    AnnotationInjections.InjectAnnotation.Where[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'target' in @" + QUALIFIED_NAME);
    }
  }

  private static class DeduplicationKeyExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'deduplicationKey' in @" + QUALIFIED_NAME);
    }
  }
}
