package org.immutables.value.processor.meta;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
/**
 * DepluralizeMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.value.Value.Style.Depluralize}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.ValueMirrors.Depluralize} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class DepluralizeMirror implements ValueMirrors.Depluralize {
  public static final String QUALIFIED_NAME = "org.immutables.value.Value.Style.Depluralize";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.ValueMirrors.Depluralize";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Depluralize";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code DepluralizeMirror}, present if this annotation found
   */
  public static Optional<DepluralizeMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code DepluralizeMirror}, present if this annotation found
   */
  public static Optional<DepluralizeMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new DepluralizeMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code DepluralizeMirror}s
   */
  public static ImmutableList<DepluralizeMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<DepluralizeMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new DepluralizeMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code DepluralizeMirror}
   */
  public static DepluralizeMirror from(TypeElement element) {
    return new DepluralizeMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code DepluralizeMirror}, present if mirror matched this annotation type
   */
  public static Optional<DepluralizeMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final String[] dictionary;

  private DepluralizeMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    String[] dictionary = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("dictionary".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Depluralize");
        }
        DictionaryExtractor dictionaryExtractor$ = new DictionaryExtractor();
        annotationValue$.accept(dictionaryExtractor$, null);

        dictionary = dictionaryExtractor$.get();
        continue;
      }
    }
    this.dictionary = Preconditions.checkNotNull(dictionary, "default attribute 'dictionary'");
  }

  private DepluralizeMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    String[] dictionary = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("dictionary".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'dictionary' attribute of @Depluralize");
        }
        DictionaryExtractor dictionaryExtractor$ = new DictionaryExtractor();
        annotationValue$.accept(dictionaryExtractor$, null);

        dictionary = dictionaryExtractor$.get();
        continue;
      }
    }
    this.dictionary = Preconditions.checkNotNull(dictionary, "value for 'dictionary'");
  }

  /**
   * @return value of attribute {@code dictionary}
   */
  @Override
  public String[] dictionary() {
    return dictionary.clone();
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code Depluralize.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return ValueMirrors.Depluralize.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "dictionary".hashCode() ^ Arrays.hashCode(dictionary);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof DepluralizeMirror) {
      DepluralizeMirror otherMirror = (DepluralizeMirror) other;
      return Arrays.equals(dictionary, otherMirror.dictionary);
    }
    return false;
  }

  @Override
  public String toString() {
    return "DepluralizeMirror:" + annotationMirror;
  }

  private static class DictionaryExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'dictionary' in @" + QUALIFIED_NAME);
    }
  }
}
