package org.immutables.value.processor.meta;

import com.google.common.primitives.Floats;
import java.lang.annotation.Annotation;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
/**
 * DefaultFloatMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.value.Value.Default.Float}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.ValueMirrors.DefaultFloat} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class DefaultFloatMirror implements ValueMirrors.DefaultFloat {
  public static final String QUALIFIED_NAME = "org.immutables.value.Value.Default.Float";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.ValueMirrors.DefaultFloat";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Float";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code DefaultFloatMirror}, present if this annotation found
   */
  public static Optional<DefaultFloatMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code DefaultFloatMirror}, present if this annotation found
   */
  public static Optional<DefaultFloatMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new DefaultFloatMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code DefaultFloatMirror}s
   */
  public static ImmutableList<DefaultFloatMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<DefaultFloatMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new DefaultFloatMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code DefaultFloatMirror}
   */
  public static DefaultFloatMirror from(TypeElement element) {
    return new DefaultFloatMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code DefaultFloatMirror}, present if mirror matched this annotation type
   */
  public static Optional<DefaultFloatMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final float value;

  private DefaultFloatMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    float value = 0;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Float");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        continue;
      }
    }
    this.value = value;
  }

  private DefaultFloatMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    float value = 0;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'value' attribute of @Float");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        continue;
      }
    }
    this.value = value;
  }

  /**
   * @return value of attribute {@code value}
   */
  @Override
  public float value() {
    return value;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code DefaultFloat.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return ValueMirrors.DefaultFloat.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "value".hashCode() ^ Floats.hashCode(value);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof DefaultFloatMirror) {
      DefaultFloatMirror otherMirror = (DefaultFloatMirror) other;
      return Float.floatToIntBits(value) == Float.floatToIntBits(otherMirror.value);
    }
    return false;
  }

  @Override
  public String toString() {
    return "DefaultFloatMirror:" + annotationMirror;
  }

  private static class ValueExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    float value;

    @Override
    public Void visitFloat(float value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    float get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'value' in @" + QUALIFIED_NAME);
    }
  }
}
