package org.immutables.fixture.style;

import com.google.common.base.MoreObjects;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Immutable implementation of {@link LightOnAnnotations}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableLightOnAnnotations.builder()}.
 */
public final class ImmutableLightOnAnnotations<X>
    implements LightOnAnnotations<X> {
  private final int a;
  private final X x;

  private ImmutableLightOnAnnotations(int a, X x) {
    this.a = a;
    this.x = x;
  }

  /**
   * @return The value of the {@code a} attribute
   */
  @Override
  public int a() {
    return a;
  }

  /**
   * @return The value of the {@code x} attribute
   */
  @Override
  public Optional<X> x() {
    return Optional.ofNullable(x);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link LightOnAnnotations#a() a} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for a
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableLightOnAnnotations<X> withA(int value) {
    if (this.a == value) return this;
    return new ImmutableLightOnAnnotations<>(value, this.x);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link LightOnAnnotations#x() x} attribute.
   * @param value The value for x
   * @return A modified copy of {@code this} object
   */
  public final ImmutableLightOnAnnotations<X> withX(X value) {
    X newValue = Objects.requireNonNull(value, "x");
    if (this.x == newValue) return this;
    return new ImmutableLightOnAnnotations<>(this.a, newValue);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link LightOnAnnotations#x() x} attribute.
   * A shallow reference equality check is used on unboxed optional value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for x
   * @return A modified copy of {@code this} object
   */
  @SuppressWarnings("unchecked") // safe covariant cast
  public final ImmutableLightOnAnnotations<X> withX(Optional<? extends X> optional) {
    X value = optional.orElse(null);
    if (this.x == value) return this;
    return new ImmutableLightOnAnnotations<>(this.a, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableLightOnAnnotations} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableLightOnAnnotations<?>
        && equalTo(0, (ImmutableLightOnAnnotations<?>) another);
  }

  private boolean equalTo(int synthetic, ImmutableLightOnAnnotations<?> another) {
    return a == another.a
        && Objects.equals(x, another.x);
  }

  /**
   * Computes a hash code from attributes: {@code a}, {@code x}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + a;
    h += (h << 5) + Objects.hashCode(x);
    return h;
  }

  /**
   * Prints the immutable value {@code LightOnAnnotations} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return MoreObjects.toStringHelper("LightOnAnnotations")
        .omitNullValues()
        .add("a", a)
        .add("x", x)
        .toString();
  }

  /**
   * Creates an immutable copy of a {@link LightOnAnnotations} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param <X> generic parameter X
   * @param instance The instance to copy
   * @return A copied immutable LightOnAnnotations instance
   */
  public static <X> ImmutableLightOnAnnotations<X> copyOf(LightOnAnnotations<X> instance) {
    if (instance instanceof ImmutableLightOnAnnotations<?>) {
      return (ImmutableLightOnAnnotations<X>) instance;
    }
    return ImmutableLightOnAnnotations.<X>builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableLightOnAnnotations ImmutableLightOnAnnotations}.
   * <pre>
   * ImmutableLightOnAnnotations.&amp;lt;X&amp;gt;builder()
   *    .a(int) // required {@link LightOnAnnotations#a() a}
   *    .x(X) // optional {@link LightOnAnnotations#x() x}
   *    .build();
   * </pre>
   * @param <X> generic parameter X
   * @return A new ImmutableLightOnAnnotations builder
   */
  public static <X> ImmutableLightOnAnnotations.Builder<X> builder() {
    return new ImmutableLightOnAnnotations.Builder<>();
  }

  /**
   * Builds instances of type {@link ImmutableLightOnAnnotations ImmutableLightOnAnnotations}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder<X> {
    private static final long INIT_BIT_A = 0x1L;
    private long initBits = 0x1L;

    private int a;
    private X x;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ModifiableLightOnAnnotations} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<X> from(ModifiableLightOnAnnotations<X> instance) {
      Objects.requireNonNull(instance, "instance");
      if (instance.aIsSet()) {
        a(instance.a());
      }
      Optional<X> xOptional = instance.x();
      if (xOptional.isPresent()) {
        x(xOptional);
      }
      return this;
    }

    /**
     * Fill a builder with attribute values from the provided {@code LightOnAnnotations} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<X> from(LightOnAnnotations<X> instance) {
      Objects.requireNonNull(instance, "instance");
      if (instance instanceof ModifiableLightOnAnnotations<?>) {
        return from((ModifiableLightOnAnnotations<X>) instance);
      }
      a(instance.a());
      Optional<X> xOptional = instance.x();
      if (xOptional.isPresent()) {
        x(xOptional);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link LightOnAnnotations#a() a} attribute.
     * @param a The value for a 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<X> a(int a) {
      this.a = a;
      initBits &= ~INIT_BIT_A;
      return this;
    }

    /**
     * Initializes the optional value {@link LightOnAnnotations#x() x} to x.
     * @param x The value for x
     * @return {@code this} builder for chained invocation
     */
    public final Builder<X> x(X x) {
      this.x = Objects.requireNonNull(x, "x");
      return this;
    }

    /**
     * Initializes the optional value {@link LightOnAnnotations#x() x} to x.
     * @param x The value for x
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<X> x(Optional<? extends X> x) {
      this.x = x.orElse(null);
      return this;
    }

    /**
     * Builds a new {@link ImmutableLightOnAnnotations ImmutableLightOnAnnotations}.
     * @return An immutable instance of LightOnAnnotations
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableLightOnAnnotations<X> build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableLightOnAnnotations<>(a, x);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_A) != 0) attributes.add("a");
      return "Cannot build LightOnAnnotations, some of required attributes are not set " + attributes;
    }
  }
}
