/*
 * SPDX-License-Identifier: Apache-2.0
 * Copyright Red Hat Inc. and Hibernate Authors
 */
package org.hibernate.search.engine.search.aggregation.dsl;

import java.util.function.Function;

import org.hibernate.search.engine.search.predicate.dsl.SearchPredicateFactory;
import org.hibernate.search.engine.search.reference.aggregation.CountAggregationFieldReference;
import org.hibernate.search.util.common.annotation.Incubating;

/**
 * The initial step in a "count distinct" aggregation definition, where the target field can be set.
 *
 * @param <SR> Scope root type.
 * @param <PDF> The type of factory used to create predicates in {@link AggregationFilterStep#filter(Function)}.
 */
@Incubating
public interface CountDistinctAggregationFieldStep<SR, PDF extends SearchPredicateFactory<SR>> {

	/**
	 * Target the given field in the count distinct aggregation.
	 *
	 * @param fieldPath The <a href="SearchAggregationFactory.html#field-paths">path</a> to the index field to aggregate.
	 * @return The next step.
	 */
	CountDistinctAggregationOptionsStep<SR, ?, PDF> field(String fieldPath);

	/**
	 * Target the given field in the avg aggregation.
	 *
	 * @param fieldReference The field reference representing a <a href="SearchAggregationFactory.html#field-references">definition</a> of the index field to aggregate.
	 * @return The next step.
	 */
	@Incubating
	default CountDistinctAggregationOptionsStep<SR, ?, PDF> field(CountAggregationFieldReference<SR> fieldReference) {
		return field( fieldReference.absolutePath() );
	}
}
