/*
 * Copyright (c) 2018, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.graal.python.nodes.call.special;

import com.oracle.graal.python.builtins.objects.function.PKeyword;
import com.oracle.graal.python.nodes.object.GetClassNode;
import com.oracle.truffle.api.HostCompilerDirectives.InliningCutoff;
import com.oracle.truffle.api.dsl.Bind;
import com.oracle.truffle.api.dsl.Cached;
import com.oracle.truffle.api.dsl.Cached.Shared;
import com.oracle.truffle.api.dsl.NeverDefault;
import com.oracle.truffle.api.dsl.ReportPolymorphism.Megamorphic;
import com.oracle.truffle.api.dsl.Specialization;
import com.oracle.truffle.api.frame.VirtualFrame;
import com.oracle.truffle.api.nodes.Node;
import com.oracle.truffle.api.strings.TruffleString;

public abstract class LookupAndCallVarargsNode extends Node {
    protected final TruffleString name;

    public abstract Object execute(VirtualFrame frame, Object callable, Object[] arguments);

    @NeverDefault
    public static LookupAndCallVarargsNode create(TruffleString name) {
        return LookupAndCallVarargsNodeGen.create(name);
    }

    LookupAndCallVarargsNode(TruffleString name) {
        this.name = name;
    }

    @Specialization(guards = {"callable.getClass() == cachedClass"}, limit = "3")
    static Object callObject(VirtualFrame frame, Object callable, Object[] arguments,
                    @Bind("this") Node inliningTarget,
                    @SuppressWarnings("unused") @Cached("callable.getClass()") Class<?> cachedClass,
                    @Shared @Cached GetClassNode getClassNode,
                    @Shared @Cached("create(name)") LookupSpecialMethodNode getattr,
                    @Shared @Cached CallVarargsMethodNode dispatchNode) {
        return dispatchNode.execute(frame, getattr.execute(frame, getClassNode.execute(inliningTarget, callable), callable), arguments, PKeyword.EMPTY_KEYWORDS);
    }

    @Specialization(replaces = "callObject")
    @InliningCutoff
    @Megamorphic
    static Object callObjectMegamorphic(VirtualFrame frame, Object callable, Object[] arguments,
                    @Bind("this") Node inliningTarget,
                    @Shared @Cached GetClassNode getClassNode,
                    @Shared @Cached("create(name)") LookupSpecialMethodNode getattr,
                    @Shared @Cached CallVarargsMethodNode dispatchNode) {
        return dispatchNode.execute(frame, getattr.execute(frame, getClassNode.execute(inliningTarget, callable), callable), arguments, PKeyword.EMPTY_KEYWORDS);
    }
}
