/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.python.embedding.tools.vfs;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.invoke.CallSite;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Consumer;
import java.util.stream.Stream;
import org.graalvm.python.embedding.tools.exec.BuildToolLog;
import org.graalvm.python.embedding.tools.exec.GraalPyRunner;
import org.graalvm.python.embedding.tools.vfs.SuppressFBWarnings;

public final class VFSUtils {
    private static final String[] DEFAULT_EXCLUDES = new String[]{"**/*~", "**/#*#", "**/.#*", "**/%*%", "**/._*", "**/CVS", "**/CVS/**", "**/.cvsignore", "**/RCS", "**/RCS/**", "**/SCCS", "**/SCCS/**", "**/vssver.scc", "**/project.pj", "**/.svn", "**/.svn/**", "**/.arch-ids", "**/.arch-ids/**", "**/.bzr", "**/.bzr/**", "**/.MySCMServerInfo", "**/.DS_Store", "**/.metadata", "**/.metadata/**", "**/.hg", "**/.hg/**", "**/.git", "**/.git/**", "**/.gitignore", "**/BitKeeper", "**/BitKeeper/**", "**/ChangeSet", "**/ChangeSet/**", "**/_darcs", "**/_darcs/**", "**/.darcsrepo", "**/.darcsrepo/**", "**/-darcs-backup*", "**/.darcs-temp-mail"};
    public static final String VFS_ROOT = "org.graalvm.python.vfs";
    public static final String VFS_VENV = "venv";
    public static final String VFS_FILESLIST = "fileslist.txt";
    public static final String GRAALPY_GROUP_ID = "org.graalvm.python";
    private static final String PLATFORM = System.getProperty("os.name") + " " + System.getProperty("os.arch");
    private static final String NATIVE_IMAGE_RESOURCES_CONFIG = "{\n  \"resources\": {\n    \"includes\": [\n      {\"pattern\": \"$vfs/.*\"}\n    ]\n  }\n}\n";
    private static final boolean IS_WINDOWS = System.getProperty("os.name").startsWith("Windows");
    public static final String LAUNCHER_NAME = IS_WINDOWS ? "graalpy.exe" : "graalpy.sh";
    private static final String GRAALPY_MAIN_CLASS = "com.oracle.graal.python.shell.GraalPythonMain";
    private static final boolean REPLACE_BACKSLASHES = File.separatorChar == '\\';
    private static final String GRAALPY_VERSION_PREFIX = "# graalpy-version: ";
    private static final String INPUT_PACKAGES_PREFIX = "# input-packages: ";
    private static final String INPUT_PACKAGES_DELIMITER = ",";

    public static void writeNativeImageConfig(Path metaInfRoot, String pluginId) throws IOException {
        VFSUtils.writeNativeImageConfig(metaInfRoot, pluginId, VFS_ROOT);
    }

    public static void writeNativeImageConfig(Path metaInfRoot, String pluginId, String vfsRoot) throws IOException {
        Path p = metaInfRoot.resolve(Path.of("native-image", GRAALPY_GROUP_ID, pluginId));
        VFSUtils.write(p.resolve("resource-config.json"), NATIVE_IMAGE_RESOURCES_CONFIG.replace("$vfs", vfsRoot));
    }

    private static void write(Path config, String txt) throws IOException {
        try {
            VFSUtils.createParentDirectories(config);
            Files.writeString(config, (CharSequence)txt, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        }
        catch (IOException e) {
            throw new IOException(String.format("failed to write %s", config), e);
        }
    }

    private static void createParentDirectories(Path path) throws IOException {
        Path parent = path.getParent();
        if (parent != null) {
            Files.createDirectories(parent, new FileAttribute[0]);
        }
    }

    public static void generateVFSFilesList(Path resourcesRoot, Path vfs) throws IOException {
        TreeSet<String> entriesSorted = new TreeSet<String>();
        VFSUtils.generateVFSFilesList(resourcesRoot, vfs, entriesSorted, null);
        Path filesList = vfs.resolve(VFS_FILESLIST);
        Files.write(filesList, entriesSorted, new OpenOption[0]);
    }

    public static void generateVFSFilesList(Path vfs) throws IOException {
        VFSUtils.generateVFSFilesList(null, vfs);
    }

    private static String normalizeResourcePath(String path) {
        return REPLACE_BACKSLASHES ? path.replace("\\", "/") : path;
    }

    public static void generateVFSFilesList(Path resourcesRoot, Path vfs, Set<String> ret, Consumer<String> duplicateHandler) throws IOException {
        int rootEndIdx;
        if (!Files.isDirectory(vfs, new LinkOption[0])) {
            throw new IOException(String.format("'%s' has to exist and be a directory.%n", vfs));
        }
        String rootPath = VFSUtils.makeDirPath(vfs.toAbsolutePath());
        if (resourcesRoot == null) {
            rootEndIdx = rootPath.lastIndexOf(File.separator, rootPath.lastIndexOf(File.separator) - 1);
        } else {
            String resRootPath = VFSUtils.makeDirPath(resourcesRoot);
            rootEndIdx = resRootPath.length() - 1;
        }
        try (Stream<Path> s = Files.walk(vfs, new FileVisitOption[0]);){
            s.forEach(p -> {
                if (!VFSUtils.shouldPathBeExcluded(p)) {
                    String entry = null;
                    if (Files.isDirectory(p, new LinkOption[0])) {
                        String dirPath = VFSUtils.makeDirPath(p.toAbsolutePath());
                        entry = dirPath.substring(rootEndIdx);
                    } else if (Files.isRegularFile(p, new LinkOption[0])) {
                        entry = p.toAbsolutePath().toString().substring(rootEndIdx);
                    }
                    if (entry != null && !ret.add(entry = VFSUtils.normalizeResourcePath(entry)) && duplicateHandler != null) {
                        duplicateHandler.accept(entry);
                    }
                }
            });
        }
    }

    private static boolean shouldPathBeExcluded(Path path) {
        for (String glob : DEFAULT_EXCLUDES) {
            PathMatcher matcher = FileSystems.getDefault().getPathMatcher("glob:" + glob);
            if (!matcher.matches(path)) continue;
            return true;
        }
        return false;
    }

    private static String makeDirPath(Path p) {
        Object ret = p.toString();
        if (!((String)ret).endsWith(File.separator)) {
            ret = (String)ret + File.separator;
        }
        return ret;
    }

    private static void delete(Path dir) throws IOException {
        if (!Files.exists(dir, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> s = Files.walk(dir, new FileVisitOption[0]);){
            s.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
        }
        catch (IOException e) {
            throw new IOException(String.format("failed to delete %s", dir), e);
        }
    }

    public static void createVenv(Path venvDirectory, List<String> packagesArgs, Launcher launcherArgs, String graalPyVersion, BuildToolLog log) throws IOException {
        try {
            VFSUtils.createVenv(venvDirectory, packagesArgs, null, null, launcherArgs, graalPyVersion, log);
        }
        catch (PackagesChangedException e) {
            assert (false);
            throw new IllegalStateException(e);
        }
    }

    public static void createVenv(Path venvDirectory, List<String> packages, Path lockFilePath, String missingLockFileWarning, Launcher launcher, String graalPyVersion, BuildToolLog log) throws IOException, PackagesChangedException {
        Objects.requireNonNull(venvDirectory);
        Objects.requireNonNull(packages);
        Objects.requireNonNull(launcher);
        Objects.requireNonNull(graalPyVersion);
        Objects.requireNonNull(log);
        VFSUtils.logVenvArgs(venvDirectory, packages, lockFilePath, launcher, graalPyVersion, log);
        List<String> pluginPackages = VFSUtils.trim(packages);
        LockFile lockFile = null;
        if (lockFilePath != null && Files.exists(lockFilePath, new LinkOption[0])) {
            lockFile = LockFile.fromFile(lockFilePath, log);
        }
        if (!VFSUtils.checkPackages(venvDirectory, pluginPackages, lockFile, log)) {
            return;
        }
        VenvContents venvContents = VFSUtils.ensureVenv(venvDirectory, graalPyVersion, launcher, log);
        InstalledPackages installedPackages = InstalledPackages.fromVenv(venvDirectory);
        boolean installed = lockFile != null ? VFSUtils.install(venvDirectory, installedPackages, lockFile, log) : VFSUtils.install(venvDirectory, pluginPackages, venvContents, log);
        if (installed) {
            venvContents.write(pluginPackages);
            installedPackages.freeze(log);
        }
        if (lockFile == null) {
            VFSUtils.missingLockFileWarning(venvDirectory, pluginPackages, missingLockFileWarning, log);
        }
    }

    private static boolean removedFromPluginPackages(Path venvDirectory, List<String> pluginPackages) throws IOException {
        if (Files.exists(venvDirectory, new LinkOption[0])) {
            VenvContents contents = VenvContents.fromVenv(venvDirectory);
            if (contents == null || contents.packages == null) {
                return false;
            }
            List<String> installedPackages = InstalledPackages.fromVenv((Path)venvDirectory).packages;
            return VFSUtils.removedFromPluginPackages(pluginPackages, contents.packages, installedPackages);
        }
        return false;
    }

    private static boolean removedFromPluginPackages(List<String> pluginPackages, List<String> contentsPackages, List<String> installedPackages) {
        for (String contentsPackage : contentsPackages) {
            String pkgAndVersion;
            if (pluginPackages.contains(contentsPackage) || !contentsPackage.contains("==") && (pkgAndVersion = VFSUtils.getByName(contentsPackage, pluginPackages)) != null && installedPackages.contains(pkgAndVersion)) continue;
            return true;
        }
        return false;
    }

    private static String getByName(String name, List<String> packages) {
        for (String p : packages) {
            String n;
            int idx = p.indexOf("==");
            if (idx <= -1 || !(n = p.split("==")[0]).equals(name)) continue;
            return p;
        }
        return null;
    }

    public static void lockPackages(Path venvDirectory, List<String> packages, Path lockFile, String lockFileHeader, Launcher launcher, String graalPyVersion, BuildToolLog log) throws IOException {
        Objects.requireNonNull(venvDirectory);
        Objects.requireNonNull(packages);
        Objects.requireNonNull(lockFile);
        Objects.requireNonNull(lockFileHeader);
        Objects.requireNonNull(graalPyVersion);
        Objects.requireNonNull(log);
        VFSUtils.createVenv(venvDirectory, packages, launcher, graalPyVersion, log);
        if (Files.exists(venvDirectory, new LinkOption[0])) {
            LockFile.write(venvDirectory, lockFile, lockFileHeader, packages, graalPyVersion, log);
        } else {
            VFSUtils.warning(log, "did not generate new python lock file due to missing python virtual environment");
        }
    }

    private static void logVenvArgs(Path venvDirectory, List<String> packages, Path lockFile, Launcher launcherArgs, String graalPyVersion, BuildToolLog log) throws IOException {
        if (log.isDebugEnabled()) {
            Set<String> lcp = launcherArgs.computeClassPath();
            log.debug("VFSUtils.createVenv with:");
            log.debug("  graalPyVersion: " + graalPyVersion);
            log.debug("  venvDirectory: " + String.valueOf(venvDirectory));
            log.debug("  packages: " + String.valueOf(packages));
            log.debug("  lock file: " + String.valueOf(lockFile));
            log.debug("  launcher: " + String.valueOf(launcherArgs.launcherPath));
            log.debug("  launcher classpath: " + String.valueOf(lcp));
        }
    }

    private static boolean checkPackages(Path venvDirectory, List<String> pluginPackages, LockFile lockFile, BuildToolLog log) throws IOException, PackagesChangedException {
        if (lockFile != null) {
            VFSUtils.checkPluginPackagesInLockFile(pluginPackages, lockFile);
            VFSUtils.logPackages(lockFile.packages, lockFile.path, log);
            return VFSUtils.needVenv(venvDirectory, lockFile.packages, log);
        }
        if (VFSUtils.removedFromPluginPackages(venvDirectory, pluginPackages)) {
            VFSUtils.info(log, "A package with transitive dependencies was removed since last install, setting up a clean venv", new Object[0]);
            VFSUtils.delete(venvDirectory);
        }
        VFSUtils.logPackages(pluginPackages, null, log);
        return VFSUtils.needVenv(venvDirectory, pluginPackages, log);
    }

    private static boolean needVenv(Path venvDirectory, List<String> packages, BuildToolLog log) throws IOException {
        if (packages.isEmpty()) {
            if (Files.exists(venvDirectory, new LinkOption[0])) {
                VFSUtils.info(log, "No packages to install, deleting venv", new Object[0]);
                VFSUtils.delete(venvDirectory);
            } else {
                VFSUtils.debug(log, "VFSUtils skipping venv create - no package or lock file provided");
            }
            return false;
        }
        return true;
    }

    private static void logPackages(List<String> packages, Path lockFile, BuildToolLog log) {
        if (lockFile != null) {
            VFSUtils.info(log, "Got %s python package(s) in lock file: %s", packages.size(), lockFile);
        } else {
            VFSUtils.info(log, "Got %s python package(s) in GraalPy plugin configuration", packages.size());
        }
        if (log.isDebugEnabled()) {
            for (String pkg : packages) {
                log.debug("    " + pkg);
            }
        }
    }

    private static List<String> readPackagesFromFile(Path file) throws IOException {
        return Files.readAllLines(file).stream().filter(s -> {
            if (s == null) {
                return false;
            }
            String l = s.trim();
            return !l.startsWith("#") && !s.isEmpty();
        }).toList();
    }

    private static VenvContents ensureVenv(Path venvDirectory, String graalPyVersion, Launcher launcher, BuildToolLog log) throws IOException {
        Path launcherPath = VFSUtils.ensureLauncher(launcher, log);
        VenvContents contents = null;
        if (Files.exists(venvDirectory, new LinkOption[0])) {
            VFSUtils.checkVenvLauncher(venvDirectory, launcherPath, log);
            contents = VenvContents.fromVenv(venvDirectory);
            if (contents == null) {
                VFSUtils.warning(log, "Reinstalling GraalPy venv due to corrupt contents file");
                VFSUtils.delete(venvDirectory);
            } else if (!graalPyVersion.equals(contents.graalPyVersion)) {
                contents = null;
                VFSUtils.info(log, "Stale GraalPy virtual environment, updating to %s", graalPyVersion);
                VFSUtils.delete(venvDirectory);
            } else if (!PLATFORM.equals(contents.platform)) {
                VFSUtils.info(log, "Reinstalling GraalPy venv created on %s, but current is'%s", contents.platform, PLATFORM);
                contents = null;
                VFSUtils.delete(venvDirectory);
            }
        }
        if (!Files.exists(venvDirectory, new LinkOption[0])) {
            VFSUtils.info(log, "Creating GraalPy %s venv", graalPyVersion);
            VFSUtils.runLauncher(launcherPath.toString(), log, "-m", VFS_VENV, venvDirectory.toString(), "--without-pip");
            VFSUtils.runVenvBin(venvDirectory, "graalpy", log, "-I", "-m", "ensurepip");
        }
        if (contents == null) {
            contents = VenvContents.create(venvDirectory, graalPyVersion);
        }
        return contents;
    }

    private static boolean install(Path venvDirectory, InstalledPackages installedPackages, LockFile lockFile, BuildToolLog log) throws IOException {
        if (installedPackages.packages.size() != lockFile.packages.size() || VFSUtils.deleteUnwantedPackages(venvDirectory, lockFile.packages, installedPackages.packages, log)) {
            VFSUtils.runPip(venvDirectory, "install", log, "--compile", "-r", lockFile.path.toString());
            return true;
        }
        VFSUtils.info(log, "Virtual environment is up to date with lock file, skipping install", new Object[0]);
        return false;
    }

    private static boolean install(Path venvDirectory, List<String> newPackages, VenvContents venvContents, BuildToolLog log) throws IOException {
        boolean needsUpdate = false;
        needsUpdate |= VFSUtils.deleteUnwantedPackages(venvDirectory, newPackages, venvContents.packages, log);
        return needsUpdate |= VFSUtils.installWantedPackages(venvDirectory, newPackages, venvContents.packages, log);
    }

    private static void missingLockFileWarning(Path venvDirectory, List<String> newPackages, String missingLockFileWarning, BuildToolLog log) throws IOException {
        if (missingLockFileWarning != null && !Boolean.getBoolean("graalpy.vfs.skipMissingLockFileWarning") && !newPackages.containsAll(InstalledPackages.fromVenv((Path)venvDirectory).packages) && log.isWarningEnabled()) {
            String txt = missingLockFileWarning + "\n";
            for (String t : txt.split("\n")) {
                log.warning(t);
            }
        }
    }

    private static void checkPluginPackagesInLockFile(List<String> pluginPackages, LockFile lockFile) throws PackagesChangedException {
        if (pluginPackages.size() != lockFile.inputPackages.size() || !pluginPackages.containsAll(lockFile.inputPackages)) {
            throw new PackagesChangedException(new ArrayList<String>(pluginPackages), new ArrayList<String>(lockFile.inputPackages));
        }
    }

    private static void checkVenvLauncher(Path venvDirectory, Path launcherPath, BuildToolLog log) throws IOException {
        if (!Files.exists(launcherPath, new LinkOption[0])) {
            throw new IOException(String.format("Launcher file does not exist '%s'", launcherPath));
        }
        Path cfg = venvDirectory.resolve("pyvenv.cfg");
        if (Files.exists(cfg, new LinkOption[0])) {
            try {
                List<String> lines = Files.readAllLines(cfg);
                for (String line : lines) {
                    int idx = line.indexOf("=");
                    if (idx <= -1) continue;
                    String l = line.substring(0, idx).trim();
                    String r = line.substring(idx + 1).trim();
                    if (!l.trim().equals("executable")) continue;
                    Path cfgLauncherPath = Path.of(r, new String[0]);
                    if (Files.exists(cfgLauncherPath, new LinkOption[0]) && Files.isSameFile(launcherPath, cfgLauncherPath)) break;
                    VFSUtils.info(log, "Deleting GraalPy venv due to changed launcher path", new Object[0]);
                    VFSUtils.delete(venvDirectory);
                }
            }
            catch (IOException e) {
                e.printStackTrace();
                throw new IOException(String.format("failed to read config file %s", cfg), e);
            }
        } else {
            VFSUtils.info(log, "Missing venv config file: '%s'", cfg);
        }
    }

    private static Path ensureLauncher(Launcher launcherArgs, BuildToolLog log) throws IOException {
        String externalLauncher = System.getProperty("graalpy.vfs.venvLauncher");
        if (externalLauncher == null || externalLauncher.trim().isEmpty()) {
            VFSUtils.generateLaunchers(launcherArgs, log);
            return launcherArgs.launcherPath;
        }
        return Path.of(externalLauncher, new String[0]);
    }

    private static boolean checkWinLauncherJavaPath(Path venvCfg, Path java) {
        try {
            for (String line : Files.readAllLines(venvCfg)) {
                if (!line.trim().startsWith("venvlauncher_command = " + String.valueOf(java))) continue;
                return true;
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return false;
    }

    private static String formatMultiline(String str, Object ... args) {
        return str.formatted(args);
    }

    private static void generateLaunchers(Launcher launcherArgs, BuildToolLog log) throws IOException {
        VFSUtils.debug(log, "Generating GraalPy launchers");
        VFSUtils.createParentDirectories(launcherArgs.launcherPath);
        Path java = Paths.get(System.getProperty("java.home"), "bin", "java");
        String classpath = String.join((CharSequence)File.pathSeparator, launcherArgs.computeClassPath());
        String extraJavaOptions = String.join((CharSequence)" ", GraalPyRunner.getExtraJavaOptions());
        if (!IS_WINDOWS) {
            String script = VFSUtils.formatMultiline("#!/usr/bin/env bash\n%s --enable-native-access=ALL-UNNAMED %s -classpath %s %s --python.Executable=\"$0\" \"$@\"\n", java, extraJavaOptions, String.join((CharSequence)File.pathSeparator, classpath), GRAALPY_MAIN_CLASS);
            try {
                Files.writeString(launcherArgs.launcherPath, (CharSequence)script, new OpenOption[0]);
                Set<PosixFilePermission> perms = Files.getPosixFilePermissions(launcherArgs.launcherPath, new LinkOption[0]);
                perms.addAll(List.of(PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_EXECUTE));
                Files.setPosixFilePermissions(launcherArgs.launcherPath, perms);
            }
            catch (IOException e) {
                throw new IOException(String.format("failed to create launcher %s", launcherArgs.launcherPath), e);
            }
        }
        if (!Files.exists(launcherArgs.launcherPath, new LinkOption[0]) || !VFSUtils.checkWinLauncherJavaPath(launcherArgs.launcherPath.getParent().resolve("pyenv.cfg"), java)) {
            File tmp;
            String script = VFSUtils.formatMultiline("import os, shutil, struct, venv\nfrom pathlib import Path\nvl = os.path.join(venv.__path__[0], 'scripts', 'nt', 'graalpy.exe')\ntl = os.path.join(r'%s')\nos.makedirs(Path(tl).parent.absolute(), exist_ok=True)\nshutil.copy(vl, tl)\ncmd = r'%s --enable-native-access=ALL-UNNAMED %s -classpath \"%s\" %s'\npyvenvcfg = os.path.join(os.path.dirname(tl), \"pyvenv.cfg\")\nwith open(pyvenvcfg, 'w', encoding='utf-8') as f:\n    f.write('venvlauncher_command = ')\n    f.write(cmd)\n", launcherArgs.launcherPath, java, extraJavaOptions, classpath, GRAALPY_MAIN_CLASS);
            try {
                tmp = File.createTempFile("create_launcher", ".py");
            }
            catch (IOException e) {
                throw new IOException("failed to create tmp launcher", e);
            }
            tmp.deleteOnExit();
            try (FileWriter wr = new FileWriter(tmp, StandardCharsets.UTF_8);){
                wr.write(script);
            }
            catch (IOException e) {
                throw new IOException(String.format("failed to write tmp launcher %s", tmp), e);
            }
            try {
                GraalPyRunner.run(classpath, log, tmp.getAbsolutePath());
            }
            catch (InterruptedException e) {
                throw new IOException("failed to run Graalpy launcher", e);
            }
        }
    }

    private static boolean installWantedPackages(Path venvDirectory, List<String> packages, List<String> installedPackages, BuildToolLog log) throws IOException {
        HashSet<String> pkgsToInstall = new HashSet<String>(packages);
        pkgsToInstall.removeAll(installedPackages);
        if (pkgsToInstall.isEmpty()) {
            return false;
        }
        ArrayList<String> args = new ArrayList<String>(pkgsToInstall.size() + 1);
        args.add("--compile");
        args.addAll(pkgsToInstall);
        VFSUtils.runPip(venvDirectory, "install", log, args.toArray(new String[0]));
        return true;
    }

    private static boolean deleteUnwantedPackages(Path venvDirectory, List<String> packages, List<String> installedPackages, BuildToolLog log) throws IOException {
        ArrayList<String> args = new ArrayList<String>(installedPackages);
        args.removeAll(packages);
        if (args.isEmpty()) {
            return false;
        }
        args.add(0, "-y");
        VFSUtils.runPip(venvDirectory, "uninstall", log, args.toArray(new String[args.size()]));
        return true;
    }

    private static void runLauncher(String launcherPath, BuildToolLog log, String ... args) throws IOException {
        try {
            GraalPyRunner.runLauncher(launcherPath, log, args);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to execute launcher command %s", List.of(args)));
        }
    }

    private static void runPip(Path venvDirectory, String command, BuildToolLog log, String ... args) throws IOException {
        try {
            GraalPyRunner.runPip(venvDirectory, command, log, args);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to execute pip %s", List.of(args)), e);
        }
    }

    private static void runVenvBin(Path venvDirectory, String bin, BuildToolLog log, String ... args) throws IOException {
        try {
            GraalPyRunner.runVenvBin(venvDirectory, bin, log, args);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to execute venv %s", List.of(args)), e);
        }
    }

    private static List<String> trim(List<String> l) {
        Iterator<String> it = l.iterator();
        while (it.hasNext()) {
            String p = it.next();
            if (p != null && !p.trim().isEmpty()) continue;
            it.remove();
        }
        return l;
    }

    private static void warning(BuildToolLog log, String txt) {
        log.warning(txt);
    }

    @SuppressFBWarnings(value={"UC_USELESS_VOID_METHOD"})
    private static void info(BuildToolLog log, String txt, Object ... args) {
        if (log.isInfoEnabled()) {
            log.info(String.format(txt, args));
        }
    }

    private static void lifecycle(BuildToolLog log, String txt, Object ... args) {
        if (log.isLifecycleEnabled()) {
            log.lifecycle(String.format(txt, args));
        }
    }

    private static void debug(BuildToolLog log, String txt) {
        log.debug(txt);
    }

    private static void logDebug(BuildToolLog log, List<String> l, String msg, Object ... args) {
        if (log.isDebugEnabled()) {
            if (msg != null) {
                log.debug(String.format(msg, args));
            }
            for (String p : l) {
                log.debug("  " + p);
            }
        }
    }

    public static abstract class Launcher {
        private final Path launcherPath;

        protected Launcher(Path launcherPath) {
            Objects.requireNonNull(launcherPath);
            this.launcherPath = launcherPath;
        }

        protected abstract Set<String> computeClassPath() throws IOException;
    }

    public static final class PackagesChangedException
    extends Exception {
        private static final long serialVersionUID = 9162516912727973035L;
        private final transient List<String> pluginPackages;
        private final transient List<String> lockFilePackages;

        private PackagesChangedException(List<String> pluginPackages, List<String> lockFilePackages) {
            this.pluginPackages = pluginPackages;
            this.lockFilePackages = lockFilePackages;
        }

        public List<String> getPluginPackages() {
            return this.pluginPackages;
        }

        public List<String> getLockFilePackages() {
            return this.lockFilePackages;
        }
    }

    private static class LockFile {
        final Path path;
        final List<String> packages;
        final List<String> inputPackages;

        private LockFile(Path path, List<String> inputPackages, List<String> packages) {
            this.path = path;
            this.packages = packages;
            this.inputPackages = inputPackages;
        }

        static LockFile fromFile(Path file, BuildToolLog log) throws IOException {
            List<String> lines;
            ArrayList<String> packages = new ArrayList<String>();
            List<String> inputPackages = null;
            try {
                lines = Files.readAllLines(file);
            }
            catch (IOException e) {
                throw new IOException(String.format("Cannot read the lock file from '%s'", file), e);
            }
            if (lines.isEmpty()) {
                throw LockFile.wrongFormat(file, lines, log);
            }
            Iterator<String> it = lines.iterator();
            try {
                String line;
                String graalPyVersion = null;
                while (it.hasNext()) {
                    line = it.next();
                    if (!line.startsWith(VFSUtils.GRAALPY_VERSION_PREFIX)) continue;
                    graalPyVersion = line.substring(VFSUtils.GRAALPY_VERSION_PREFIX.length()).trim();
                    if (!graalPyVersion.isEmpty()) break;
                    throw LockFile.wrongFormat(file, lines, log);
                }
                if (graalPyVersion == null) {
                    throw LockFile.wrongFormat(file, lines, log);
                }
                line = it.next();
                if (!line.startsWith(VFSUtils.INPUT_PACKAGES_PREFIX)) {
                    throw LockFile.wrongFormat(file, lines, log);
                }
                String pkgs = line.substring(VFSUtils.INPUT_PACKAGES_PREFIX.length()).trim();
                if (pkgs.isEmpty()) {
                    throw LockFile.wrongFormat(file, lines, log);
                }
                inputPackages = Arrays.asList(pkgs.split(VFSUtils.INPUT_PACKAGES_DELIMITER));
                while (it.hasNext()) {
                    packages.add(it.next());
                }
            }
            catch (NoSuchElementException e) {
                throw LockFile.wrongFormat(file, lines, log);
            }
            return new LockFile(file, inputPackages, packages);
        }

        private static IOException wrongFormat(Path file, List<String> lines, BuildToolLog log) {
            if (log.isDebugEnabled()) {
                log.debug("wrong format of lock file " + String.valueOf(file));
                for (String l : lines) {
                    log.debug(l);
                }
                log.debug("");
            }
            return new IOException(String.format("Cannot read the lock file from '%s'%n(turn on debug log level to see the contents)", file));
        }

        private static void write(Path venvDirectory, Path lockFile, String lockFileHeader, List<String> inputPackages, String graalPyVersion, BuildToolLog log) throws IOException {
            Objects.requireNonNull(venvDirectory);
            Objects.requireNonNull(lockFile);
            Objects.requireNonNull(lockFileHeader);
            Objects.requireNonNull(log);
            assert (Files.exists(venvDirectory, new LinkOption[0]));
            InstalledPackages installedPackages = InstalledPackages.fromVenv(venvDirectory);
            List<String> header = LockFile.getHeaderList(lockFileHeader);
            header.add(VFSUtils.GRAALPY_VERSION_PREFIX + graalPyVersion);
            header.add(VFSUtils.INPUT_PACKAGES_PREFIX + String.join((CharSequence)VFSUtils.INPUT_PACKAGES_DELIMITER, inputPackages));
            Files.write(lockFile, header, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
            Files.write(lockFile, installedPackages.packages, StandardOpenOption.APPEND);
            VFSUtils.lifecycle(log, "Created GraalPy lock file: %s", lockFile);
            VFSUtils.logDebug(log, installedPackages.packages, null, new Object[0]);
        }

        private static List<String> getHeaderList(String lockFileHeader) {
            String[] lines;
            ArrayList<String> list = new ArrayList<String>();
            for (String l : lines = lockFileHeader.split("\n")) {
                list.add("# " + l);
            }
            return list;
        }
    }

    private static class VenvContents {
        private static final String KEY_VERSION = "version";
        private static final String KEY_PACKAGES = "input_packages";
        private static final String KEY_PLATFORM = "platform";
        private static final String CONTENTS_FILE_NAME = "contents";
        final Path contentsFile;
        List<String> packages;
        final String graalPyVersion;
        final String platform;

        private VenvContents(Path contentsFile, List<String> packages, String graalPyVersion, String platform) {
            this.contentsFile = contentsFile;
            this.packages = packages;
            this.graalPyVersion = graalPyVersion;
            this.platform = platform;
        }

        static VenvContents create(Path venvDirectory, String graalPyVersion) {
            return new VenvContents(venvDirectory.resolve(CONTENTS_FILE_NAME), Collections.emptyList(), graalPyVersion, PLATFORM);
        }

        static VenvContents fromVenv(Path venvDirectory) throws IOException {
            Path contentsFile = venvDirectory.resolve(CONTENTS_FILE_NAME);
            if (Files.exists(contentsFile, new LinkOption[0])) {
                List<String> lines = Files.readAllLines(contentsFile);
                if (lines.isEmpty()) {
                    return null;
                }
                if (lines.get(0).startsWith("version=")) {
                    HashMap<String, String> m = new HashMap<String, String>();
                    for (String l : lines) {
                        int idx = l.indexOf("=");
                        m.put(l.substring(0, idx), l.substring(idx + 1));
                    }
                    String graalPyVersion = (String)m.get(KEY_VERSION);
                    String platform = (String)m.get(KEY_PLATFORM);
                    String packagesLine = (String)m.get(KEY_PACKAGES);
                    List<String> packages = packagesLine != null ? Arrays.asList(packagesLine.split(VFSUtils.INPUT_PACKAGES_DELIMITER)) : Collections.emptyList();
                    return new VenvContents(contentsFile, packages, graalPyVersion, platform);
                }
                ArrayList<String> packages = new ArrayList<String>();
                Iterator<String> it = lines.iterator();
                String graalPyVersion = it.next();
                while (it.hasNext()) {
                    packages.add(it.next());
                }
                return new VenvContents(contentsFile, packages, graalPyVersion, null);
            }
            return null;
        }

        void write(List<String> pkgs) throws IOException {
            ArrayList<CallSite> lines = new ArrayList<CallSite>();
            lines.add((CallSite)((Object)("version=" + this.graalPyVersion)));
            lines.add((CallSite)((Object)("platform=" + PLATFORM)));
            lines.add((CallSite)((Object)("input_packages=" + String.join((CharSequence)VFSUtils.INPUT_PACKAGES_DELIMITER, pkgs))));
            Files.write(this.contentsFile, lines, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        }
    }

    private static class InstalledPackages {
        final Path venvDirectory;
        final Path installedFile;
        List<String> packages;

        private InstalledPackages(Path venvDirectory, Path installedFile, List<String> packages) {
            this.venvDirectory = venvDirectory;
            this.installedFile = installedFile;
            this.packages = packages;
        }

        static InstalledPackages fromVenv(Path venvDirectory) throws IOException {
            Path installed = venvDirectory.resolve("installed.txt");
            List<String> pkgs = Files.exists(installed, new LinkOption[0]) ? VFSUtils.readPackagesFromFile(installed) : Collections.emptyList();
            return new InstalledPackages(venvDirectory, installed, pkgs);
        }

        List<String> freeze(BuildToolLog log) throws IOException {
            BuildToolLog.CollectOutputLog collectOutputLog = new BuildToolLog.CollectOutputLog(log);
            VFSUtils.runPip(this.venvDirectory, "freeze", collectOutputLog, "--local");
            this.packages = new ArrayList<String>(collectOutputLog.getOutput());
            String toWrite = "# Generated by GraalPy Maven or Gradle plugin using pip freeze\n# This file is used by GraalPy VirtualFileSystem\n" + String.join((CharSequence)"\n", this.packages);
            Files.write(this.installedFile, toWrite.getBytes(StandardCharsets.UTF_8), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
            VFSUtils.logDebug(log, this.packages, "VFSUtils venv packages after install %s:", this.installedFile);
            return this.packages;
        }
    }
}

