package com.foxcode.android.common.nointernet.components

import android.content.Context
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkCapabilities
import android.net.NetworkInfo
import android.net.NetworkRequest
import android.util.Log
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleObserver
import androidx.lifecycle.OnLifecycleEvent
import com.foxcode.android.common.nointernet.utils.NoInternetUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch

internal class NoInternetObserveComponent(
    private val applicationContext: Context,
    lifecycle: Lifecycle,
    private val listener: NoInternetObserverListener
) : LifecycleObserver {
    private val coroutineScope = CoroutineScope(Dispatchers.IO)
    private var currentJob: Job? = null

    private var connectivityManager: ConnectivityManager =
        applicationContext.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

    private var connectivityManagerCallback: ConnectivityManager.NetworkCallback? = null

    init {
        lifecycle.addObserver(this)
    }

    private fun initReceivers() {
        updateConnection()

        val builder = NetworkRequest.Builder()
            .addTransportType(NetworkCapabilities.TRANSPORT_CELLULAR)
            .addTransportType(NetworkCapabilities.TRANSPORT_WIFI)
            .addTransportType(NetworkCapabilities.TRANSPORT_ETHERNET)
            .removeCapability(NetworkCapabilities.NET_CAPABILITY_NOT_VPN)
        connectivityManager.registerNetworkCallback(
            builder.build(),
            getConnectivityManagerCallback()
        )
    }

    private fun updateConnection() {
        val activeNetwork: NetworkInfo? = connectivityManager.activeNetworkInfo
        if (activeNetwork?.isConnected == true) {
            Log.d(TAG, "activeNetwork?.isConnected == true")
            listener.onConnected()
        } else {
            Log.d(TAG, "activeNetwork?.isConnected != true")
            checkInternetAndInvokeListener()
        }
    }

    private fun getConnectivityManagerCallback(): ConnectivityManager.NetworkCallback {
        connectivityManagerCallback = object : ConnectivityManager.NetworkCallback() {
            override fun onAvailable(network: Network) {
                Log.d(TAG, "onAvailable(): $network")
                listener.onConnected()
            }

            override fun onLost(network: Network) {
                Log.d(TAG, "onLost(): $network")
                checkInternetAndInvokeListener()
            }
        }
        return connectivityManagerCallback!!
    }

    /**
     * Check if the device is connected to the internet.
     * It checks if the device is connected (by wifi or data connection etc.) to the internet,
     * and ping google to make sure the connection is active.
     */
    private fun checkInternetAndInvokeListener() {
        currentJob = coroutineScope.launch {
            val hasActiveConnection = NoInternetUtils.isConnectedToInternet(applicationContext)
//                && NoInternetUtils.hasActiveInternetConnection()

            if (!hasActiveConnection) {
                launch(Dispatchers.Main) {
                    listener.onDisconnected(NoInternetUtils.isAirplaneModeOn(applicationContext))
                }
            }
        }
    }

    /**
     * Start listening.
     */
    @OnLifecycleEvent(Lifecycle.Event.ON_RESUME)
    fun start() {
        Log.d(TAG, "start")

        listener.onStart()
        initReceivers()
    }

    /**
     * Stop listening.
     */
    @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
    fun stop() {
        Log.d(TAG, "stop")

        connectivityManagerCallback?.let {
            try {
                connectivityManager.unregisterNetworkCallback(it)
            } catch (e: Exception) {
                // no-op
            }
        }

        currentJob?.cancel()
        listener.onStop()
    }

    interface NoInternetObserverListener {
        fun onStart()

        fun onConnected()

        fun onDisconnected(isAirplaneModeOn: Boolean)

        fun onStop()
    }

    companion object {
        private const val TAG = "NoInternetObserve"
    }
}
