/*
 * Copyright (c) 2009 by Cosylab
 *
 * The full license specifying the redistribution, modification, usage and other
 * rights and obligations is included with the distribution of this project in
 * the file "LICENSE-CAJ". If the license is not included visit Cosylab web site,
 * <http://www.cosylab.com>.
 *
 * THIS SOFTWARE IS PROVIDED AS-IS WITHOUT WARRANTY OF ANY KIND, NOT EVEN THE
 * IMPLIED WARRANTY OF MERCHANTABILITY. THE AUTHOR OF THIS SOFTWARE, ASSUMES
 * _NO_ RESPONSIBILITY FOR ANY CONSEQUENCE RESULTING FROM THE USE, MODIFICATION,
 * OR REDISTRIBUTION OF THIS SOFTWARE.
 */

package org.epics.pvaccess.impl.remote.tcp;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.SocketChannel;

import org.epics.pvaccess.PVAConstants;
import org.epics.pvaccess.impl.remote.Context;
import org.epics.pvaccess.impl.remote.IntrospectionRegistry;
import org.epics.pvaccess.impl.remote.ProtocolType;
import org.epics.pvaccess.impl.remote.Transport;
import org.epics.pvaccess.impl.remote.codec.AbstractCodec;
import org.epics.pvaccess.impl.remote.codec.impl.BlockingSocketAbstractCodec;
import org.epics.pvaccess.impl.remote.request.ResponseHandler;
import org.epics.pvaccess.server.ServerContext;
import org.epics.pvdata.pv.Field;
import org.epics.pvdata.pv.Status;


/**
 * TCP transport implementation.
 * @author <a href="mailto:matej.sekoranjaATcosylab.com">Matej Sekoranja</a>
 * @version $Id$
 */
public abstract class BlockingTCPTransport extends BlockingSocketAbstractCodec implements Transport {

	/**
	 * Context instance.
	 */
	protected final Context context;

	/**
	 * Priority.
	 * NOTE: Priority cannot just be changed, since it is registered in transport registry with given priority.
	 */
	protected final short priority;
	// TODO to be implemeneted 
	
	/**
	 * PVAS response handler.
	 */
	protected final ResponseHandler responseHandler;

	/**
	 * Incoming (codes generated by other party) introspection registry.
	 */
	protected final IntrospectionRegistry incomingIR = new IntrospectionRegistry();

	/**
	 * Outgoing (codes generated by this party) introspection registry.
	 */
	protected final IntrospectionRegistry outgoingIR = new IntrospectionRegistry();
	
	/**
	 * Cached byte-order flag. To be used only in send thread.
	 */
	//private int byteOrderFlag = 0x80;		// TODO
	
	/**
	 * Remote side transport revision (minor).
	 */
	protected byte remoteTransportRevision;		// TODO sync

	/**
	 * TCP transport constructor.
	 * @param context context where transport lives in.
	 * @param channel used socket channel.
	 * @param responseHandler response handler used to process PVA headers.
	 * @param receiveBufferSize receive buffer size.
	 * @param priority transport priority.
	 * @throws SocketException thrown on any socket exception.
	 */
	public BlockingTCPTransport(Context context, 
					   SocketChannel channel,
					   ResponseHandler responseHandler,
					   int receiveBufferSize,
					   short priority) throws SocketException {
		super(context instanceof ServerContext, channel, 
				ByteBuffer.allocate(Math.max(PVAConstants.MAX_TCP_RECV + AbstractCodec.MAX_ENSURE_DATA_SIZE, receiveBufferSize)),
				ByteBuffer.allocate(Math.max(PVAConstants.MAX_TCP_RECV + AbstractCodec.MAX_ENSURE_DATA_SIZE, receiveBufferSize)),
				context.getLogger());
		this.context = context;
		this.responseHandler = responseHandler;
		this.remoteTransportRevision = 0;
		this.priority = priority;

		// add to registry
		context.getTransportRegistry().put(this);
	}
	

	@Override
	protected void internalDestroy() {
		super.internalDestroy();

		// remove from registry
		context.getTransportRegistry().remove(this);
	
		// clean resources
		internalClose();
	}
	
	// TODO
	/**
	 * Called to any resources just before closing transport
	 */
	protected void internalClose()
	{
		InetSocketAddress remoteAddress = getRemoteAddress();
		if (remoteAddress != null)
			context.getLogger().finer("TCP socket to " + remoteAddress + " closed.");
		else
			context.getLogger().finer("TCP socket to 'unknown' closed.");
	}
	
	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#getType()
	 */
	@Override
	public String getType() {
		return ProtocolType.tcp.name();
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#getRemoteAddress()
	 */
	@Override
	public InetSocketAddress getRemoteAddress() {
		return socketAddress;
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#getContext()
	 */
	@Override
	public Context getContext() {
		return context;
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#getMinorRevision()
	 */
	@Override
	public byte getRevision() {
		return PVAConstants.PVA_PROTOCOL_REVISION;
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#getReceiveBufferSize()
	 */
	@Override
	public int getReceiveBufferSize() {
		return socketBuffer.capacity();
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#getSocketReceiveBufferSize()
	 */
	@Override
	public int getSocketReceiveBufferSize() {
		try {
			return channel.socket().getReceiveBufferSize();
		} catch (SocketException e) {
			// error
			return -1;
		}
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#getPriority()
	 */
	@Override
	public short getPriority() {
		// TODO Auto-generated method stub
		return priority;
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#setRemoteMinorRevision(byte)
	 */
	@Override
	public void setRemoteRevision(byte minor) {
		// TODO Auto-generated method stub
		//this.remoteTransportRevision = minor;
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#setRemoteTransportReceiveBufferSize(int)
	 */
	@Override
	public void setRemoteTransportReceiveBufferSize(int receiveBufferSize) {
		// TODO Auto-generated method stub
		
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#setRemoteTransportSocketReceiveBufferSize(int)
	 */
	@Override
	public void setRemoteTransportSocketReceiveBufferSize(
			int socketReceiveBufferSize) {
		// TODO Auto-generated method stub
		
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#changedTransport()
	 */
	@Override
	public void changedTransport() {
		// TODO Auto-generated method stub
		
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.codec.AbstractCodec#processControlMessage()
	 */
	@Override
	public void processControlMessage() {

		// TODO
		/*
		// marker request sent
		if (command == 0)
		{
			if (markerToSend.getAndSet(payloadSize) == 0)
				; // TODO send back response
		}
		// marker received back
		else if (command == 1)
		{
			int difference = (int)totalBytesSent - payloadSize + PVAConstants.PVA_MESSAGE_HEADER_SIZE;
			// overrun check
			if (difference < 0)
				difference += Integer.MAX_VALUE;
			remoteBufferFreeSpace = remoteTransportReceiveBufferSize + remoteTransportSocketReceiveBufferSize - difference; 
			// TODO if this is calculated wrong, this can be critical !!!
		}
		// set byte order
		else */if (command == 2)
		{
			// check 7-th bit
			setByteOrder(flags < 0 ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN);
		}
		
	}

	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.codec.AbstractCodec#processApplicationMessage()
	 */
	@Override
	public void processApplicationMessage() throws IOException {
		responseHandler.handleResponse(socketAddress, this, version, command, payloadSize, socketBuffer);
	}

	/* (non-Javadoc)
	 * @see org.epics.pvdata.pv.DeserializableControl#cachedDeserialize(java.nio.ByteBuffer)
	 */
	@Override
	public Field cachedDeserialize(ByteBuffer buffer) {
		return incomingIR.deserialize(buffer, this);
	}

	/* (non-Javadoc)
	 * @see org.epics.pvdata.pv.SerializableControl#cachedSerialize(org.epics.pvdata.pv.Field, java.nio.ByteBuffer)
	 */
	@Override
	public void cachedSerialize(Field field, ByteBuffer buffer) {
		outgoingIR.serialize(field, buffer, this);
	}
	
	protected boolean verifiedCalled = false;
	protected boolean verified = false;
	private Object verifiedMonitor = new Object();
	
	@Override
	public void verified(Status status) {
		synchronized (verifiedMonitor) {
			
			if (!status.isOK())
			{
				String logMessage ="Failed to verify connection to " + socketAddress + ": " + status.getMessage();
				String stackDump = status.getStackDump();
				if (stackDump != null && !stackDump.isEmpty())
					logMessage += "\n" + stackDump;
				context.getLogger().fine(logMessage);
			}
			
			verifiedCalled = true;
			verified = status.isSuccess();
			verifiedMonitor.notifyAll();
		}
	}
	
	/* (non-Javadoc)
	 * @see org.epics.pvaccess.impl.remote.Transport#verify(long)
	 */
	@Override
	public boolean verify(long timeoutMs) {
		synchronized (verifiedMonitor) {
			try {
				final long start = System.currentTimeMillis();
				while (!verifiedCalled && (System.currentTimeMillis() - start) < timeoutMs)
						verifiedMonitor.wait(timeoutMs);
			} catch (InterruptedException e) {
				// noop
			}
			return verified;
		}
	}
	
}
