/*
 * Decompiled with CFR 0.152.
 */
package org.entur.gbfs;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.entur.gbfs.HttpUtils;
import org.entur.gbfs.authentication.DummyRequestAuthenticator;
import org.entur.gbfs.authentication.RequestAuthenticationException;
import org.entur.gbfs.authentication.RequestAuthenticator;
import org.entur.gbfs.v2_3.gbfs.GBFS;
import org.entur.gbfs.v2_3.gbfs.GBFSFeed;
import org.entur.gbfs.v2_3.gbfs.GBFSFeedName;
import org.entur.gbfs.v2_3.gbfs.GBFSFeeds;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GbfsLoader {
    private static final Logger LOG = LoggerFactory.getLogger(GbfsLoader.class);
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private final Map<GBFSFeedName, GBFSFeedUpdater<?>> feedUpdaters = new HashMap();
    private final String url;
    private final Map<String, String> httpHeaders;
    private final String languageCode;
    private final RequestAuthenticator requestAuthenticator;
    private byte[] rawDiscoveryFileData;
    private GBFS disoveryFileData;
    private final AtomicBoolean setupComplete = new AtomicBoolean(false);
    private final Lock updateLock = new ReentrantLock();
    private final Long timeoutConnection;

    public GbfsLoader(String url) {
        this(url, new HashMap<String, String>(), null);
    }

    public GbfsLoader(String url, String languageCode) {
        this(url, new HashMap<String, String>(), languageCode);
    }

    public GbfsLoader(String url, Map<String, String> httpHeaders, String languageCode) {
        this(url, httpHeaders, languageCode, null, null);
    }

    public GbfsLoader(String url, String languageCode, RequestAuthenticator requestAuthenticator) {
        this(url, new HashMap<String, String>(), languageCode, requestAuthenticator, null);
    }

    public GbfsLoader(String url, String languageCode, RequestAuthenticator requestAuthenticator, Long timeoutConnection) {
        this(url, new HashMap<String, String>(), languageCode, requestAuthenticator, timeoutConnection);
    }

    public GbfsLoader(String url, Map<String, String> httpHeaders, String languageCode, RequestAuthenticator requestAuthenticator, Long timeoutConnection) {
        this.requestAuthenticator = requestAuthenticator == null ? new DummyRequestAuthenticator() : requestAuthenticator;
        this.url = url;
        this.httpHeaders = httpHeaders;
        this.languageCode = languageCode;
        this.timeoutConnection = timeoutConnection;
        this.init();
    }

    public AtomicBoolean getSetupComplete() {
        return this.setupComplete;
    }

    private synchronized void init() {
        URI uri;
        if (this.setupComplete.get()) {
            return;
        }
        try {
            uri = new URI(this.url);
        }
        catch (URISyntaxException e) {
            throw new RuntimeException("Invalid url " + this.url);
        }
        if (!this.url.endsWith("gbfs.json")) {
            LOG.warn("GBFS autoconfiguration url {} does not end with gbfs.json. Make sure it follows the specification, if you get any errors using it.", (Object)this.url);
        }
        if (this.authenticateRequest()) {
            this.rawDiscoveryFileData = GbfsLoader.fetchFeed(uri, this.httpHeaders, this.timeoutConnection);
            try {
                if (this.rawDiscoveryFileData != null) {
                    this.disoveryFileData = (GBFS)objectMapper.readValue(this.rawDiscoveryFileData, GBFS.class);
                }
            }
            catch (IOException e) {
                LOG.warn("Error unmarshalling discovery feed", (Throwable)e);
            }
            if (this.disoveryFileData != null) {
                this.createUpdaters();
                this.setupComplete.set(true);
            } else {
                LOG.warn("Could not fetch the feed auto-configuration file from {}", (Object)uri);
            }
        }
    }

    private void createUpdaters() {
        GBFSFeeds feeds;
        GBFSFeeds gBFSFeeds = feeds = this.languageCode == null ? (GBFSFeeds)this.disoveryFileData.getFeedsData().values().iterator().next() : (GBFSFeeds)this.disoveryFileData.getFeedsData().get(this.languageCode);
        if (feeds == null) {
            throw new RuntimeException("Language " + this.languageCode + " does not exist in feed " + this.url);
        }
        for (GBFSFeed feed : feeds.getFeeds()) {
            GBFSFeedName feedName = feed.getName();
            if (this.feedUpdaters.containsKey(feedName)) {
                throw new RuntimeException("Feed contains duplicate url for feed " + feedName + ". Urls: " + feed.getUrl() + ", " + this.feedUpdaters.get((Object)feedName).url);
            }
            if (feed.getName() == null) continue;
            this.feedUpdaters.put(feedName, new GBFSFeedUpdater(feed));
        }
    }

    private boolean authenticateRequest() {
        try {
            this.requestAuthenticator.authenticateRequest(this.httpHeaders);
            return true;
        }
        catch (RequestAuthenticationException e) {
            LOG.warn("Unable to authenticate request: {}", (Object)e.getMessage());
            return false;
        }
    }

    public boolean update() {
        if (!this.setupComplete.get()) {
            this.init();
        }
        boolean didUpdate = false;
        if (this.updateLock.tryLock()) {
            for (GBFSFeedUpdater<?> updater : this.feedUpdaters.values()) {
                if (!updater.shouldUpdate()) continue;
                updater.fetchData();
                didUpdate = true;
            }
            this.updateLock.unlock();
        }
        return didUpdate;
    }

    public GBFS getDiscoveryFeed() {
        return this.disoveryFileData;
    }

    public <T> T getFeed(Class<T> feed) {
        GBFSFeedUpdater<?> updater = this.feedUpdaters.get(GBFSFeedName.fromClass(feed));
        if (updater == null) {
            return null;
        }
        return feed.cast(updater.getData());
    }

    public byte[] getRawFeed(GBFSFeedName feedName) {
        if (feedName.equals((Object)GBFSFeedName.GBFS)) {
            return this.rawDiscoveryFileData;
        }
        GBFSFeedUpdater<?> updater = this.feedUpdaters.get(feedName);
        if (updater == null) {
            return null;
        }
        return updater.getRawData();
    }

    private static byte[] fetchFeed(URI uri, Map<String, String> httpHeaders) {
        return GbfsLoader.fetchFeed(uri, httpHeaders, null);
    }

    private static byte[] fetchFeed(URI uri, Map<String, String> httpHeaders, Long timeout) {
        try {
            String proto = uri.getScheme();
            InputStream is = proto.equals("http") || proto.equals("https") ? HttpUtils.getData(uri, timeout, httpHeaders) : uri.toURL().openStream();
            if (is == null) {
                LOG.warn("Failed to get data from url {}", (Object)uri);
                return null;
            }
            byte[] asBytes = is.readAllBytes();
            is.close();
            return asBytes;
        }
        catch (IllegalArgumentException e) {
            LOG.warn("Error parsing GBFS feed from {}", (Object)uri, (Object)e);
            return null;
        }
        catch (JsonProcessingException e) {
            LOG.warn("Error parsing (bad JSON) GBFS feed from {}", (Object)uri, (Object)e);
            return null;
        }
        catch (IOException e) {
            LOG.warn("Error (bad connection) reading GBFS feed from {}", (Object)uri, (Object)e);
            return null;
        }
    }

    static {
        objectMapper.configure(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_AS_NULL, true);
    }

    private class GBFSFeedUpdater<T> {
        private final URI url;
        private final Class<T> implementingClass;
        private int nextUpdate;
        private T data;
        private byte[] rawData;

        private GBFSFeedUpdater(GBFSFeed feed) {
            this.url = feed.getUrl();
            this.implementingClass = feed.getName().implementingClass();
        }

        private T getData() {
            return this.data;
        }

        public byte[] getRawData() {
            return this.rawData;
        }

        private void fetchData() {
            if (!GbfsLoader.this.authenticateRequest()) {
                return;
            }
            this.rawData = GbfsLoader.fetchFeed(this.url, GbfsLoader.this.httpHeaders);
            if (this.rawData == null) {
                LOG.warn("Invalid data for {}", (Object)this.url);
                this.nextUpdate = this.getCurrentTimeSeconds();
                this.data = null;
                return;
            }
            try {
                this.data = objectMapper.readValue(this.rawData, this.implementingClass);
            }
            catch (IOException e) {
                LOG.warn("Error unmarshalling feed", (Throwable)e);
                this.data = null;
                return;
            }
            try {
                Integer lastUpdated = (Integer)this.implementingClass.getMethod("getLastUpdated", new Class[0]).invoke(this.data, new Object[0]);
                Integer ttl = (Integer)this.implementingClass.getMethod("getTtl", new Class[0]).invoke(this.data, new Object[0]);
                this.nextUpdate = lastUpdated == null || ttl == null ? this.getCurrentTimeSeconds() : lastUpdated + ttl;
            }
            catch (ClassCastException | IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
                LOG.warn("Invalid data for {}", (Object)this.url);
                this.nextUpdate = this.getCurrentTimeSeconds();
            }
        }

        private boolean shouldUpdate() {
            return this.getCurrentTimeSeconds() >= this.nextUpdate;
        }

        private int getCurrentTimeSeconds() {
            return (int)(System.currentTimeMillis() / 1000L);
        }
    }
}

