/*
 * Copyright (c) 1998, 2024 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
//     Oracle - initial API and implementation from Oracle TopLink
package org.eclipse.persistence.descriptors.copying;

import org.eclipse.persistence.descriptors.ClassDescriptor;
import org.eclipse.persistence.exceptions.DescriptorException;
import org.eclipse.persistence.queries.ObjectBuildingQuery;
import org.eclipse.persistence.sessions.DataRecord;
import org.eclipse.persistence.sessions.Session;
import org.eclipse.persistence.sessions.UnitOfWork;

import java.io.Serializable;

/**
 * <p><b>Purpose</b>: Allows customization of how an object is cloned.
 * An implementer of {@code CopyPolicy} can be set on a descriptor to provide
 * special cloning routine for how an object is cloned in a unit of work.
 * <p>
 * By default, the {@linkplain InstantiationCopyPolicy} is used which creates a new instance of
 * the class to be copied into.
 * <p>
 * The {@linkplain CloneCopyPolicy} can also be used that uses a clone method in the object
 * to clone the object.  When a clone method is used it avoid the requirement of having to
 * copy over each of the direct attributes.
 *
 * @see CloneCopyPolicy
 * @see InstantiationCopyPolicy
 */
public interface CopyPolicy extends Cloneable, Serializable {

    /**
     * Return a shallow clone of the object for usage with object copying, or unit of work backup cloning.
     */
    Object buildClone(Object object, Session session) throws DescriptorException;

    /**
     * Return a shallow clone of the object for usage with the unit of work working copy.
     */
    Object buildWorkingCopyClone(Object object, Session session) throws DescriptorException;

    /**
     * Return an instance with the primary key set from the row, used for building a working copy during a unit of work transactional read.
     */
    Object buildWorkingCopyCloneFromRow(DataRecord row, ObjectBuildingQuery query, Object primaryKey, UnitOfWork uow) throws DescriptorException;

    /**
     * Clone the CopyPolicy.
     */
    Object clone();

    /**
     * Allow for any initialization or validation required.
     */
    void initialize(Session session) throws DescriptorException;

    /**
     * Set the descriptor.
     */
    void setDescriptor(ClassDescriptor descriptor);

    /**
     * Return if this copy policy creates a new instance, vs a clone.
     */
    boolean buildsNewInstance();
}
