/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.util.validation;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.CertPath;
import java.security.cert.CertPathBuilder;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertSelector;
import java.security.cert.CertStore;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertificateParsingException;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.PKIXCertPathBuilderResult;
import java.security.cert.PKIXParameters;
import java.security.cert.PKIXRevocationChecker;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CRL;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.cert.X509CertificateHolder;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.util.validation.OpcUaCertificateRevocationChecker;
import org.eclipse.milo.opcua.stack.core.util.validation.OpcUaCertificateUsageChecker;
import org.eclipse.milo.opcua.stack.core.util.validation.ValidationCheck;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertificateValidationUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(CertificateValidationUtil.class);
    private static final String KEY_USAGE_OID = "2.5.29.15";
    private static final String EXTENDED_KEY_USAGE_OID = "2.5.29.37";
    private static final String SERVER_AUTH_OID = "1.3.6.1.5.5.7.3.1";
    private static final String CLIENT_AUTH_OID = "1.3.6.1.5.5.7.3.2";
    private static final int SUBJECT_ALT_NAME_URI = 6;
    private static final int SUBJECT_ALT_NAME_DNS_NAME = 2;
    private static final int SUBJECT_ALT_NAME_IP_ADDRESS = 7;

    public static PKIXCertPathBuilderResult buildTrustedCertPath(List<X509Certificate> certificateChain, Collection<X509Certificate> trustedCertificates, Collection<X509Certificate> issuerCertificates) throws UaException {
        Preconditions.checkArgument(!certificateChain.isEmpty(), "certificateChain must not be empty");
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("certificateChain: {}", (Object)certificateChain);
            LOGGER.trace("trustedCertificates: {}", (Object)trustedCertificates);
            LOGGER.trace("issuerCertificates: {}", (Object)issuerCertificates);
        }
        HashSet<TrustAnchor> trustAnchors = new HashSet<TrustAnchor>();
        for (X509Certificate c2 : trustedCertificates) {
            if (!CertificateValidationUtil.certificateIsSelfSigned(c2)) continue;
            if (certificateChain.size() == 1 && certificateChain.get(0).equals(c2)) {
                trustAnchors.add(new TrustAnchor(c2, null));
                if (c2.getBasicConstraints() == -1) continue;
                LOGGER.warn("self-signed certificate '{}' has BasicConstraint cA bit set", (Object)c2.getSubjectX500Principal().getName());
                continue;
            }
            if (!CertificateValidationUtil.certificateIsCa(c2)) continue;
            trustAnchors.add(new TrustAnchor(c2, null));
        }
        for (X509Certificate c2 : issuerCertificates) {
            if (!CertificateValidationUtil.certificateIsCa(c2) || !CertificateValidationUtil.certificateIsSelfSigned(c2)) continue;
            trustAnchors.add(new TrustAnchor(c2, null));
        }
        PKIXCertPathBuilderResult certPathResult = CertificateValidationUtil.buildCertPath(certificateChain, trustedCertificates, issuerCertificates, trustAnchors);
        CertPath certPath = certPathResult.getCertPath();
        TrustAnchor trustAnchor = certPathResult.getTrustAnchor();
        ArrayList<X509Certificate> certificates = new ArrayList<X509Certificate>();
        certPath.getCertificates().stream().map(X509Certificate.class::cast).forEach(certificates::add);
        certificates.add(trustAnchor.getTrustedCert());
        if (LOGGER.isTraceEnabled()) {
            List path = certificates.stream().map(c -> c.getSubjectX500Principal().getName()).collect(Collectors.toList());
            LOGGER.trace("certificate path: {}", (Object)path);
        }
        if (certificates.stream().noneMatch(trustedCertificates::contains)) {
            throw new UaException(2149187584L, "certificate chain did not contain a trusted certificate");
        }
        return certPathResult;
    }

    public static void validateTrustedCertPath(CertPath certPath, TrustAnchor trustAnchor, Collection<X509CRL> crls, Set<ValidationCheck> validationChecks, boolean endEntityIsClient) throws UaException {
        X509Certificate anchorCert = trustAnchor.getTrustedCert();
        boolean anchorIsEndEntity = certPath.getCertificates().isEmpty();
        CertificateValidationUtil.checkAnchorValidity(anchorCert, validationChecks, anchorIsEndEntity, endEntityIsClient);
        if (!anchorIsEndEntity) {
            try {
                CertPathValidator certPathValidator = CertPathValidator.getInstance("PKIX", "SUN");
                PKIXParameters parameters = new PKIXParameters(Sets.newHashSet(trustAnchor));
                parameters.addCertPathChecker(new OpcUaCertificateUsageChecker(certPath, validationChecks, endEntityIsClient));
                try {
                    parameters.setRevocationEnabled(true);
                    if (!crls.isEmpty()) {
                        parameters.addCertStore(CertStore.getInstance("Collection", new CollectionCertStoreParameters(crls)));
                    }
                    parameters.addCertPathChecker(new OpcUaCertificateRevocationChecker(certPath, trustAnchor, parameters, validationChecks));
                }
                catch (Exception e) {
                    if (validationChecks.contains((Object)ValidationCheck.REVOCATION)) {
                        parameters.setRevocationEnabled(true);
                        PKIXRevocationChecker pkixRevocationChecker = (PKIXRevocationChecker)certPathValidator.getRevocationChecker();
                        pkixRevocationChecker.setOptions(Sets.newHashSet(PKIXRevocationChecker.Option.NO_FALLBACK, PKIXRevocationChecker.Option.PREFER_CRLS, PKIXRevocationChecker.Option.SOFT_FAIL));
                    } else {
                        parameters.setRevocationEnabled(false);
                    }
                    LOGGER.warn("Failed to add custom revocation checker; REVOCATION_LIST_FOUND check will be ignored.");
                }
                certPathValidator.validate(certPath, parameters);
            }
            catch (CertPathValidatorException e) {
                CertPath path = e.getCertPath();
                CertPathValidatorException.Reason reason = e.getReason();
                int failedAtIndex = e.getIndex();
                if (failedAtIndex < 0) {
                    throw new UaException(2148728832L, (Throwable)e);
                }
                X509Certificate failed = (X509Certificate)path.getCertificates().get(failedAtIndex);
                LOGGER.debug("cert path validation failed at index={} reason={}, certificate={}", failedAtIndex, reason, failed.getSubjectX500Principal().getName());
                if (reason == CertPathValidatorException.BasicReason.REVOKED) {
                    if (failedAtIndex == 0) {
                        throw new UaException(2149384192L, (Throwable)e);
                    }
                    throw new UaException(2149449728L, (Throwable)e);
                }
                if (reason == CertPathValidatorException.BasicReason.UNDETERMINED_REVOCATION_STATUS) {
                    if (failedAtIndex == 0) {
                        throw new UaException(2149253120L, (Throwable)e);
                    }
                    throw new UaException(2149318656L, (Throwable)e);
                }
                if (reason == CertPathValidatorException.BasicReason.EXPIRED || reason == CertPathValidatorException.BasicReason.NOT_YET_VALID) {
                    if (failedAtIndex == 0) {
                        throw new UaException(2148794368L, (Throwable)e);
                    }
                    throw new UaException(2148859904L, (Throwable)e);
                }
                throw new UaException(2148728832L, (Throwable)e);
            }
            catch (GeneralSecurityException e) {
                throw new UaException(2148728832L, (Throwable)e);
            }
        }
    }

    private static void checkAnchorValidity(X509Certificate anchorCert, Set<ValidationCheck> validationChecks, boolean endEntity, boolean endEntityIsClient) throws UaException {
        Set<String> criticalExtensions = anchorCert.getCriticalExtensionOIDs();
        if (criticalExtensions == null) {
            criticalExtensions = Collections.emptySet();
        }
        try {
            CertificateValidationUtil.checkValidity(anchorCert, endEntity);
        }
        catch (UaException e) {
            if (validationChecks.contains((Object)ValidationCheck.VALIDITY)) {
                throw e;
            }
            LOGGER.warn("check suppressed: certificate failed end-entity validity check: {}", (Object)anchorCert.getSubjectX500Principal().getName());
        }
        if (endEntity) {
            try {
                CertificateValidationUtil.checkEndEntityKeyUsage(anchorCert);
            }
            catch (UaException e) {
                if (validationChecks.contains((Object)ValidationCheck.KEY_USAGE_END_ENTITY) || criticalExtensions.contains(KEY_USAGE_OID)) {
                    throw e;
                }
                LOGGER.warn("check suppressed: certificate failed end-entity KeyUsage check: {}", (Object)anchorCert.getSubjectX500Principal().getName());
            }
            try {
                CertificateValidationUtil.checkEndEntityExtendedKeyUsage(anchorCert, endEntityIsClient);
            }
            catch (UaException e) {
                if (validationChecks.contains((Object)ValidationCheck.EXTENDED_KEY_USAGE_END_ENTITY) || criticalExtensions.contains(EXTENDED_KEY_USAGE_OID)) {
                    throw e;
                }
                LOGGER.warn("check suppressed: certificate failed end-entity ExtendedKeyUsage check: {}", (Object)anchorCert.getSubjectX500Principal().getName());
            }
        }
    }

    private static PKIXCertPathBuilderResult buildCertPath(List<X509Certificate> certificateChain, Collection<X509Certificate> trustedCertificates, Collection<X509Certificate> issuerCertificates, Set<TrustAnchor> trustAnchors) throws UaException {
        X509Certificate certificate = certificateChain.get(0);
        X509CertSelector selector = new X509CertSelector();
        selector.setCertificate(certificate);
        try {
            PKIXBuilderParameters builderParams = new PKIXBuilderParameters(trustAnchors, (CertSelector)selector);
            ArrayList<X509Certificate> intermediates = Lists.newArrayList();
            intermediates.addAll(certificateChain.subList(1, certificateChain.size()));
            for (X509Certificate c : trustedCertificates) {
                if (!CertificateValidationUtil.certificateIsCa(c) || CertificateValidationUtil.certificateIsSelfSigned(c)) continue;
                intermediates.add(c);
            }
            for (X509Certificate c : issuerCertificates) {
                if (!CertificateValidationUtil.certificateIsCa(c) || CertificateValidationUtil.certificateIsSelfSigned(c)) continue;
                intermediates.add(c);
            }
            if (intermediates.size() > 0) {
                CertStore certStore = CertStore.getInstance("Collection", new CollectionCertStoreParameters(intermediates));
                builderParams.addCertStore(certStore);
            }
            builderParams.setRevocationEnabled(false);
            CertPathBuilder builder = CertPathBuilder.getInstance("PKIX");
            return (PKIXCertPathBuilderResult)builder.build(builderParams);
        }
        catch (GeneralSecurityException e) {
            throw new UaException(2148728832L, (Throwable)e);
        }
    }

    static boolean certificateIsCa(X509Certificate certificate) {
        boolean[] keyUsage = certificate.getKeyUsage();
        int basicConstraints = certificate.getBasicConstraints();
        if (keyUsage == null) {
            return basicConstraints >= 0;
        }
        if (keyUsage[5] && basicConstraints == -1) {
            LOGGER.debug("'{}' violates RFC 5280: KeyUsage keyCertSign bit set without BasicConstraint cA bit set", (Object)certificate.getSubjectX500Principal().getName());
        }
        return keyUsage[5] || basicConstraints >= 0;
    }

    private static boolean certificateIsSelfSigned(X509Certificate cert) throws UaException {
        try {
            PublicKey key = cert.getPublicKey();
            cert.verify(key);
            return Objects.equals(cert.getSubjectX500Principal(), cert.getIssuerX500Principal());
        }
        catch (InvalidKeyException | SignatureException sigEx) {
            return false;
        }
        catch (Exception e) {
            throw new UaException(2148663296L, (Throwable)e);
        }
    }

    public static void checkValidity(X509Certificate certificate, boolean endEntity) throws UaException {
        try {
            certificate.checkValidity();
        }
        catch (CertificateExpiredException e) {
            throw new UaException(endEntity ? 2148794368L : 2148859904L, String.format("certificate is expired: %s - %s", certificate.getNotBefore(), certificate.getNotAfter()));
        }
        catch (CertificateNotYetValidException e) {
            throw new UaException(endEntity ? 2148794368L : 2148859904L, String.format("certificate not yet valid: %s - %s", certificate.getNotBefore(), certificate.getNotAfter()));
        }
    }

    public static void checkHostnameOrIpAddress(X509Certificate certificate, String ... hostNames) throws UaException {
        boolean dnsNameMatch = Arrays.stream(hostNames).anyMatch(n -> {
            try {
                return CertificateValidationUtil.checkSubjectAltNameField(certificate, 2, n::equals);
            }
            catch (Throwable t) {
                return false;
            }
        });
        boolean ipAddressMatch = Arrays.stream(hostNames).anyMatch(n -> {
            try {
                return CertificateValidationUtil.checkSubjectAltNameField(certificate, 7, n::equals);
            }
            catch (Throwable t) {
                return false;
            }
        });
        if (!dnsNameMatch && !ipAddressMatch) {
            throw new UaException(2148925440L);
        }
    }

    public static void checkEndEntityKeyUsage(X509Certificate certificate) throws UaException {
        boolean[] keyUsage = certificate.getKeyUsage();
        if (keyUsage == null) {
            throw new UaException(0x80180000L, "KeyUsage extension not found");
        }
        boolean digitalSignature = keyUsage[0];
        boolean nonRepudiation = keyUsage[1];
        boolean keyEncipherment = keyUsage[2];
        boolean dataEncipherment = keyUsage[3];
        boolean keyCertSign = keyUsage[5];
        if (!digitalSignature) {
            throw new UaException(0x80180000L, "required KeyUsage 'digitalSignature' not found");
        }
        if (!nonRepudiation) {
            throw new UaException(0x80180000L, "required KeyUsage 'nonRepudiation' not found");
        }
        if (!keyEncipherment) {
            throw new UaException(0x80180000L, "required KeyUsage 'keyEncipherment' not found");
        }
        if (!dataEncipherment) {
            throw new UaException(0x80180000L, "required KeyUsage 'dataEncipherment' not found");
        }
        if (!keyCertSign && CertificateValidationUtil.certificateIsSelfSigned(certificate)) {
            throw new UaException(0x80180000L, "required KeyUsage 'keyCertSign' not found");
        }
    }

    public static void checkEndEntityExtendedKeyUsage(X509Certificate certificate, boolean endEntityIsClient) throws UaException {
        try {
            List<String> extendedKeyUsage = certificate.getExtendedKeyUsage();
            if (extendedKeyUsage == null) {
                throw new UaException(0x80180000L, "ExtendedKeyUsage extension not found");
            }
            if (endEntityIsClient && !extendedKeyUsage.contains(CLIENT_AUTH_OID)) {
                throw new UaException(0x80180000L, "required ExtendedKeyUsage 'clientAuth' not found");
            }
            if (!endEntityIsClient && !extendedKeyUsage.contains(SERVER_AUTH_OID)) {
                throw new UaException(0x80180000L, "required ExtendedKeyUsage 'serverAuth' not found");
            }
        }
        catch (CertificateParsingException e) {
            throw new UaException(0x80180000L);
        }
    }

    public static void checkApplicationUri(X509Certificate certificate, String applicationUri) throws UaException {
        if (!CertificateValidationUtil.checkSubjectAltNameField(certificate, 6, applicationUri::equals)) {
            try {
                X509CertificateHolder certificateHolder = new X509CertificateHolder(certificate.getEncoded());
                GeneralNames generalNames = GeneralNames.fromExtensions(certificateHolder.getExtensions(), Extension.subjectAlternativeName);
                if (generalNames != null) {
                    for (GeneralName generalName : generalNames.getNames()) {
                        if (generalName.getTagNo() != 6) continue;
                        String uri = generalName.getName().toString();
                        if (!Objects.equals(applicationUri, uri)) {
                            throw new UaException(2148990976L);
                        }
                        return;
                    }
                }
                throw new UaException(2148990976L, "no match in certificate for application URI '" + applicationUri + "'");
            }
            catch (IOException | CertificateEncodingException e) {
                throw new UaException(2148990976L, (Throwable)e);
            }
        }
    }

    private static boolean checkSubjectAltNameField(X509Certificate certificate, int field, Predicate<Object> fieldPredicate) throws UaException {
        try {
            Collection<List<?>> subjectAltNames = certificate.getSubjectAlternativeNames();
            if (subjectAltNames == null) {
                subjectAltNames = Collections.emptyList();
            }
            for (List<?> idAndValue : subjectAltNames) {
                if (idAndValue == null || idAndValue.size() != 2 || !idAndValue.get(0).equals(field) || !fieldPredicate.test(idAndValue.get(1))) continue;
                return true;
            }
            return false;
        }
        catch (CertificateParsingException e) {
            throw new UaException(2148663296L, (Throwable)e);
        }
    }

    public static String getSubjectAltNameUri(X509Certificate certificate) throws UaException {
        try {
            Collection<List<?>> subjectAltNames = certificate.getSubjectAlternativeNames();
            if (subjectAltNames == null) {
                subjectAltNames = Collections.emptyList();
            }
            for (List<?> idAndValue : subjectAltNames) {
                if (idAndValue == null || idAndValue.size() != 2 || !idAndValue.get(0).equals(6)) continue;
                Object uri = idAndValue.get(1);
                return uri != null ? uri.toString() : null;
            }
            return null;
        }
        catch (CertificateParsingException e) {
            throw new UaException(2148663296L, (Throwable)e);
        }
    }
}

