/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.annotations;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.eclipse.jetty.util.JavaVersion;
import org.eclipse.jetty.util.Loader;
import org.eclipse.jetty.util.ManifestUtils;
import org.eclipse.jetty.util.MultiException;
import org.eclipse.jetty.util.MultiReleaseJarFile;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.TypeUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.resource.Resource;
import org.objectweb.asm.AnnotationVisitor;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.FieldVisitor;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Opcodes;

public class AnnotationParser {
    private static final Logger LOG = Log.getLogger(AnnotationParser.class);
    protected static int ASM_OPCODE_VERSION = 458752;
    protected static String ASM_OPCODE_VERSION_STR = "ASM7";
    protected Map<String, Resource> _parsedClassNames = new ConcurrentHashMap<String, Resource>();
    private final int _javaPlatform;
    private int _asmVersion;

    public static int asmVersion() {
        int asmVersion = ASM_OPCODE_VERSION;
        String version = ManifestUtils.getVersion(Opcodes.class).orElse(null);
        if (version == null) {
            LOG.warn("Unknown ASM version, assuming {}", new Object[]{ASM_OPCODE_VERSION_STR});
        } else {
            int dot = version.indexOf(46);
            version = version.substring(0, dot < 0 ? version.length() : dot).trim();
            try {
                int v = Integer.parseInt(version);
                switch (v) {
                    case 4: {
                        asmVersion = 262144;
                        break;
                    }
                    case 5: {
                        asmVersion = 327680;
                        break;
                    }
                    case 6: {
                        asmVersion = 393216;
                        break;
                    }
                    case 7: {
                        asmVersion = 458752;
                        break;
                    }
                    default: {
                        LOG.warn("Unrecognized ASM version, assuming {}", new Object[]{ASM_OPCODE_VERSION_STR});
                        break;
                    }
                }
            }
            catch (NumberFormatException e) {
                LOG.warn("Unable to parse ASM version, assuming {}", new Object[]{ASM_OPCODE_VERSION_STR});
            }
        }
        return asmVersion;
    }

    public static String normalize(String name) {
        if (name == null) {
            return null;
        }
        if (name.startsWith("L") && name.endsWith(";")) {
            name = name.substring(1, name.length() - 1);
        }
        if (name.endsWith(".class")) {
            name = name.substring(0, name.length() - ".class".length());
        }
        return StringUtil.replace((String)name, (char)'/', (char)'.');
    }

    public static String[] normalize(String[] list) {
        if (list == null) {
            return null;
        }
        String[] normalList = new String[list.length];
        int i = 0;
        for (String s : list) {
            normalList[i++] = AnnotationParser.normalize(s);
        }
        return normalList;
    }

    public AnnotationParser() {
        this(JavaVersion.VERSION.getPlatform());
    }

    public AnnotationParser(int javaPlatform) {
        this._asmVersion = AnnotationParser.asmVersion();
        if (javaPlatform == 0) {
            javaPlatform = JavaVersion.VERSION.getPlatform();
        }
        this._javaPlatform = javaPlatform;
    }

    public AnnotationParser(int javaPlatform, int asmVersion) {
        if (javaPlatform == 0) {
            javaPlatform = JavaVersion.VERSION.getPlatform();
        }
        this._javaPlatform = javaPlatform;
        if (asmVersion == 0) {
            asmVersion = ASM_OPCODE_VERSION;
        }
        this._asmVersion = asmVersion;
    }

    public void addParsedClass(String classname, Resource location) {
        Resource existing = this._parsedClassNames.putIfAbsent(classname, location);
        if (existing != null) {
            LOG.warn("{} scanned from multiple locations: {}, {}", new Object[]{classname, existing, location});
        }
    }

    public void parse(Set<? extends Handler> handlers, String className) throws Exception {
        if (className == null) {
            return;
        }
        String classRef = TypeUtil.toClassReference((String)className);
        URL resource = Loader.getResource((String)classRef);
        if (resource != null) {
            Resource r = Resource.newResource((URL)resource);
            this.addParsedClass(className, r);
            try (InputStream is = r.getInputStream();){
                this.scanClass(handlers, null, is);
            }
        }
    }

    public void parse(Set<? extends Handler> handlers, Class<?> clazz, boolean visitSuperClasses) throws Exception {
        for (Class<?> cz = clazz; cz != Object.class; cz = cz.getSuperclass()) {
            String nameAsResource = TypeUtil.toClassReference(cz);
            URL resource = Loader.getResource((String)nameAsResource);
            if (resource != null) {
                Resource r = Resource.newResource((URL)resource);
                this.addParsedClass(clazz.getName(), r);
                try (InputStream is = r.getInputStream();){
                    this.scanClass(handlers, null, is);
                }
            }
            if (!visitSuperClasses) break;
        }
    }

    public void parse(Set<? extends Handler> handlers, String[] classNames) throws Exception {
        if (classNames == null) {
            return;
        }
        this.parse(handlers, Arrays.asList(classNames));
    }

    public void parse(Set<? extends Handler> handlers, List<String> classNames) throws Exception {
        MultiException me = new MultiException();
        for (String className : classNames) {
            try {
                String classRef = TypeUtil.toClassReference((String)className);
                URL resource = Loader.getResource((String)classRef);
                if (resource == null) continue;
                Resource r = Resource.newResource((URL)resource);
                this.addParsedClass(className, r);
                InputStream is = r.getInputStream();
                try {
                    this.scanClass(handlers, null, is);
                }
                finally {
                    if (is == null) continue;
                    is.close();
                }
            }
            catch (Exception e) {
                me.add((Throwable)new RuntimeException("Error scanning class " + className, e));
            }
        }
        me.ifExceptionThrow();
    }

    public void parse(Set<? extends Handler> handlers, URI[] uris) throws Exception {
        if (uris == null) {
            return;
        }
        MultiException me = new MultiException();
        for (URI uri : uris) {
            try {
                this.parse(handlers, uri);
            }
            catch (Exception e) {
                me.add((Throwable)new RuntimeException("Problem parsing classes from " + uri, e));
            }
        }
        me.ifExceptionThrow();
    }

    public void parse(Set<? extends Handler> handlers, URI uri) throws Exception {
        if (uri == null) {
            return;
        }
        this.parse(handlers, Resource.newResource((URI)uri));
    }

    public void parse(Set<? extends Handler> handlers, Resource r) throws Exception {
        if (r == null) {
            return;
        }
        if (r.exists() && r.isDirectory()) {
            this.parseDir(handlers, r);
            return;
        }
        String fullname = r.toString();
        if (fullname.endsWith(".jar")) {
            this.parseJar(handlers, r);
            return;
        }
        if (fullname.endsWith(".class")) {
            try (InputStream is = r.getInputStream();){
                this.scanClass(handlers, null, is);
                return;
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.warn("Resource not scannable for classes: {}", new Object[]{r});
        }
    }

    protected void parseDir(Set<? extends Handler> handlers, Resource root) throws Exception {
        if (!root.isDirectory() || !root.exists() || root.getName().startsWith(".")) {
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Scanning dir {}", new Object[]{root});
        }
        File rootFile = root.getFile();
        MultiException me = new MultiException();
        Collection resources = root.getAllResources();
        if (resources != null) {
            for (Resource r : resources) {
                if (r.isDirectory()) continue;
                File file = r.getFile();
                if (this.isValidClassFileName(file == null ? null : file.getName())) {
                    Path classpath = rootFile.toPath().relativize(file.toPath());
                    String str = classpath.toString();
                    str = str.substring(0, str.lastIndexOf(".class"));
                    str = StringUtil.replace((String)str, (char)File.separatorChar, (char)'.');
                    try {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Scanning class {}", new Object[]{r});
                        }
                        this.addParsedClass(str, r);
                        try (InputStream is = r.getInputStream();){
                            this.scanClass(handlers, Resource.newResource((File)file.getParentFile()), is);
                        }
                    }
                    catch (Exception ex) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Error scanning file " + file, (Throwable)ex);
                        }
                        me.add((Throwable)new RuntimeException("Error scanning file " + file, ex));
                    }
                    continue;
                }
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Skipping scan on invalid file {}", new Object[]{file});
            }
        }
        me.ifExceptionThrow();
    }

    protected void parseJar(Set<? extends Handler> handlers, Resource jarResource) throws Exception {
        if (jarResource == null) {
            return;
        }
        if (jarResource.toString().endsWith(".jar")) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Scanning jar {}", new Object[]{jarResource});
            }
            MultiException me = new MultiException();
            try (MultiReleaseJarFile jarFile = new MultiReleaseJarFile(jarResource.getFile(), this._javaPlatform, false);){
                jarFile.stream().forEach(e -> {
                    try {
                        this.parseJarEntry(handlers, jarResource, (MultiReleaseJarFile.VersionedJarEntry)e);
                    }
                    catch (Exception ex) {
                        me.add((Throwable)new RuntimeException("Error scanning entry " + e.getName() + " from jar " + jarResource, ex));
                    }
                });
            }
            me.ifExceptionThrow();
        }
    }

    protected void parseJarEntry(Set<? extends Handler> handlers, Resource jar, MultiReleaseJarFile.VersionedJarEntry entry) throws Exception {
        if (jar == null || entry == null) {
            return;
        }
        if (entry.isDirectory()) {
            return;
        }
        String name = entry.getName();
        if (this.isValidClassFileName(name) && this.isValidClassFilePath(name)) {
            String shortName = StringUtil.replace((String)name, (char)'/', (char)'.').substring(0, name.length() - 6);
            this.addParsedClass(shortName, Resource.newResource((String)("jar:" + jar.getURI() + "!/" + entry.getNameInJar())));
            if (LOG.isDebugEnabled()) {
                LOG.debug("Scanning class from jar {}!/{}", new Object[]{jar, entry});
            }
            try (InputStream is = entry.getInputStream();){
                this.scanClass(handlers, jar, is);
            }
        }
    }

    protected void scanClass(Set<? extends Handler> handlers, Resource containingResource, InputStream is) throws IOException {
        ClassReader reader = new ClassReader(is);
        reader.accept((ClassVisitor)new MyClassVisitor(handlers, containingResource, this._asmVersion), 7);
    }

    public void resetParsedClasses() {
        this._parsedClassNames.clear();
    }

    private boolean isValidClassFileName(String name) {
        if (name == null || name.length() == 0) {
            return false;
        }
        String lc = name.toLowerCase(Locale.ENGLISH);
        if (!lc.endsWith(".class")) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Not a class: {}", new Object[]{name});
            }
            return false;
        }
        if (lc.equals("module-info.class")) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Skipping module-info.class", new Object[0]);
            }
            return false;
        }
        int c0 = 0;
        int ldir = name.lastIndexOf(47, name.length() - 6);
        int n = c0 = ldir > -1 ? ldir + 1 : c0;
        if (!Character.isJavaIdentifierStart(name.charAt(c0))) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Not a java identifier: {}", new Object[]{name});
            }
            return false;
        }
        return true;
    }

    private boolean isValidClassFilePath(String path) {
        if (path == null || path.length() == 0) {
            return false;
        }
        if (path.startsWith(".") || path.contains("/.")) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Contains hidden dirs: " + path, new Object[0]);
            }
            return false;
        }
        return true;
    }

    public class MyClassVisitor
    extends ClassVisitor {
        int _asmVersion;
        final Resource _containingResource;
        final Set<? extends Handler> _handlers;
        ClassInfo _ci;

        public MyClassVisitor(Set<? extends Handler> handlers, Resource containingResource, int asmVersion) {
            super(asmVersion);
            this._asmVersion = asmVersion;
            this._handlers = handlers;
            this._containingResource = containingResource;
        }

        public void visit(int version, int access, String name, String signature, String superName, String[] interfaces) {
            this._ci = new ClassInfo(this._containingResource, AnnotationParser.normalize(name), version, access, signature, AnnotationParser.normalize(superName), AnnotationParser.normalize(interfaces));
            for (Handler handler : this._handlers) {
                handler.handle(this._ci);
            }
        }

        public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
            String annotationName = AnnotationParser.normalize(desc);
            for (Handler handler : this._handlers) {
                handler.handle(this._ci, annotationName);
            }
            return null;
        }

        public MethodVisitor visitMethod(int access, String name, String methodDesc, String signature, String[] exceptions) {
            return new MyMethodVisitor(this._handlers, this._ci, access, name, methodDesc, signature, exceptions, this._asmVersion);
        }

        public FieldVisitor visitField(int access, String fieldName, String fieldType, String signature, Object value) {
            return new MyFieldVisitor(this._handlers, this._ci, access, fieldName, fieldType, signature, value, this._asmVersion);
        }
    }

    public class MyFieldVisitor
    extends FieldVisitor {
        final FieldInfo _fieldInfo;
        final Set<? extends Handler> _handlers;

        public MyFieldVisitor(Set<? extends Handler> handlers, ClassInfo classInfo, int access, String fieldName, String fieldType, String signature, Object value, int asmVersion) {
            super(asmVersion);
            this._handlers = handlers;
            this._fieldInfo = new FieldInfo(classInfo, fieldName, access, fieldType, signature, value);
        }

        public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
            String annotationName = AnnotationParser.normalize(desc);
            for (Handler handler : this._handlers) {
                handler.handle(this._fieldInfo, annotationName);
            }
            return null;
        }
    }

    public class MyMethodVisitor
    extends MethodVisitor {
        final MethodInfo _mi;
        final Set<? extends Handler> _handlers;

        public MyMethodVisitor(Set<? extends Handler> handlers, ClassInfo classInfo, int access, String name, String methodDesc, String signature, String[] exceptions, int asmVersion) {
            super(asmVersion);
            this._handlers = handlers;
            this._mi = new MethodInfo(classInfo, name, access, methodDesc, signature, exceptions);
        }

        public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
            String annotationName = AnnotationParser.normalize(desc);
            for (Handler handler : this._handlers) {
                handler.handle(this._mi, annotationName);
            }
            return null;
        }
    }

    public static abstract class AbstractHandler
    implements Handler {
        @Override
        public void handle(ClassInfo classInfo) {
        }

        @Override
        public void handle(MethodInfo methodInfo) {
        }

        @Override
        public void handle(FieldInfo fieldInfo) {
        }

        @Override
        public void handle(ClassInfo info, String annotationName) {
        }

        @Override
        public void handle(MethodInfo info, String annotationName) {
        }

        @Override
        public void handle(FieldInfo info, String annotationName) {
        }
    }

    public static interface Handler {
        public void handle(ClassInfo var1);

        public void handle(MethodInfo var1);

        public void handle(FieldInfo var1);

        public void handle(ClassInfo var1, String var2);

        public void handle(MethodInfo var1, String var2);

        public void handle(FieldInfo var1, String var2);
    }

    public class FieldInfo {
        final ClassInfo _classInfo;
        final String _fieldName;
        final int _access;
        final String _fieldType;
        final String _signature;
        final Object _value;

        public FieldInfo(ClassInfo classInfo, String fieldName, int access, String fieldType, String signature, Object value) {
            this._classInfo = classInfo;
            this._fieldName = fieldName;
            this._access = access;
            this._fieldType = fieldType;
            this._signature = signature;
            this._value = value;
        }

        public ClassInfo getClassInfo() {
            return this._classInfo;
        }

        public String getFieldName() {
            return this._fieldName;
        }

        public int getAccess() {
            return this._access;
        }

        public String getFieldType() {
            return this._fieldType;
        }

        public String getSignature() {
            return this._signature;
        }

        public Object getValue() {
            return this._value;
        }
    }

    public class MethodInfo {
        final ClassInfo _classInfo;
        final String _methodName;
        final int _access;
        final String _desc;
        final String _signature;
        final String[] _exceptions;

        public MethodInfo(ClassInfo classInfo, String methodName, int access, String desc, String signature, String[] exceptions) {
            this._classInfo = classInfo;
            this._methodName = methodName;
            this._access = access;
            this._desc = desc;
            this._signature = signature;
            this._exceptions = exceptions;
        }

        public ClassInfo getClassInfo() {
            return this._classInfo;
        }

        public String getMethodName() {
            return this._methodName;
        }

        public int getAccess() {
            return this._access;
        }

        public String getDesc() {
            return this._desc;
        }

        public String getSignature() {
            return this._signature;
        }

        public String[] getExceptions() {
            return this._exceptions;
        }
    }

    public class ClassInfo {
        final Resource _containingResource;
        final String _className;
        final int _version;
        final int _access;
        final String _signature;
        final String _superName;
        final String[] _interfaces;

        public ClassInfo(Resource resource, String className, int version, int access, String signature, String superName, String[] interfaces) {
            this._containingResource = resource;
            this._className = className;
            this._version = version;
            this._access = access;
            this._signature = signature;
            this._superName = superName;
            this._interfaces = interfaces;
        }

        public String getClassName() {
            return this._className;
        }

        public int getVersion() {
            return this._version;
        }

        public int getAccess() {
            return this._access;
        }

        public String getSignature() {
            return this._signature;
        }

        public String getSuperName() {
            return this._superName;
        }

        public String[] getInterfaces() {
            return this._interfaces;
        }

        public Resource getContainingResource() {
            return this._containingResource;
        }
    }
}

