//
// ========================================================================
// Copyright (c) 1995 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
// which is available at https://www.apache.org/licenses/LICENSE-2.0.
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.ee11.websocket.jakarta.common.decoders;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Type;

import jakarta.websocket.Decoder;
import jakarta.websocket.EndpointConfig;
import org.eclipse.jetty.ee11.websocket.jakarta.common.InitException;
import org.eclipse.jetty.websocket.core.WebSocketComponents;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.eclipse.jetty.websocket.core.util.ReflectUtils.isAssignableFrom;

public class RegisteredDecoder
{
    private static final Logger LOG = LoggerFactory.getLogger(RegisteredDecoder.class);

    // The user supplied Decoder class
    public final Class<? extends Decoder> decoder;
    // The jakarta.websocket.Decoder.* type (eg: Decoder.Binary, Decoder.BinaryStream, Decoder.Text, Decoder.TextStream)
    public final Class<? extends Decoder> interfaceType;
    public final Type objectType;
    public final boolean primitive;
    public final EndpointConfig config;
    private final WebSocketComponents components;

    private Decoder instance;

    public RegisteredDecoder(Class<? extends Decoder> decoder, Class<? extends Decoder> interfaceType, Type objectType, EndpointConfig endpointConfig, WebSocketComponents components)
    {
        this(decoder, interfaceType, objectType, endpointConfig, components, false);
    }

    public RegisteredDecoder(Class<? extends Decoder> decoder, Class<? extends Decoder> interfaceType, Type objectType, EndpointConfig endpointConfig, WebSocketComponents components, boolean primitive)
    {
        this.decoder = decoder;
        this.interfaceType = interfaceType;
        this.objectType = objectType;
        this.primitive = primitive;
        this.config = endpointConfig;
        this.components = components;
    }

    public boolean implementsInterface(Class<? extends Decoder> type)
    {
        return interfaceType.isAssignableFrom(type);
    }

    public boolean isType(Type type)
    {
        return isAssignableFrom(objectType, type);
    }

    @SuppressWarnings("unchecked")
    public <T extends Decoder> T getInstance()
    {
        if (instance == null)
        {
            try
            {
                instance = components.getObjectFactory().createInstance(decoder);
                instance.init(config);
                return (T)instance;
            }
            catch (InstantiationException | IllegalAccessException | NoSuchMethodException | InvocationTargetException e)
            {
                throw new InitException("Unable to init Decoder for type:" + decoder.getName(), e);
            }
        }

        return (T)instance;
    }

    public void destroyInstance()
    {
        if (instance != null)
        {
            try
            {
                instance.destroy();
            }
            catch (Throwable t)
            {
                LOG.warn("Error destroying Decoder", t);
            }

            instance = null;
        }
    }

    @Override
    public String toString()
    {
        StringBuilder str = new StringBuilder();
        str.append(RegisteredDecoder.class.getSimpleName());
        str.append('[').append(decoder.getName());
        str.append(',').append(interfaceType.getName());
        str.append(',').append(objectType.getTypeName());
        if (primitive)
        {
            str.append(",PRIMITIVE");
        }
        str.append(']');
        return str.toString();
    }
}
