/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.epsilon.common.util.profiling;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryUsage;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalUnit;
import java.util.Collection;
import java.util.Collections;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import org.eclipse.epsilon.common.function.CheckedConsumer;
import org.eclipse.epsilon.common.function.CheckedFunction;
import org.eclipse.epsilon.common.function.CheckedRunnable;
import org.eclipse.epsilon.common.function.CheckedSupplier;
import org.eclipse.epsilon.common.util.OperatingSystem;
import org.eclipse.epsilon.common.util.profiling.ProfileDiagnostic;

public final class BenchmarkUtils {
    public static final String GC_PROFILE_STAGE = "GARBAGE_COLLECTION";
    public static final ProfileDiagnostic.MemoryUnit DEFAULT_MEMORY_UNITS = ProfileDiagnostic.MemoryUnit.MB;
    public static final TemporalUnit DEFAULT_TIME_UNITS = ChronoUnit.NANOS;
    static final Runtime RT = Runtime.getRuntime();

    private BenchmarkUtils() {
    }

    static String formatExecutionStages(Iterable<ProfileDiagnostic> profileInfo, boolean includeTime, Optional<ProfileDiagnostic.MemoryUnit> conversionFactor) {
        StringBuilder formatted = new StringBuilder();
        for (ProfileDiagnostic pd : profileInfo) {
            formatted.append(String.valueOf(pd.stageName) + ": ");
            if (includeTime) {
                formatted.append(String.valueOf(BenchmarkUtils.formatDuration(pd.executionTime)) + " (" + pd.executionTime.toMillis() + " ms)");
            }
            if (conversionFactor != null) {
                if (includeTime) {
                    formatted.append(", ");
                }
                formatted.append(BenchmarkUtils.formatMemory(pd, conversionFactor));
            }
            formatted.append(System.lineSeparator());
        }
        return formatted.toString();
    }

    public static String formatDuration(Duration duration) {
        String pattern = "";
        if (duration.toHours() >= 1L) {
            pattern = String.valueOf(pattern) + "H:";
        }
        if (duration.toMinutes() >= 1L) {
            pattern = String.valueOf(pattern) + "mm:";
        }
        if (duration.getSeconds() >= 1L) {
            pattern = String.valueOf(pattern) + "ss.";
        }
        pattern = String.valueOf(pattern) + "SSS";
        return LocalTime.MIDNIGHT.plus(duration).format(DateTimeFormatter.ofPattern(pattern));
    }

    public static Duration getTotalExecutionTimeFrom(Collection<ProfileDiagnostic> profiledStages) {
        return Duration.ofNanos(profiledStages.stream().mapToLong(pd -> pd.executionTime.toNanos()).sum());
    }

    public static long getTotalMemoryUsage() {
        return ManagementFactory.getMemoryPoolMXBeans().stream().map(MemoryPoolMXBean::getPeakUsage).filter(usage -> usage != null).mapToLong(MemoryUsage::getUsed).sum();
    }

    public static long getCurrentMemoryUsage() {
        return RT.totalMemory() - RT.freeMemory();
    }

    public static String getTime() {
        return LocalDateTime.now().format(DateTimeFormatter.ofLocalizedDateTime(FormatStyle.MEDIUM));
    }

    public static String formatMemory(ProfileDiagnostic pd, Optional<ProfileDiagnostic.MemoryUnit> conversionFactor) {
        String formatted;
        if (conversionFactor.isPresent()) {
            ProfileDiagnostic.MemoryUnit cf = conversionFactor.get();
            formatted = String.valueOf((long)ProfileDiagnostic.MemoryUnit.convertUnits(pd.memoryUnits, cf, pd.memoryUsage)) + " " + (Object)((Object)cf);
        } else {
            formatted = String.valueOf(pd.memoryUsage) + " " + (Object)((Object)pd.memoryUnits);
        }
        return formatted;
    }

    public static String getAvailableMemory(ProfileDiagnostic.MemoryUnit units) {
        return BenchmarkUtils.formatMemory(RT.totalMemory(), units);
    }

    public static String getMaxMemory(ProfileDiagnostic.MemoryUnit units) {
        return BenchmarkUtils.formatMemory(RT.maxMemory(), units);
    }

    public static String formatMemory(long amountInBytes, ProfileDiagnostic.MemoryUnit units) {
        ProfileDiagnostic.MemoryUnit mu = units != null ? units : DEFAULT_MEMORY_UNITS;
        return String.valueOf((long)ProfileDiagnostic.MemoryUnit.convertFromBytes(mu, amountInBytes)) + " " + mu.name();
    }

    public static String getCpuName() {
        try {
            switch (OperatingSystem.getOSFamily()) {
                case WINDOWS: {
                    return OperatingSystem.executeCommand("powershell.exe", "-Command", "\"(wmic CPU get NAME)[2]\"");
                }
                case MAC: {
                    return OperatingSystem.executeCommand("/bin/sh", "-c", "sysctl -n machdep.cpu.brand_string");
                }
            }
            return OperatingSystem.executeCommand("/bin/sh", "-c", "cat /proc/cpuinfo | grep -m 1 'model name' | cut -c 14-");
        }
        catch (Exception ex) {
            System.err.println("Could not get CPU name: " + ex.getMessage());
            return "";
        }
    }

    public static int getNumberOfHardwareThreads() {
        return RT.availableProcessors();
    }

    public static ProfileDiagnostic getProfileStageByName(Collection<ProfileDiagnostic> profileInfo, String stageName) {
        return profileInfo.stream().filter(pd -> pd.stageName.equals(stageName)).findAny().orElse(null);
    }

    public static ProfileDiagnostic removeGCTimeFromStage(Collection<ProfileDiagnostic> profileInfo, String stageName) {
        ProfileDiagnostic gc = BenchmarkUtils.getProfileStageByName(profileInfo, GC_PROFILE_STAGE);
        ProfileDiagnostic target = BenchmarkUtils.getProfileStageByName(profileInfo, stageName);
        if (gc == null || target == null) {
            return target;
        }
        ProfileDiagnostic updated = new ProfileDiagnostic(stageName, target.executionTime.minus(gc.executionTime), target.memoryUsage, target.memoryUnits);
        profileInfo.remove(target);
        profileInfo.add(updated);
        return updated;
    }

    public static long measureAndAddGCTime(Collection<ProfileDiagnostic> profileInfo) {
        ProfileDiagnostic existingGCStage = BenchmarkUtils.getProfileStageByName(profileInfo, GC_PROFILE_STAGE);
        long gcTime = BenchmarkUtils.measureTimeNanos(RT::gc);
        if (existingGCStage != null) {
            gcTime += existingGCStage.executionTime.toNanos();
            profileInfo.remove(existingGCStage);
        }
        profileInfo.add(new ProfileDiagnostic(GC_PROFILE_STAGE, gcTime, 0L));
        return gcTime;
    }

    public static <T, R, E extends Exception> R profileExecutionStage(Collection<ProfileDiagnostic> profileInfo, String description, CheckedFunction<T, R, E> code, T argument) throws E {
        BenchmarkUtils.measureAndAddGCTime(profileInfo);
        long startMemory = BenchmarkUtils.getCurrentMemoryUsage();
        long startTime = System.nanoTime();
        R result = code.applyThrows(argument);
        long endTime = System.nanoTime();
        long endMemory = BenchmarkUtils.getCurrentMemoryUsage();
        BenchmarkUtils.measureAndAddGCTime(profileInfo);
        BenchmarkUtils.addProfileInfo(profileInfo, description, endTime - startTime, endMemory - startMemory);
        return result;
    }

    public static <T, E extends Exception> void profileExecutionStage(Collection<ProfileDiagnostic> profileInfo, String description, CheckedConsumer<T, E> code, T argument) throws E {
        CheckedFunction funcEquivalent = t -> {
            code.acceptThrows(t);
            return null;
        };
        BenchmarkUtils.profileExecutionStage(profileInfo, description, funcEquivalent, argument);
    }

    public static <R, E extends Exception> R profileExecutionStage(Collection<ProfileDiagnostic> profileInfo, String description, CheckedSupplier<R, E> code) throws E {
        CheckedFunction funcEquivalent = v -> code.getThrows();
        return (R)BenchmarkUtils.profileExecutionStage(profileInfo, description, funcEquivalent, null);
    }

    public static <E extends Exception> void profileExecutionStage(Collection<ProfileDiagnostic> profileInfo, String description, CheckedRunnable<E> code) throws E {
        CheckedFunction funcEquivalent = v -> {
            code.runThrows();
            return null;
        };
        BenchmarkUtils.profileExecutionStage(profileInfo, description, funcEquivalent, null);
    }

    public static <T, R> R profileExecutionStage(Collection<ProfileDiagnostic> profileInfo, String description, Function<T, R> code, T argument) {
        CheckedFunction checkedEquivalent = code::apply;
        return (R)BenchmarkUtils.profileExecutionStage(profileInfo, description, checkedEquivalent, argument);
    }

    public static <T> void profileExecutionStage(Collection<ProfileDiagnostic> profileInfo, String description, Consumer<T> code, T argument) {
        CheckedConsumer checkedEquivalent = code::accept;
        BenchmarkUtils.profileExecutionStage(profileInfo, description, checkedEquivalent, argument);
    }

    public static <R> R profileExecutionStage(Collection<ProfileDiagnostic> profileInfo, String description, Supplier<R> code) {
        CheckedSupplier checkedEquivalent = code::get;
        return (R)BenchmarkUtils.profileExecutionStage(profileInfo, description, checkedEquivalent);
    }

    public static void profileExecutionStage(Collection<ProfileDiagnostic> profileInfo, String description, Runnable code) {
        CheckedRunnable checkedEquivalent = code::run;
        BenchmarkUtils.profileExecutionStage(profileInfo, description, checkedEquivalent);
    }

    public static <T, E extends Exception> long measureTimeNanos(CheckedConsumer<T, E> code, T argument) throws E {
        long startTime = System.nanoTime();
        code.acceptThrows(argument);
        long endTime = System.nanoTime();
        return endTime - startTime;
    }

    public static <E extends Exception> long measureTimeNanos(CheckedRunnable<E> code) throws E {
        CheckedConsumer consEquivalent = v -> code.run();
        return BenchmarkUtils.measureTimeNanos(consEquivalent, null);
    }

    public static long measureTimeNanos(Runnable code) {
        CheckedRunnable checkedEquivalent = () -> code.run();
        return BenchmarkUtils.measureTimeNanos(checkedEquivalent);
    }

    public static <T> long measureTimeNanos(Consumer<T> code, T argument) {
        CheckedConsumer checkedEquivalent = code::accept;
        return BenchmarkUtils.measureTimeNanos(checkedEquivalent, argument);
    }

    public static <T, E extends Exception> long measureTimeMillis(CheckedConsumer<T, E> code, T argument) throws E {
        long startTime = System.currentTimeMillis();
        code.acceptThrows(argument);
        long endTime = System.currentTimeMillis();
        return endTime - startTime;
    }

    public static <E extends Exception> long measureTimeMillis(CheckedRunnable<E> code) throws E {
        CheckedConsumer consEquivalent = v -> code.run();
        return BenchmarkUtils.measureTimeMillis(consEquivalent, null);
    }

    public static long measureTimeMillis(Runnable code) {
        CheckedRunnable checkedEquivalent = () -> code.run();
        return BenchmarkUtils.measureTimeMillis(checkedEquivalent);
    }

    public static <T> long measureTimeMillis(Consumer<T> code, T argument) {
        CheckedConsumer checkedEquivalent = code::accept;
        return BenchmarkUtils.measureTimeMillis(checkedEquivalent, argument);
    }

    public static void printExecutionTime(String stageTag, Runnable code) {
        System.out.print(String.valueOf(stageTag) + " " + BenchmarkUtils.formatExecutionTime(BenchmarkUtils.measureTimeNanos(code)));
    }

    public static <E extends Exception> void printExecutionTime(String stageTag, CheckedRunnable<E> code) throws E {
        System.out.print(String.valueOf(stageTag) + " " + BenchmarkUtils.formatExecutionTime(BenchmarkUtils.measureTimeNanos(code)));
    }

    public static void addProfileInfo(Collection<ProfileDiagnostic> profileStages, String stage, long nanos, long memory) {
        profileStages.add(new ProfileDiagnostic(stage, Duration.ofNanos(nanos), memory, ProfileDiagnostic.MemoryUnit.BYTES));
    }

    public static String formatExecutionStages(Iterable<ProfileDiagnostic> profileInfo) {
        return BenchmarkUtils.formatExecutionStages(profileInfo, true, Optional.of(DEFAULT_MEMORY_UNITS));
    }

    public static String formatMemoryConsumption(Iterable<ProfileDiagnostic> profileInfo, ProfileDiagnostic.MemoryUnit units) {
        return BenchmarkUtils.formatExecutionStages(profileInfo, false, Optional.ofNullable(units));
    }

    public static String formatExecutionTimes(Iterable<ProfileDiagnostic> profileInfo) {
        return BenchmarkUtils.formatExecutionStages(profileInfo, true, null);
    }

    public static String formatExecutionTime(Duration time) {
        return BenchmarkUtils.formatExecutionTime("Execution time", time);
    }

    public static String formatExecutionTime(String title, Duration time) {
        return BenchmarkUtils.formatExecutionTimes(Collections.singleton(new ProfileDiagnostic(title, time, 0.0, ProfileDiagnostic.MemoryUnit.BYTES)));
    }

    public static String formatExecutionTime(long nanos) {
        return BenchmarkUtils.formatExecutionTime(Duration.ofNanos(nanos));
    }

    public static String formatExecutionTime(String title, long nanos) {
        return BenchmarkUtils.formatExecutionTime(title, Duration.ofNanos(nanos));
    }

    public static String formatMillis(long millis) {
        return BenchmarkUtils.formatDuration(Duration.ofMillis(millis));
    }

    public static String formatNanos(long nanos) {
        return BenchmarkUtils.formatDuration(Duration.ofNanos(nanos));
    }

    public static long nanosToMillis(long nanos) {
        return Duration.ofNanos(nanos).toMillis();
    }
}

