/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.edc.security.token.jwt;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.JWSSigner;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.Requirement;
import com.nimbusds.jose.crypto.ECDSASigner;
import com.nimbusds.jose.crypto.ECDSAVerifier;
import com.nimbusds.jose.crypto.Ed25519Signer;
import com.nimbusds.jose.crypto.Ed25519Verifier;
import com.nimbusds.jose.crypto.RSASSASigner;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jose.crypto.bc.BouncyCastleProviderSingleton;
import com.nimbusds.jose.jwk.Curve;
import com.nimbusds.jose.jwk.ECKey;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.KeyUse;
import com.nimbusds.jose.jwk.OctetKeyPair;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jose.util.Base64URL;
import java.security.AlgorithmParameters;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.ECKey;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.EdECKey;
import java.security.interfaces.EdECPrivateKey;
import java.security.interfaces.EdECPublicKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.EdECPoint;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.InvalidParameterSpecException;
import java.security.spec.RSAPublicKeySpec;
import java.text.ParseException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.bouncycastle.jcajce.provider.asymmetric.util.EC5Util;
import org.eclipse.edc.spi.EdcException;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class CryptoConverter {
    public static final String ALGORITHM_RSA = "rsa";
    public static final String ALGORITHM_EC = "ec";
    public static final String ALGORITHM_ECDSA = "ecdsa";
    public static final String ALGORITHM_EDDSA = "eddsa";
    public static final String ALGORITHM_ED25519 = "ed25519";
    public static final List<String> SUPPORTED_ALGORITHMS = List.of("ec", "rsa", "eddsa", "ed25519");

    public static JWSSigner createSignerFor(PrivateKey key) {
        String algorithm = key.getAlgorithm().toLowerCase();
        try {
            return switch (algorithm) {
                case ALGORITHM_EC, ALGORITHM_ECDSA -> CryptoConverter.getEcdsaSigner((ECPrivateKey)key);
                case ALGORITHM_RSA -> new RSASSASigner(key);
                case ALGORITHM_EDDSA, ALGORITHM_ED25519 -> CryptoConverter.createEdDsaVerifier(key);
                default -> throw new IllegalArgumentException(CryptoConverter.notSupportedError(algorithm));
            };
        }
        catch (JOSEException ex) {
            throw new EdcException(CryptoConverter.notSupportedError(algorithm), (Throwable)ex);
        }
    }

    @NotNull
    private static ECDSASigner getEcdsaSigner(ECPrivateKey key) throws JOSEException {
        ECDSASigner signer = new ECDSASigner(key);
        signer.getJCAContext().setProvider(BouncyCastleProviderSingleton.getInstance());
        return signer;
    }

    public static JWSVerifier createVerifierFor(PublicKey publicKey) {
        String algorithm = publicKey.getAlgorithm().toLowerCase();
        try {
            return switch (algorithm) {
                case ALGORITHM_EC, ALGORITHM_ECDSA -> CryptoConverter.getEcdsaVerifier((ECPublicKey)publicKey);
                case ALGORITHM_RSA -> new RSASSAVerifier((RSAPublicKey)publicKey);
                case ALGORITHM_EDDSA, ALGORITHM_ED25519 -> CryptoConverter.createEdDsaVerifier(publicKey);
                default -> throw new IllegalArgumentException(CryptoConverter.notSupportedError(algorithm));
            };
        }
        catch (JOSEException e) {
            throw new EdcException(CryptoConverter.notSupportedError(algorithm), (Throwable)e);
        }
    }

    public static JWK createJwk(KeyPair keypair) {
        return CryptoConverter.createJwk(keypair, null);
    }

    public static JWK createJwk(KeyPair keypair, @Nullable String kid) {
        if (keypair.getPrivate() == null && keypair.getPublic() == null) {
            throw new IllegalArgumentException("Invalid KeyPair: public and private key were both null!");
        }
        String alg = ((Key)Optional.ofNullable(keypair.getPrivate()).orElse((PrivateKey)((Object)keypair.getPublic()))).getAlgorithm();
        return switch (alg.toLowerCase()) {
            case ALGORITHM_EC -> CryptoConverter.convertEcKey(keypair, kid);
            case ALGORITHM_RSA -> CryptoConverter.convertRsaKey(keypair, kid);
            case ALGORITHM_EDDSA, ALGORITHM_ED25519 -> CryptoConverter.convertEdDsaKey(keypair, kid);
            default -> throw new IllegalArgumentException(CryptoConverter.notSupportedError(keypair.getPublic().getAlgorithm()));
        };
    }

    public static JWSAlgorithm getRecommendedAlgorithm(JWSSigner signer) {
        return CryptoConverter.getWithRequirement(signer, Requirement.REQUIRED).orElseGet(() -> CryptoConverter.getWithRequirement(signer, Requirement.RECOMMENDED).orElseGet(() -> CryptoConverter.getWithRequirement(signer, Requirement.OPTIONAL).orElse(null)));
    }

    public static JWK create(Map<String, Object> jsonObject) {
        if (jsonObject == null) {
            return null;
        }
        try {
            return JWK.parse(jsonObject);
        }
        catch (ParseException e) {
            throw new RuntimeException(e);
        }
    }

    public static JWK create(String json) {
        if (json == null) {
            return null;
        }
        try {
            return JWK.parse((String)json);
        }
        catch (ParseException e) {
            throw new RuntimeException(e);
        }
    }

    public static JWSVerifier createVerifier(JWK jwk) {
        Objects.requireNonNull(jwk, "jwk cannot be null");
        String value = jwk.getKeyType().getValue();
        try {
            return switch (value) {
                case "EC" -> new ECDSAVerifier((com.nimbusds.jose.jwk.ECKey)jwk);
                case "OKP" -> new Ed25519Verifier((OctetKeyPair)jwk);
                case "RSA" -> new RSASSAVerifier((RSAKey)jwk);
                default -> throw new UnsupportedOperationException(String.format("Cannot create JWSVerifier for JWK-type [%s], currently only supporting EC, OKP and RSA", value));
            };
        }
        catch (JOSEException ex) {
            throw new UnsupportedOperationException(ex);
        }
    }

    public static JWSSigner createSigner(JWK jwk) {
        String value = jwk.getKeyType().getValue();
        try {
            return switch (value) {
                case "EC" -> new ECDSASigner((com.nimbusds.jose.jwk.ECKey)jwk);
                case "OKP" -> new Ed25519Signer((OctetKeyPair)jwk);
                case "RSA" -> new RSASSASigner((RSAKey)jwk);
                default -> throw new UnsupportedOperationException(String.format("Cannot create JWSVerifier for JWK-type [%s], currently only supporting EC, OKP and RSA", value));
            };
        }
        catch (JOSEException ex) {
            throw new UnsupportedOperationException(ex);
        }
    }

    private static Curve getCurveAllowing(EdECKey edKey, String ... allowedCurves) {
        String curveName = edKey.getParams().getName();
        if (!Arrays.asList(allowedCurves).contains(curveName.toLowerCase())) {
            throw new IllegalArgumentException("Unsupported curve: %s. Only the following curves is supported: %s.".formatted(curveName, String.join((CharSequence)",", allowedCurves)));
        }
        return Curve.parse((String)curveName);
    }

    private static RSAKey convertRsaKey(KeyPair keypair, @Nullable String kid) {
        if (keypair.getPublic() == null && keypair.getPrivate() == null) {
            throw new IllegalArgumentException("Either the public or the private key of a keypair must be non-null when converting RSA -> JWK");
        }
        PublicKey key = Optional.ofNullable(keypair.getPublic()).orElseGet(() -> {
            RSAPublicKeySpec keySpec = new RSAPublicKeySpec(((RSAPrivateCrtKey)keypair.getPrivate()).getModulus(), ((RSAPrivateCrtKey)keypair.getPrivate()).getPublicExponent());
            try {
                KeyFactory gen = KeyFactory.getInstance("RSA");
                return gen.generatePublic(keySpec);
            }
            catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
                throw new RuntimeException(e);
            }
        });
        RSAKey.Builder builder = new RSAKey.Builder((RSAPublicKey)key);
        if (keypair.getPrivate() != null) {
            builder.privateKey(keypair.getPrivate());
        }
        return builder.keyID(kid).keyUse(KeyUse.SIGNATURE).build();
    }

    private static com.nimbusds.jose.jwk.ECKey convertEcKey(KeyPair keypair, @Nullable String kid) {
        ECPublicKey pub = (ECPublicKey)keypair.getPublic();
        ECPrivateKey priv = (ECPrivateKey)keypair.getPrivate();
        ECKey key = Optional.ofNullable(pub).orElse((ECPublicKey)((Object)priv));
        try {
            AlgorithmParameters algorithmParameters = AlgorithmParameters.getInstance("EC");
            algorithmParameters.init(key.getParams());
            String curveName = algorithmParameters.getParameterSpec(ECGenParameterSpec.class).getName();
            if (pub == null) {
                org.bouncycastle.jce.spec.ECParameterSpec bcSpec = EC5Util.convertSpec((ECParameterSpec)priv.getParams());
                org.bouncycastle.math.ec.ECPoint q = bcSpec.getG().multiply(priv.getS()).normalize();
                ECPoint pointQjce = new ECPoint(q.getAffineXCoord().toBigInteger(), q.getAffineYCoord().toBigInteger());
                ECPublicKeySpec spec = new ECPublicKeySpec(pointQjce, priv.getParams());
                pub = (ECPublicKey)KeyFactory.getInstance("EC").generatePublic(spec);
            }
            return new ECKey.Builder(Curve.forOID((String)curveName), pub).privateKey(priv).keyID(kid).keyUse(KeyUse.SIGNATURE).build();
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException | InvalidParameterSpecException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static byte[] reverseArray(byte[] array) {
        for (int i = 0; i < array.length / 2; ++i) {
            byte temp = array[i];
            array[i] = array[array.length - 1 - i];
            array[array.length - 1 - i] = temp;
        }
        return array;
    }

    private static Ed25519Verifier createEdDsaVerifier(PublicKey publicKey) throws JOSEException {
        EdECPublicKey edKey = (EdECPublicKey)publicKey;
        Curve curve = CryptoConverter.getCurveAllowing(edKey, ALGORITHM_ED25519);
        Base64URL urlX = CryptoConverter.encodeX(edKey.getPoint());
        OctetKeyPair okp = new OctetKeyPair.Builder(curve, urlX).build();
        return new Ed25519Verifier(okp);
    }

    @NotNull
    private static ECDSAVerifier getEcdsaVerifier(ECPublicKey publicKey) throws JOSEException {
        ECDSAVerifier verifier = new ECDSAVerifier(publicKey);
        verifier.getJCAContext().setProvider(BouncyCastleProviderSingleton.getInstance());
        return verifier;
    }

    @NotNull
    private static Optional<JWSAlgorithm> getWithRequirement(JWSSigner signer, Requirement requirement) {
        return signer.supportedJWSAlgorithms().stream().filter(alg -> alg.getRequirement() == requirement).findFirst();
    }

    private static Ed25519Signer createEdDsaVerifier(PrivateKey key) throws JOSEException {
        EdECPrivateKey edKey = (EdECPrivateKey)key;
        Curve curve = CryptoConverter.getCurveAllowing(edKey, ALGORITHM_ED25519);
        Base64URL urlX = Base64URL.encode((byte[])new byte[0]);
        Base64URL urlD = CryptoConverter.encodeD(edKey);
        OctetKeyPair octetKeyPair = new OctetKeyPair.Builder(curve, urlX).d(urlD).build();
        return new Ed25519Signer(octetKeyPair);
    }

    private static OctetKeyPair convertEdDsaKey(KeyPair keypair, @Nullable String kid) {
        EdECPublicKey pub = (EdECPublicKey)keypair.getPublic();
        EdECPrivateKey priv = (EdECPrivateKey)keypair.getPrivate();
        Base64URL urlX = Optional.ofNullable(pub).map(pubkey -> CryptoConverter.encodeX(pubkey.getPoint())).orElseGet(() -> Base64URL.encode((byte[])new byte[0]));
        Base64URL urlD = Optional.ofNullable(priv).map(CryptoConverter::encodeD).orElse(null);
        String curveName = ((EdECKey)Optional.ofNullable(priv).orElse((EdECPrivateKey)((Object)pub))).getParams().getName();
        return new OctetKeyPair.Builder(Curve.parse((String)curveName), urlX).d(urlD).keyID(kid).build();
    }

    @NotNull
    private static Base64URL encodeD(EdECPrivateKey edKey) {
        byte[] bytes = edKey.getBytes().orElseThrow(() -> new EdcException("Private key is not willing to disclose its bytes"));
        return Base64URL.encode((byte[])bytes);
    }

    @NotNull
    private static Base64URL encodeX(EdECPoint point) {
        byte[] bytes = CryptoConverter.reverseArray(point.getY().toByteArray());
        if (point.isXOdd()) {
            int mask = -128;
            int n = bytes.length - 1;
            bytes[n] = (byte)(bytes[n] ^ mask);
        }
        return Base64URL.encode((byte[])bytes);
    }

    private static String notSupportedError(String algorithm) {
        return "Could not convert PrivateKey to a JWSSigner, currently only the following types are supported: %s. The specified key was a %s".formatted(String.join((CharSequence)",", SUPPORTED_ALGORITHMS), algorithm);
    }
}

