/*
 * Decompiled with CFR 0.152.
 */
package org.easypeelsecurity.springdog.manager.ratelimit;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Optional;
import java.util.logging.Logger;
import org.easypeelsecurity.springdog.domain.ratelimit.EndpointService;
import org.easypeelsecurity.springdog.domain.ratelimit.RuleCache;
import org.easypeelsecurity.springdog.manager.ratelimit.RatelimitCache;
import org.easypeelsecurity.springdog.manager.statistics.EndpointMetricCacheManager;
import org.easypeelsecurity.springdog.manager.util.RequestHandlerUtil;
import org.easypeelsecurity.springdog.shared.dto.EndpointDto;
import org.easypeelsecurity.springdog.shared.dto.EndpointHeaderDto;
import org.easypeelsecurity.springdog.shared.dto.EndpointParameterDto;
import org.easypeelsecurity.springdog.shared.enums.RuleStatus;
import org.easypeelsecurity.springdog.shared.util.IpAddressUtil;
import org.easypeelsecurity.springdog.shared.util.MethodSignatureParser;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.util.ContentCachingRequestWrapper;

@Service
public class RatelimitInterceptor
implements HandlerInterceptor {
    private final EndpointService endpointService;
    private final Logger logger = Logger.getLogger(RatelimitInterceptor.class.getName());
    private final ObjectMapper objectMapper = new ObjectMapper();

    public RatelimitInterceptor(EndpointService endpointService) {
        this.endpointService = endpointService;
    }

    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        if (!(request instanceof ContentCachingRequestWrapper)) {
            request = new ContentCachingRequestWrapper(request);
        }
        if (handler instanceof HandlerMethod) {
            HandlerMethod handlerMethod = (HandlerMethod)handler;
            Object controller = handlerMethod.getBean();
            Class<?> controllerClass = controller.getClass();
            if (RequestHandlerUtil.shouldSkipRequest(controllerClass)) {
                return true;
            }
            String methodSignature = MethodSignatureParser.parse((HandlerMethod)handlerMethod);
            Optional<EndpointDto> optionalEndpoint = this.getValidEndpoint(methodSignature);
            if (optionalEndpoint.isEmpty()) {
                return true;
            }
            EndpointDto endpoint = optionalEndpoint.get();
            if (!RuleStatus.ACTIVE.equals((Object)endpoint.getRuleStatus())) {
                return true;
            }
            String requestHashed = this.generateRequestHash(request, endpoint);
            if (RatelimitCache.isBannedRequest(requestHashed, endpoint, LocalDateTime.now())) {
                int banTimeSeconds = endpoint.isRulePermanentBan() ? Integer.MAX_VALUE : endpoint.getRuleBanTimeInSeconds();
                this.applyRatelimitResponse(response, String.valueOf(banTimeSeconds));
                EndpointMetricCacheManager.incrementFailureCount(methodSignature);
                return false;
            }
            response.setHeader("X-RateLimit-Remaining", "");
        }
        return true;
    }

    private Optional<EndpointDto> getValidEndpoint(String methodSignature) {
        EndpointDto endpoint = RuleCache.findEndpointByMethodSignature((String)methodSignature).orElseGet(() -> {
            EndpointDto item = this.endpointService.getEndpointByMethodSignature(methodSignature);
            if (item == null) {
                return null;
            }
            RuleCache.cachingRule((EndpointDto)item);
            return item;
        });
        return Optional.ofNullable(endpoint);
    }

    private void applyRatelimitResponse(HttpServletResponse response, String retryAfter) throws IOException {
        response.setStatus(HttpStatus.TOO_MANY_REQUESTS.value());
        response.getWriter().write("Too many requests");
        response.setHeader("Retry-After", retryAfter);
        response.setHeader("X-RateLimit-Remaining", "0");
    }

    private String generateRequestHash(HttpServletRequest request, EndpointDto endpoint) {
        StringBuilder result = new StringBuilder();
        result.append(endpoint.getMethodSignature()).append("\n");
        if (endpoint.isRuleIpBased()) {
            result.append(IpAddressUtil.getClientIp((HttpServletRequest)request)).append("\n");
        }
        endpoint.getParameters().stream().sorted(Comparator.comparing(EndpointParameterDto::getName)).filter(EndpointParameterDto::isEnabled).forEach(param -> {
            String value = request.getParameter(param.getName());
            if (value != null) {
                result.append(param.getName()).append("=").append(value).append("\n");
            }
        });
        endpoint.getHeaders().stream().sorted(Comparator.comparing(EndpointHeaderDto::getName)).filter(EndpointHeaderDto::isEnabled).forEach(header -> {
            String value = request.getHeader(header.getName());
            if (value != null) {
                result.append(header.getName()).append("=").append(value).append("\n");
            }
        });
        try {
            Method[] methods;
            JsonNode requestBody = this.getRequestBodyAsJson(request);
            Class<?> clazz = Class.forName(endpoint.getFqcn());
            for (Method method : methods = clazz.getDeclaredMethods()) {
                Parameter[] parameters;
                if (!method.toString().contains(endpoint.getMethodSignature())) continue;
                for (Parameter parameter : parameters = method.getParameters()) {
                    if (!parameter.isAnnotationPresent(RequestBody.class)) continue;
                    Class<?> paramType = parameter.getType();
                    Field[] fields = paramType.getDeclaredFields();
                    Arrays.sort(fields, Comparator.comparing(Field::getName));
                    this.processFields(fields, request, requestBody, result);
                }
            }
            return result.toString();
        }
        catch (ClassNotFoundException e) {
            this.logger.warning("Failed to generate request hash. " + e.getMessage());
            throw new RuntimeException(e);
        }
    }

    private JsonNode getRequestBodyAsJson(HttpServletRequest request) {
        try {
            return this.objectMapper.readTree((Reader)request.getReader());
        }
        catch (IOException e) {
            this.logger.warning("Failed to read request body as json" + e.getMessage());
            return null;
        }
    }

    private void processFields(Field[] fields, HttpServletRequest request, JsonNode requestBody, StringBuilder result) {
        for (Field field : fields) {
            String value = request.getParameter(field.getName());
            if (value == null && requestBody != null && requestBody.has(field.getName())) {
                value = requestBody.get(field.getName()).asText();
            }
            if (value == null) continue;
            result.append(field.getName()).append("=").append(value).append("\n");
        }
    }
}

