/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.spi;

import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.enumerations.RoleOfPspOid;
import eu.europa.esig.dss.enumerations.SemanticsIdentifier;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.Digest;
import eu.europa.esig.dss.model.TimestampBinary;
import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.model.x509.X500PrincipalHelper;
import eu.europa.esig.dss.spi.DSSSecurityProvider;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.OID;
import eu.europa.esig.dss.spi.x509.CertificateIdentifier;
import eu.europa.esig.dss.spi.x509.CertificatePolicy;
import eu.europa.esig.dss.spi.x509.CertificateRef;
import eu.europa.esig.dss.spi.x509.PSD2QcType;
import eu.europa.esig.dss.spi.x509.RoleOfPSP;
import eu.europa.esig.dss.utils.Utils;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.CertificateException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import javax.naming.ldap.Rdn;
import javax.security.auth.x500.X500Principal;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1GeneralizedTime;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1OutputStream;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1Set;
import org.bouncycastle.asn1.ASN1String;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.asn1.DLSet;
import org.bouncycastle.asn1.cms.Attribute;
import org.bouncycastle.asn1.cms.AttributeTable;
import org.bouncycastle.asn1.cms.Attributes;
import org.bouncycastle.asn1.esf.RevocationValues;
import org.bouncycastle.asn1.ess.OtherCertID;
import org.bouncycastle.asn1.ocsp.BasicOCSPResponse;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.x500.AttributeTypeAndValue;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x509.AccessDescription;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.AuthorityInformationAccess;
import org.bouncycastle.asn1.x509.AuthorityKeyIdentifier;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.IssuerSerial;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.asn1.x509.PolicyInformation;
import org.bouncycastle.asn1.x509.PolicyQualifierId;
import org.bouncycastle.asn1.x509.PolicyQualifierInfo;
import org.bouncycastle.asn1.x509.SubjectKeyIdentifier;
import org.bouncycastle.asn1.x509.Time;
import org.bouncycastle.asn1.x509.X509ObjectIdentifiers;
import org.bouncycastle.asn1.x509.qualified.QCStatement;
import org.bouncycastle.asn1.x509.qualified.RFC3739QCObjectIdentifiers;
import org.bouncycastle.asn1.x509.qualified.SemanticsInformation;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.SignerId;
import org.bouncycastle.cms.SignerInformation;
import org.bouncycastle.tsp.TSPException;
import org.bouncycastle.tsp.TimeStampToken;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DSSASN1Utils {
    private static final Logger LOG = LoggerFactory.getLogger(DSSASN1Utils.class);
    private static final String QC_TYPE_STATEMENT_OID = "0.4.0.1862.1.6";
    private static final String QC_LEGISLATION_OID = "0.4.0.1862.1.7";

    private DSSASN1Utils() {
    }

    public static <T extends ASN1Primitive> T toASN1Primitive(byte[] bytes) {
        try {
            return (T)ASN1Primitive.fromByteArray((byte[])bytes);
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    private static boolean isDEROctetStringNull(DEROctetString derOctetString) {
        byte[] derOctetStringBytes = derOctetString.getOctets();
        Object asn1Null = DSSASN1Utils.toASN1Primitive(derOctetStringBytes);
        return DERNull.INSTANCE.equals(asn1Null);
    }

    public static byte[] getDEREncoded(ASN1Encodable asn1Encodable) {
        return DSSASN1Utils.getEncoded(asn1Encodable, "DER");
    }

    public static byte[] getBEREncoded(ASN1Encodable asn1Encodable) {
        return DSSASN1Utils.getEncoded(asn1Encodable, "BER");
    }

    private static byte[] getEncoded(ASN1Encodable asn1Encodable, String encoding) {
        try {
            return asn1Encodable.toASN1Primitive().getEncoded(encoding);
        }
        catch (IOException e) {
            throw new DSSException("Unable to encode to " + encoding, (Throwable)e);
        }
    }

    public static byte[] getEncoded(BasicOCSPResp basicOCSPResp) {
        try {
            BasicOCSPResponse basicOCSPResponse = BasicOCSPResponse.getInstance((Object)basicOCSPResp.getEncoded());
            return DSSASN1Utils.getDEREncoded((ASN1Encodable)basicOCSPResponse);
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static Date toDate(ASN1GeneralizedTime asn1Date) {
        try {
            return asn1Date.getDate();
        }
        catch (ParseException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static String toString(ASN1OctetString value) {
        return new String(value.getOctets());
    }

    public static byte[] getEncoded(TimeStampToken timeStampToken) {
        return DSSASN1Utils.getEncoded(timeStampToken.toCMSSignedData());
    }

    public static byte[] getEncoded(CMSSignedData cmsSignedData) {
        try {
            return cmsSignedData.getEncoded();
        }
        catch (IOException e) {
            throw new DSSException("Unable to encode to DER", (Throwable)e);
        }
    }

    public static byte[] getDEREncoded(TimeStampToken timeStampToken) {
        return DSSASN1Utils.getDEREncoded(timeStampToken.toCMSSignedData());
    }

    public static byte[] getDEREncoded(CMSSignedData data) {
        byte[] byArray;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            ASN1OutputStream asn1OutputStream = ASN1OutputStream.create((OutputStream)baos, (String)"DER");
            asn1OutputStream.writeObject((ASN1Encodable)data.toASN1Structure());
            asn1OutputStream.close();
            byArray = baos.toByteArray();
        }
        catch (Throwable throwable) {
            try {
                try {
                    baos.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new DSSException("Unable to encode to DER", (Throwable)e);
            }
        }
        baos.close();
        return byArray;
    }

    public static byte[] getDEREncoded(TimestampBinary timestampBinary) {
        return DSSASN1Utils.getDEREncoded(timestampBinary.getBytes());
    }

    public static byte[] getDEREncoded(byte[] bytes) {
        try {
            return DSSASN1Utils.getDEREncoded((ASN1Encodable)ASN1Primitive.fromByteArray((byte[])bytes));
        }
        catch (IOException e) {
            throw new DSSException("Unable to encode to DER", (Throwable)e);
        }
    }

    public static ASN1Sequence getAsn1SequenceFromDerOctetString(byte[] bytes) {
        return DSSASN1Utils.getASN1Sequence(DSSASN1Utils.getDEROctetStringContent(bytes));
    }

    private static byte[] getDEROctetStringContent(byte[] bytes) {
        byte[] byArray;
        ASN1InputStream input = new ASN1InputStream(bytes);
        try {
            DEROctetString s = (DEROctetString)input.readObject();
            byArray = s.getOctets();
        }
        catch (Throwable throwable) {
            try {
                try {
                    input.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new DSSException("Unable to retrieve the DEROctetString content", (Throwable)e);
            }
        }
        input.close();
        return byArray;
    }

    private static ASN1Sequence getASN1Sequence(byte[] bytes) {
        ASN1Sequence aSN1Sequence;
        ASN1InputStream input = new ASN1InputStream(bytes);
        try {
            aSN1Sequence = (ASN1Sequence)input.readObject();
        }
        catch (Throwable throwable) {
            try {
                try {
                    input.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new DSSException("Unable to retrieve the ASN1Sequence", (Throwable)e);
            }
        }
        input.close();
        return aSN1Sequence;
    }

    public static byte[] getAsn1SignaturePolicyDigest(DigestAlgorithm digestAlgorithm, byte[] policyBytes) {
        ASN1Sequence asn1Seq = (ASN1Sequence)DSSASN1Utils.toASN1Primitive(policyBytes);
        ASN1Sequence signPolicyHashAlgObject = (ASN1Sequence)asn1Seq.getObjectAt(0);
        AlgorithmIdentifier signPolicyHashAlgIdentifier = AlgorithmIdentifier.getInstance((Object)signPolicyHashAlgObject);
        ASN1Sequence signPolicyInfo = (ASN1Sequence)asn1Seq.getObjectAt(1);
        byte[] hashAlgorithmDEREncoded = DSSASN1Utils.getDEREncoded((ASN1Encodable)signPolicyHashAlgIdentifier);
        byte[] signPolicyInfoDEREncoded = DSSASN1Utils.getDEREncoded((ASN1Encodable)signPolicyInfo);
        return DSSUtils.digest(digestAlgorithm, hashAlgorithmDEREncoded, signPolicyInfoDEREncoded);
    }

    public static AlgorithmIdentifier getAlgorithmIdentifier(ASN1Sequence atsHashIndexValue) {
        if (atsHashIndexValue != null && atsHashIndexValue.size() > 3) {
            boolean algorithmIndex = false;
            ASN1Encodable asn1Encodable = atsHashIndexValue.getObjectAt(0);
            if (asn1Encodable instanceof ASN1Sequence) {
                ASN1Sequence asn1Sequence = (ASN1Sequence)asn1Encodable;
                return AlgorithmIdentifier.getInstance((Object)asn1Sequence);
            }
            if (asn1Encodable instanceof ASN1ObjectIdentifier) {
                ASN1ObjectIdentifier derObjectIdentifier = ASN1ObjectIdentifier.getInstance((Object)asn1Encodable);
                return new AlgorithmIdentifier(derObjectIdentifier);
            }
        }
        return null;
    }

    public static AlgorithmIdentifier getAlgorithmIdentifier(DigestAlgorithm digestAlgorithm) {
        ASN1ObjectIdentifier asn1ObjectIdentifier = new ASN1ObjectIdentifier(digestAlgorithm.getOid());
        return new AlgorithmIdentifier(asn1ObjectIdentifier, (ASN1Encodable)DERNull.INSTANCE);
    }

    public static ASN1Sequence getCertificatesHashIndex(ASN1Sequence atsHashIndexValue) {
        if (atsHashIndexValue != null) {
            int certificateIndex = 0;
            if (atsHashIndexValue.size() > 3) {
                ++certificateIndex;
            }
            return (ASN1Sequence)atsHashIndexValue.getObjectAt(certificateIndex).toASN1Primitive();
        }
        return null;
    }

    public static ASN1Sequence getCRLHashIndex(ASN1Sequence atsHashIndexValue) {
        if (atsHashIndexValue != null) {
            int crlIndex = 1;
            if (atsHashIndexValue.size() > 3) {
                ++crlIndex;
            }
            return (ASN1Sequence)atsHashIndexValue.getObjectAt(crlIndex).toASN1Primitive();
        }
        return null;
    }

    public static ASN1Sequence getUnsignedAttributesHashIndex(ASN1Sequence atsHashIndexValue) {
        if (atsHashIndexValue != null) {
            int unsignedAttributesIndex = 2;
            if (atsHashIndexValue.size() > 3) {
                ++unsignedAttributesIndex;
            }
            return (ASN1Sequence)atsHashIndexValue.getObjectAt(unsignedAttributesIndex).toASN1Primitive();
        }
        return null;
    }

    public static List<DEROctetString> getDEROctetStrings(ASN1Sequence asn1Sequence) {
        ArrayList<DEROctetString> derOctetStrings = new ArrayList<DEROctetString>();
        if (asn1Sequence != null) {
            derOctetStrings.addAll(Collections.list(asn1Sequence.getObjects()));
        }
        return derOctetStrings;
    }

    public static boolean hasIdPkixOcspNoCheckExtension(CertificateToken token) {
        byte[] extensionValue = token.getCertificate().getExtensionValue(OCSPObjectIdentifiers.id_pkix_ocsp_nocheck.getId());
        if (extensionValue != null) {
            try {
                Object derObject = DSSASN1Utils.toASN1Primitive(extensionValue);
                if (derObject instanceof DEROctetString) {
                    return DSSASN1Utils.isDEROctetStringNull((DEROctetString)derObject);
                }
            }
            catch (Exception e) {
                LOG.debug("Exception when processing 'id_pkix_ocsp_no_check'", (Throwable)e);
            }
        }
        return false;
    }

    public static List<CertificatePolicy> getCertificatePolicies(CertificateToken certToken) {
        ArrayList<CertificatePolicy> certificatePolicies = new ArrayList<CertificatePolicy>();
        byte[] certificatePoliciesBinaries = certToken.getCertificate().getExtensionValue(Extension.certificatePolicies.getId());
        if (Utils.isArrayNotEmpty((byte[])certificatePoliciesBinaries)) {
            try {
                ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(certificatePoliciesBinaries);
                for (int ii = 0; ii < seq.size(); ++ii) {
                    CertificatePolicy cp = new CertificatePolicy();
                    PolicyInformation policyInfo = PolicyInformation.getInstance((Object)seq.getObjectAt(ii));
                    cp.setOid(policyInfo.getPolicyIdentifier().getId());
                    ASN1Sequence policyQualifiersSeq = policyInfo.getPolicyQualifiers();
                    if (policyQualifiersSeq != null) {
                        for (int jj = 0; jj < policyQualifiersSeq.size(); ++jj) {
                            PolicyQualifierInfo pqi = PolicyQualifierInfo.getInstance((Object)policyQualifiersSeq.getObjectAt(jj));
                            if (!PolicyQualifierId.id_qt_cps.equals((ASN1Primitive)pqi.getPolicyQualifierId())) continue;
                            cp.setCpsUrl(DSSASN1Utils.getString(pqi.getQualifier()));
                        }
                    }
                    certificatePolicies.add(cp);
                }
            }
            catch (Exception e) {
                LOG.warn("Unable to parse the certificatePolicies extension '{}' : {}", new Object[]{Utils.toBase64((byte[])certificatePoliciesBinaries), e.getMessage(), e});
            }
        }
        return certificatePolicies;
    }

    public static List<String> getQCStatementsIdList(CertificateToken certToken) {
        ArrayList<String> extensionIdList = new ArrayList<String>();
        byte[] qcStatement = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (Utils.isArrayNotEmpty((byte[])qcStatement)) {
            try {
                ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatement);
                for (int ii = 0; ii < seq.size(); ++ii) {
                    QCStatement statement = QCStatement.getInstance((Object)seq.getObjectAt(ii));
                    extensionIdList.add(statement.getStatementId().getId());
                }
            }
            catch (Exception e) {
                LOG.warn("Unable to parse the qCStatements extension '{}' : {}", new Object[]{Utils.toBase64((byte[])qcStatement), e.getMessage(), e});
            }
        }
        return extensionIdList;
    }

    public static List<String> getQCTypesIdList(CertificateToken certToken) {
        ArrayList<String> qcTypesIdList = new ArrayList<String>();
        byte[] qcStatement = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (Utils.isArrayNotEmpty((byte[])qcStatement)) {
            try {
                ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatement);
                for (int ii = 0; ii < seq.size(); ++ii) {
                    QCStatement statement = QCStatement.getInstance((Object)seq.getObjectAt(ii));
                    if (!QC_TYPE_STATEMENT_OID.equals(statement.getStatementId().getId())) continue;
                    ASN1Encodable qcTypeInfo1 = statement.getStatementInfo();
                    if (qcTypeInfo1 instanceof ASN1Sequence) {
                        ASN1Sequence qcTypeInfo = (ASN1Sequence)qcTypeInfo1;
                        for (int jj = 0; jj < qcTypeInfo.size(); ++jj) {
                            ASN1Encodable e1 = qcTypeInfo.getObjectAt(jj);
                            if (e1 instanceof ASN1ObjectIdentifier) {
                                ASN1ObjectIdentifier oid = (ASN1ObjectIdentifier)e1;
                                qcTypesIdList.add(oid.getId());
                                continue;
                            }
                            LOG.warn("ASN1Sequence in QcTypes does not contain ASN1ObjectIdentifer, but {}", (Object)e1.getClass().getName());
                        }
                        continue;
                    }
                    LOG.warn("QcTypes not an ASN1Sequence, but {}", (Object)qcTypeInfo1.getClass().getName());
                }
            }
            catch (Exception e) {
                LOG.warn("Unable to parse the qCStatements extension '{}' : {}", new Object[]{Utils.toBase64((byte[])qcStatement), e.getMessage(), e});
            }
        }
        return qcTypesIdList;
    }

    public static List<String> getQCLegislations(CertificateToken certToken) {
        ArrayList<String> result = new ArrayList<String>();
        byte[] qcStatement = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (Utils.isArrayNotEmpty((byte[])qcStatement)) {
            try {
                ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatement);
                for (int ii = 0; ii < seq.size(); ++ii) {
                    QCStatement statement = QCStatement.getInstance((Object)seq.getObjectAt(ii));
                    if (!QC_LEGISLATION_OID.equals(statement.getStatementId().getId())) continue;
                    ASN1Sequence sequenceLegislation = ASN1Sequence.getInstance((Object)statement.getStatementInfo());
                    for (int jj = 0; jj < sequenceLegislation.size(); ++jj) {
                        result.add(DSSASN1Utils.getString(sequenceLegislation.getObjectAt(jj)));
                    }
                }
            }
            catch (Exception e) {
                LOG.warn("Unable to parse the qCStatements extension '{}' : {}", new Object[]{Utils.toBase64((byte[])qcStatement), e.getMessage(), e});
            }
        }
        return result;
    }

    public static byte[] getSki(CertificateToken certificateToken) {
        return DSSASN1Utils.getSki(certificateToken, false);
    }

    public static byte[] getSki(CertificateToken certificateToken, boolean computeIfMissing) {
        try {
            byte[] extensionValue = certificateToken.getCertificate().getExtensionValue(Extension.subjectKeyIdentifier.getId());
            if (Utils.isArrayNotEmpty((byte[])extensionValue)) {
                ASN1Primitive extension = JcaX509ExtensionUtils.parseExtensionValue((byte[])extensionValue);
                SubjectKeyIdentifier skiBC = SubjectKeyIdentifier.getInstance((Object)extension);
                return skiBC.getKeyIdentifier();
            }
            if (computeIfMissing) {
                return DSSASN1Utils.computeSkiFromCert(certificateToken);
            }
            return null;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static byte[] getAuthorityKeyIdentifier(CertificateToken certificateToken) {
        byte[] extensionValue = certificateToken.getCertificate().getExtensionValue(Extension.authorityKeyIdentifier.getId());
        if (Utils.isArrayNotEmpty((byte[])extensionValue)) {
            try {
                ASN1Primitive extension = JcaX509ExtensionUtils.parseExtensionValue((byte[])extensionValue);
                AuthorityKeyIdentifier aki = AuthorityKeyIdentifier.getInstance((Object)extension);
                return aki.getKeyIdentifier();
            }
            catch (IOException e) {
                throw new DSSException("Unable to parse the authorityKeyIdentifier extension", (Throwable)e);
            }
        }
        return null;
    }

    public static byte[] computeSkiFromCert(CertificateToken certificateToken) {
        return DSSASN1Utils.computeSkiFromCertPublicKey(certificateToken.getPublicKey());
    }

    public static byte[] computeSkiFromCertPublicKey(PublicKey publicKey) {
        try {
            DLSequence seq = (DLSequence)ASN1Primitive.fromByteArray((byte[])publicKey.getEncoded());
            DERBitString item = (DERBitString)seq.getObjectAt(1);
            return DSSUtils.digest(DigestAlgorithm.SHA1, item.getOctets());
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static boolean isSkiEqual(byte[] ski, CertificateToken certificateToken) {
        byte[] certSki = DSSASN1Utils.computeSkiFromCert(certificateToken);
        return Arrays.equals(certSki, ski);
    }

    public static List<String> getCAAccessLocations(CertificateToken certificate) {
        return DSSASN1Utils.getAccessLocations(certificate, X509ObjectIdentifiers.id_ad_caIssuers);
    }

    public static List<String> getOCSPAccessLocations(CertificateToken certificate) {
        return DSSASN1Utils.getAccessLocations(certificate, X509ObjectIdentifiers.id_ad_ocsp);
    }

    private static List<String> getAccessLocations(CertificateToken certificate, ASN1ObjectIdentifier aiaType) {
        ArrayList<String> locationsUrls = new ArrayList<String>();
        byte[] authInfoAccessExtensionValue = certificate.getCertificate().getExtensionValue(Extension.authorityInfoAccess.getId());
        if (null == authInfoAccessExtensionValue) {
            return locationsUrls;
        }
        try {
            AccessDescription[] accessDescriptions;
            ASN1Sequence asn1Sequence = DSSASN1Utils.getAsn1SequenceFromDerOctetString(authInfoAccessExtensionValue);
            if (asn1Sequence == null || asn1Sequence.size() == 0) {
                LOG.warn("Empty ASN1Sequence for AuthorityInformationAccess");
                return locationsUrls;
            }
            AuthorityInformationAccess authorityInformationAccess = AuthorityInformationAccess.getInstance((Object)asn1Sequence);
            for (AccessDescription accessDescription : accessDescriptions = authorityInformationAccess.getAccessDescriptions()) {
                GeneralName gn;
                String location;
                if (!aiaType.equals((ASN1Primitive)accessDescription.getAccessMethod()) || (location = DSSASN1Utils.parseGn(gn = accessDescription.getAccessLocation())) == null) continue;
                locationsUrls.add(location);
            }
        }
        catch (Exception e) {
            LOG.error("Unable to parse authorityInfoAccess", (Throwable)e);
        }
        return locationsUrls;
    }

    public static List<String> getCrlUrls(CertificateToken certificateToken) {
        ArrayList<String> urls = new ArrayList<String>();
        byte[] crlDistributionPointsBytes = certificateToken.getCertificate().getExtensionValue(Extension.cRLDistributionPoints.getId());
        if (crlDistributionPointsBytes != null) {
            try {
                DistributionPoint[] distributionPoints;
                ASN1Sequence asn1Sequence = DSSASN1Utils.getAsn1SequenceFromDerOctetString(crlDistributionPointsBytes);
                CRLDistPoint distPoint = CRLDistPoint.getInstance((Object)asn1Sequence);
                for (DistributionPoint distributionPoint : distributionPoints = distPoint.getDistributionPoints()) {
                    GeneralName[] names;
                    DistributionPointName distributionPointName = distributionPoint.getDistributionPoint();
                    if (0 != distributionPointName.getType()) continue;
                    GeneralNames generalNames = (GeneralNames)distributionPointName.getName();
                    for (GeneralName name : names = generalNames.getNames()) {
                        String location = DSSASN1Utils.parseGn(name);
                        if (location == null) continue;
                        urls.add(location);
                    }
                }
            }
            catch (Exception e) {
                LOG.error("Unable to parse cRLDistributionPoints", (Throwable)e);
            }
        }
        return urls;
    }

    private static String parseGn(GeneralName gn) {
        try {
            if (6 == gn.getTagNo()) {
                ASN1String str = (ASN1String)((DERTaggedObject)gn.toASN1Primitive()).getObject();
                return str.getString();
            }
        }
        catch (Exception e) {
            LOG.warn("Unable to parse GN '{}'", (Object)gn, (Object)e);
        }
        return null;
    }

    public static boolean isOCSPSigning(CertificateToken certToken) {
        return DSSASN1Utils.isExtendedKeyUsagePresent(certToken, KeyPurposeId.id_kp_OCSPSigning.toOID());
    }

    public static boolean isExtendedKeyUsagePresent(CertificateToken certToken, ASN1ObjectIdentifier oid) {
        try {
            List<String> keyPurposes = certToken.getCertificate().getExtendedKeyUsage();
            if (keyPurposes != null && keyPurposes.contains(oid.getId())) {
                return true;
            }
        }
        catch (CertificateParsingException e) {
            LOG.error("Unable to retrieve ExtendedKeyUsage from certificate", (Throwable)e);
        }
        return false;
    }

    public static X509CertificateHolder getX509CertificateHolder(CertificateToken certToken) {
        try {
            return new X509CertificateHolder(certToken.getEncoded());
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static CertificateToken getCertificate(X509CertificateHolder x509CertificateHolder) {
        try {
            JcaX509CertificateConverter converter = new JcaX509CertificateConverter().setProvider(DSSSecurityProvider.getSecurityProviderName());
            X509Certificate x509Certificate = converter.getCertificate(x509CertificateHolder);
            return new CertificateToken(x509Certificate);
        }
        catch (CertificateException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static CertificateIdentifier toIssuerSerialInfo(SignerId signerId) {
        X500Principal issuerX500Principal = DSSASN1Utils.toX500Principal(signerId.getIssuer());
        return DSSASN1Utils.toCertificateIdentifier(issuerX500Principal, signerId.getSerialNumber(), signerId.getSubjectKeyIdentifier());
    }

    public static X500Principal toX500Principal(X500Name x500Name) {
        if (x500Name == null) {
            return null;
        }
        try {
            return new X500Principal(x500Name.getEncoded());
        }
        catch (IOException e) {
            throw new DSSException(String.format("Cannot extract X500Principal! Reason : %s", e.getMessage()), (Throwable)e);
        }
    }

    public static CertificateIdentifier toCertificateIdentifier(X500Principal issuerX500Principal, BigInteger serialNumber, byte[] ski) {
        CertificateIdentifier certificateIdentifier = new CertificateIdentifier();
        certificateIdentifier.setIssuerName(issuerX500Principal);
        certificateIdentifier.setSerialNumber(serialNumber);
        certificateIdentifier.setSki(ski);
        return certificateIdentifier;
    }

    public static IssuerSerial getIssuerSerial(CertificateToken certToken) {
        X500Name issuerX500Name = DSSASN1Utils.getX509CertificateHolder(certToken).getIssuer();
        GeneralName generalName = new GeneralName(issuerX500Name);
        GeneralNames generalNames = new GeneralNames(generalName);
        BigInteger serialNumber = certToken.getCertificate().getSerialNumber();
        return new IssuerSerial(generalNames, serialNumber);
    }

    public static boolean x500PrincipalAreEquals(X500Principal firstX500Principal, X500Principal secondX500Principal) {
        if (firstX500Principal == null || secondX500Principal == null) {
            return false;
        }
        if (firstX500Principal.equals(secondX500Principal)) {
            return true;
        }
        Map<String, String> firstStringStringHashMap = DSSASN1Utils.get(firstX500Principal);
        Map<String, String> secondStringStringHashMap = DSSASN1Utils.get(secondX500Principal);
        return firstStringStringHashMap.entrySet().containsAll(secondStringStringHashMap.entrySet());
    }

    public static Map<String, String> get(X500Principal x500Principal) {
        ASN1Encodable[] asn1Encodables;
        HashMap<String, String> treeMap = new HashMap<String, String>();
        byte[] encoded = x500Principal.getEncoded();
        ASN1Sequence asn1Sequence = ASN1Sequence.getInstance((Object)encoded);
        for (ASN1Encodable asn1Encodable : asn1Encodables = asn1Sequence.toArray()) {
            DLSet dlSet = (DLSet)asn1Encodable;
            for (int ii = 0; ii < dlSet.size(); ++ii) {
                DLSequence dlSequence = (DLSequence)dlSet.getObjectAt(ii);
                if (dlSequence.size() != 2) {
                    throw new DSSException("The DLSequence must contains exactly 2 elements.");
                }
                ASN1Encodable asn1EncodableAttributeType = dlSequence.getObjectAt(0);
                String stringAttributeType = DSSASN1Utils.getString(asn1EncodableAttributeType);
                ASN1Encodable asn1EncodableAttributeValue = dlSequence.getObjectAt(1);
                String stringAttributeValue = DSSASN1Utils.getString(asn1EncodableAttributeValue);
                treeMap.put(stringAttributeType, stringAttributeValue);
            }
        }
        return treeMap;
    }

    public static X500Principal getNormalizedX500Principal(X500Principal x500Principal) {
        String utf8Name = DSSASN1Utils.getUtf8String(x500Principal);
        return new X500Principal(utf8Name);
    }

    public static String getUtf8String(X500Principal x500Principal) {
        byte[] encoded = x500Principal.getEncoded();
        ASN1Sequence asn1Sequence = ASN1Sequence.getInstance((Object)encoded);
        ASN1Encodable[] asn1Encodables = asn1Sequence.toArray();
        StringBuilder stringBuilder = new StringBuilder();
        for (int ii = asn1Encodables.length - 1; ii >= 0; --ii) {
            ASN1Encodable asn1Encodable = asn1Encodables[ii];
            DLSet dlSet = (DLSet)asn1Encodable;
            for (int jj = 0; jj < dlSet.size(); ++jj) {
                DLSequence dlSequence = (DLSequence)dlSet.getObjectAt(jj);
                if (dlSequence.size() != 2) {
                    throw new DSSException("The DLSequence must contains exactly 2 elements.");
                }
                ASN1Encodable attributeType = dlSequence.getObjectAt(0);
                ASN1Encodable attributeValue = dlSequence.getObjectAt(1);
                String string = DSSASN1Utils.getString(attributeValue);
                string = Rdn.escapeValue(string);
                if (stringBuilder.length() != 0) {
                    stringBuilder.append(',');
                }
                stringBuilder.append(attributeType).append('=').append(string);
            }
        }
        return stringBuilder.toString();
    }

    public static String getString(ASN1Encodable attributeValue) {
        String string;
        if (attributeValue instanceof ASN1String) {
            string = ((ASN1String)attributeValue).getString();
        } else if (attributeValue instanceof ASN1ObjectIdentifier) {
            string = ((ASN1ObjectIdentifier)attributeValue).getId();
        } else {
            LOG.error("!!!*******!!! This encoding is unknown: {}", (Object)attributeValue.getClass().getSimpleName());
            string = attributeValue.toString();
            LOG.error("!!!*******!!! value: {}", (Object)string);
        }
        return string;
    }

    public static String extractAttributeFromX500Principal(ASN1ObjectIdentifier identifier, X500PrincipalHelper principal) {
        RDN[] rdns;
        X500Name x500Name = X500Name.getInstance((Object)principal.getEncoded());
        for (RDN rdn : rdns = x500Name.getRDNs(identifier)) {
            if (rdn.isMultiValued()) {
                AttributeTypeAndValue[] typesAndValues;
                for (AttributeTypeAndValue typeAndValue : typesAndValues = rdn.getTypesAndValues()) {
                    if (!identifier.equals((ASN1Primitive)typeAndValue.getType())) continue;
                    return typeAndValue.getValue().toString();
                }
                continue;
            }
            AttributeTypeAndValue typeAndValue = rdn.getFirst();
            if (!identifier.equals((ASN1Primitive)typeAndValue.getType())) continue;
            return typeAndValue.getValue().toString();
        }
        return null;
    }

    public static String getSubjectCommonName(CertificateToken cert) {
        return DSSASN1Utils.extractAttributeFromX500Principal(BCStyle.CN, cert.getSubject());
    }

    public static String getHumanReadableName(CertificateToken cert) {
        return DSSASN1Utils.firstNotNull(cert, BCStyle.CN, BCStyle.GIVENNAME, BCStyle.SURNAME, BCStyle.NAME, BCStyle.PSEUDONYM, BCStyle.O, BCStyle.OU);
    }

    private static String firstNotNull(CertificateToken cert, ASN1ObjectIdentifier ... oids) {
        for (ASN1ObjectIdentifier oid : oids) {
            String value = DSSASN1Utils.extractAttributeFromX500Principal(oid, cert.getSubject());
            if (value == null) continue;
            return value;
        }
        return null;
    }

    public static SignerInformation getFirstSignerInformation(CMSSignedData cms) {
        Collection signers = cms.getSignerInfos().getSigners();
        if (signers.size() > 1) {
            LOG.warn("!!! The framework handles only one signer (SignerInformation) !!!");
        }
        return (SignerInformation)signers.iterator().next();
    }

    public static boolean isASN1SequenceTag(byte tagByte) {
        return 48 == tagByte;
    }

    public static Date getDate(ASN1Encodable encodable) {
        try {
            return Time.getInstance((Object)encodable).getDate();
        }
        catch (Exception e) {
            LOG.warn("Unable to retrieve the date {}", (Object)encodable, (Object)e);
            return null;
        }
    }

    public static boolean isEmpty(AttributeTable attributeTable) {
        return attributeTable == null || attributeTable.size() == 0;
    }

    public static AttributeTable emptyIfNull(AttributeTable original) {
        if (original == null) {
            return new AttributeTable(new Hashtable());
        }
        return original;
    }

    public static List<String> getExtendedKeyUsage(CertificateToken certToken) {
        try {
            return certToken.getCertificate().getExtendedKeyUsage();
        }
        catch (CertificateParsingException e) {
            LOG.warn("Unable to retrieve ExtendedKeyUsage : {}", (Object)e.getMessage());
            return Collections.emptyList();
        }
    }

    public static IssuerSerial getIssuerSerial(byte[] binaries) {
        IssuerSerial issuerSerial;
        ASN1InputStream is = new ASN1InputStream(binaries);
        try {
            ASN1Sequence seq = (ASN1Sequence)is.readObject();
            issuerSerial = IssuerSerial.getInstance((Object)seq);
        }
        catch (Throwable throwable) {
            try {
                try {
                    is.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                LOG.error("Unable to decode IssuerSerialV2 textContent '{}' : {}", new Object[]{Utils.toBase64((byte[])binaries), e.getMessage(), e});
                return null;
            }
        }
        is.close();
        return issuerSerial;
    }

    public static CertificateIdentifier toCertificateIdentifier(IssuerSerial issuerAndSerial) {
        if (issuerAndSerial == null) {
            return null;
        }
        try {
            ASN1Integer serialNumber;
            CertificateIdentifier certificateIdentifier = new CertificateIdentifier();
            GeneralNames gnames = issuerAndSerial.getIssuer();
            if (gnames != null) {
                GeneralName[] names = gnames.getNames();
                if (names.length == 1) {
                    certificateIdentifier.setIssuerName(new X500Principal(names[0].getName().toASN1Primitive().getEncoded("DER")));
                } else {
                    LOG.warn("More than one GeneralName");
                }
            }
            if ((serialNumber = issuerAndSerial.getSerial()) != null) {
                certificateIdentifier.setSerialNumber(serialNumber.getValue());
            }
            return certificateIdentifier;
        }
        catch (Exception e) {
            LOG.error("Unable to read the IssuerSerial object", (Throwable)e);
            return null;
        }
    }

    public static ASN1Sequence getAtsHashIndex(AttributeTable timestampUnsignedAttributes) {
        ASN1ObjectIdentifier atsHashIndexVersionIdentifier = DSSASN1Utils.getAtsHashIndexVersionIdentifier(timestampUnsignedAttributes);
        return DSSASN1Utils.getAtsHashIndexByVersion(timestampUnsignedAttributes, atsHashIndexVersionIdentifier);
    }

    public static ASN1Sequence getAtsHashIndexByVersion(AttributeTable timestampUnsignedAttributes, ASN1ObjectIdentifier atsHashIndexVersionIdentifier) {
        ASN1Set attrValues;
        Attribute atsHashIndexAttribute;
        if (timestampUnsignedAttributes != null && atsHashIndexVersionIdentifier != null && (atsHashIndexAttribute = timestampUnsignedAttributes.get(atsHashIndexVersionIdentifier)) != null && (attrValues = atsHashIndexAttribute.getAttrValues()) != null && attrValues.size() == 1) {
            return (ASN1Sequence)attrValues.getObjectAt(0).toASN1Primitive();
        }
        return null;
    }

    public static ASN1ObjectIdentifier getAtsHashIndexVersionIdentifier(AttributeTable timestampUnsignedAttributes) {
        if (timestampUnsignedAttributes != null) {
            Attributes attributes = timestampUnsignedAttributes.toASN1Structure();
            for (Attribute attribute : attributes.getAttributes()) {
                ASN1ObjectIdentifier attrType = attribute.getAttrType();
                if (!OID.id_aa_ATSHashIndex.equals((ASN1Primitive)attrType) && !OID.id_aa_ATSHashIndexV2.equals((ASN1Primitive)attrType) && !OID.id_aa_ATSHashIndexV3.equals((ASN1Primitive)attrType)) continue;
                LOG.debug("Unsigned attribute of type [{}] found in the timestamp.", (Object)attrType);
                return attrType;
            }
            LOG.warn("The timestamp unsignedAttributes does not contain ATSHashIndex!");
        }
        return null;
    }

    public static List<byte[]> getOctetStringForAtsHashIndex(Attribute attribute, ASN1ObjectIdentifier atsHashIndexVersionIdentifier) {
        ArrayList<byte[]> octets = new ArrayList<byte[]>();
        if (OID.id_aa_ATSHashIndexV3.equals((ASN1Primitive)atsHashIndexVersionIdentifier)) {
            byte[] attrType = DSSASN1Utils.getDEREncoded((ASN1Encodable)attribute.getAttrType());
            for (ASN1Encodable asn1Encodable : attribute.getAttrValues().toArray()) {
                octets.add(DSSUtils.concatenate(attrType, DSSASN1Utils.getDEREncoded(asn1Encodable)));
            }
        } else {
            octets.add(DSSASN1Utils.getDEREncoded((ASN1Encodable)attribute));
        }
        return octets;
    }

    public static ASN1Encodable getAsn1Encodable(AttributeTable unsignedAttributes, ASN1ObjectIdentifier oid) {
        ASN1Set attrValues = DSSASN1Utils.getAsn1AttributeSet(unsignedAttributes, oid);
        if (attrValues == null || attrValues.size() <= 0) {
            return null;
        }
        return attrValues.getObjectAt(0);
    }

    public static ASN1Set getAsn1AttributeSet(AttributeTable unsignedAttributes, ASN1ObjectIdentifier oid) {
        Attribute attribute = unsignedAttributes.get(oid);
        if (attribute == null) {
            return null;
        }
        return attribute.getAttrValues();
    }

    public static Attribute[] getAsn1Attributes(AttributeTable unsignedAttributes, ASN1ObjectIdentifier oid) {
        ASN1EncodableVector encodableVector = unsignedAttributes.getAll(oid);
        if (encodableVector == null) {
            return null;
        }
        Attributes attributes = new Attributes(encodableVector);
        return attributes.getAttributes();
    }

    public static List<TimeStampToken> findArchiveTimeStampTokens(AttributeTable unsignedAttributes) {
        Attribute[] attributes;
        ArrayList<TimeStampToken> timeStamps = new ArrayList<TimeStampToken>();
        for (Attribute attribute : attributes = unsignedAttributes.toASN1Structure().getAttributes()) {
            TimeStampToken timeStampToken;
            if (!DSSASN1Utils.isArchiveTimeStampToken(attribute) || (timeStampToken = DSSASN1Utils.getTimeStampToken(attribute)) == null) continue;
            timeStamps.add(timeStampToken);
        }
        return timeStamps;
    }

    public static boolean isArchiveTimeStampToken(Attribute attribute) {
        return DSSASN1Utils.isAttributeOfType(attribute, OID.id_aa_ets_archiveTimestampV2) || DSSASN1Utils.isAttributeOfType(attribute, OID.id_aa_ets_archiveTimestampV3);
    }

    public static boolean isAttributeOfType(Attribute attribute, ASN1ObjectIdentifier asn1ObjectIdentifier) {
        if (attribute == null) {
            return false;
        }
        ASN1ObjectIdentifier objectIdentifier = attribute.getAttrType();
        return asn1ObjectIdentifier.equals((ASN1Primitive)objectIdentifier);
    }

    public static TimeStampToken getTimeStampToken(Attribute attribute) {
        try {
            CMSSignedData signedData = DSSASN1Utils.getCMSSignedData(attribute);
            if (signedData != null) {
                return new TimeStampToken(signedData);
            }
        }
        catch (IOException | CMSException | TSPException e) {
            LOG.warn("The given TimeStampToken cannot be created! Reason: [{}]", (Object)e.getMessage());
        }
        return null;
    }

    public static CMSSignedData getCMSSignedData(Attribute attribute) throws CMSException, IOException {
        ASN1Encodable value = DSSASN1Utils.getAsn1Encodable(attribute);
        if (!(value instanceof DEROctetString)) {
            ASN1Primitive asn1Primitive = value.toASN1Primitive();
            return new CMSSignedData(asn1Primitive.getEncoded());
        }
        LOG.warn("Illegal content for CMSSignedData (OID : {}) : OCTET STRING is not allowed !", (Object)attribute.getAttrType());
        return null;
    }

    public static ASN1Encodable getAsn1Encodable(Attribute attribute) {
        return attribute.getAttrValues().getObjectAt(0);
    }

    public static Date getTimeStampTokenGenerationTime(TimeStampToken timeStampToken) {
        if (timeStampToken != null) {
            return timeStampToken.getTimeStampInfo().getGenTime();
        }
        return null;
    }

    public static RevocationValues getRevocationValues(ASN1Encodable encodable) {
        if (encodable != null) {
            try {
                return RevocationValues.getInstance((Object)encodable);
            }
            catch (Exception e) {
                LOG.warn("Unable to parse RevocationValues", (Throwable)e);
            }
        }
        return null;
    }

    public static CertificateRef getCertificateRef(OtherCertID otherCertId) {
        CertificateRef certRef = new CertificateRef();
        DigestAlgorithm digestAlgo = DigestAlgorithm.forOID((String)otherCertId.getAlgorithmHash().getAlgorithm().getId());
        certRef.setCertDigest(new Digest(digestAlgo, otherCertId.getCertHash()));
        certRef.setCertificateIdentifier(DSSASN1Utils.toCertificateIdentifier(otherCertId.getIssuerSerial()));
        return certRef;
    }

    public static PSD2QcType getPSD2QcStatement(CertificateToken certToken) {
        PSD2QcType result = null;
        byte[] qcStatement = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (Utils.isArrayNotEmpty((byte[])qcStatement)) {
            try {
                ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatement);
                for (int i = 0; i < seq.size(); ++i) {
                    QCStatement statement = QCStatement.getInstance((Object)seq.getObjectAt(i));
                    if (!OID.psd2_qcStatement.equals((ASN1Primitive)statement.getStatementId())) continue;
                    result = new PSD2QcType();
                    ASN1Sequence psd2Seq = ASN1Sequence.getInstance((Object)statement.getStatementInfo());
                    ASN1Sequence rolesSeq = ASN1Sequence.getInstance((Object)psd2Seq.getObjectAt(0));
                    ArrayList<RoleOfPSP> rolesOfPSP = new ArrayList<RoleOfPSP>();
                    for (int ii = 0; ii < rolesSeq.size(); ++ii) {
                        ASN1Sequence oneRoleSeq = ASN1Sequence.getInstance((Object)rolesSeq.getObjectAt(ii));
                        RoleOfPSP roleOfPSP = new RoleOfPSP();
                        ASN1ObjectIdentifier oid = (ASN1ObjectIdentifier)oneRoleSeq.getObjectAt(0);
                        roleOfPSP.setPspOid(RoleOfPspOid.fromOid((String)oid.getId()));
                        roleOfPSP.setPspName(DSSASN1Utils.getString(oneRoleSeq.getObjectAt(1)));
                        rolesOfPSP.add(roleOfPSP);
                    }
                    result.setRolesOfPSP(rolesOfPSP);
                    result.setNcaName(DSSASN1Utils.getString(psd2Seq.getObjectAt(1)));
                    result.setNcaId(DSSASN1Utils.getString(psd2Seq.getObjectAt(2)));
                }
            }
            catch (Exception e) {
                LOG.warn("Unable to read QCStatement", (Throwable)e);
            }
        }
        return result;
    }

    public static List<String> getSubjectAlternativeNames(CertificateToken certToken) {
        ArrayList<String> result = new ArrayList<String>();
        try {
            Collection<List<?>> subjectAlternativeNames = certToken.getCertificate().getSubjectAlternativeNames();
            if (Utils.isCollectionNotEmpty(subjectAlternativeNames)) {
                for (List<?> list : subjectAlternativeNames) {
                    if (Utils.collectionSize(list) != 2) continue;
                    Object value = list.get(1);
                    if (value instanceof String) {
                        result.add((String)value);
                        continue;
                    }
                    LOG.trace("Ignored value : {}", value);
                }
            }
        }
        catch (Exception e) {
            LOG.warn("Unable to extract SubjectAlternativeNames", (Throwable)e);
        }
        return result;
    }

    public static SemanticsIdentifier getSemanticsIdentifier(CertificateToken certToken) {
        byte[] qcStatement = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (Utils.isArrayNotEmpty((byte[])qcStatement)) {
            try {
                ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatement);
                for (int i = 0; i < seq.size(); ++i) {
                    SemanticsInformation semanticsInfo;
                    QCStatement statement = QCStatement.getInstance((Object)seq.getObjectAt(i));
                    if (!RFC3739QCObjectIdentifiers.id_qcs_pkixQCSyntax_v2.equals((ASN1Primitive)statement.getStatementId()) || (semanticsInfo = SemanticsInformation.getInstance((Object)statement.getStatementInfo())) == null || semanticsInfo.getSemanticsIdentifier() == null) continue;
                    return SemanticsIdentifier.fromOid((String)semanticsInfo.getSemanticsIdentifier().getId());
                }
            }
            catch (Exception e) {
                LOG.warn("Unable to extract the SemanticsIdentifier", (Throwable)e);
            }
        }
        return null;
    }

    static {
        Security.addProvider(DSSSecurityProvider.getSecurityProvider());
    }
}

