/**
 * DSS - Digital Signature Services
 * Copyright (C) 2015 European Commission, provided under the CEF programme
 * 
 * This file is part of the "DSS - Digital Signature Services" project.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package eu.europa.esig.dss.validation.reports;

import java.io.InputStream;
import java.io.StringWriter;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import eu.europa.esig.dss.validation.reports.wrapper.DiagnosticData;

/**
 * This class is a container for all reports generated by the validation
 * process: diagnostic data, detailed report and simple report.
 */
public abstract class AbstractReports {

	private static final Logger LOG = LoggerFactory.getLogger(AbstractReports.class);

	private boolean validateXml = false;

	/**
	 * This variable contains the reference to the diagnostic data object.
	 */
	protected eu.europa.esig.dss.jaxb.diagnostic.DiagnosticData diagnosticData;

	protected DiagnosticData diagnosticDataWrapper;

	/**
	 * This is the detailed report of the validation.
	 */
	protected eu.europa.esig.dss.jaxb.detailedreport.DetailedReport detailedReport;

	protected DetailedReport detailedReportWrapper;

	private String xmlDiagnosticData;
	private String xmlDetailedReport;

	/**
	 * This is the default constructor to instantiate this container.
	 *
	 * @param diagnosticDataJaxb
	 *            the JAXB {@code DiagnosticData}
	 * @param detailedReport
	 *            the JAXB {@code DetailedReport}
	 */
	protected AbstractReports(final eu.europa.esig.dss.jaxb.diagnostic.DiagnosticData diagnosticDataJaxb,
			final eu.europa.esig.dss.jaxb.detailedreport.DetailedReport detailedReport) {
		this.diagnosticData = diagnosticDataJaxb;
		this.diagnosticDataWrapper = new DiagnosticData(diagnosticDataJaxb);
		this.detailedReport = detailedReport;
		this.detailedReportWrapper = new DetailedReport(detailedReport);
	}

	public void setValidateXml(boolean validateXml) {
		this.validateXml = validateXml;
	}

	/**
	 * This method returns the reference to the diagnostic data object generated
	 * during the validation process.
	 *
	 * @return the wrapper {@code DiagnosticData}
	 */
	public DiagnosticData getDiagnosticData() {
		return diagnosticDataWrapper;
	}

	/**
	 * This method returns the wrapper to manipulate the JAXB DetailedReport
	 * 
	 * @return the wrapper {@code DetailedReport}
	 */
	public DetailedReport getDetailedReport() {
		return detailedReportWrapper;
	}

	/**
	 * This method returns the JAXB DiagnosticData
	 * 
	 * @return the JAXB {@code DiagnosticData}
	 */
	public eu.europa.esig.dss.jaxb.diagnostic.DiagnosticData getDiagnosticDataJaxb() {
		return diagnosticData;
	}

	/**
	 * This method returns the JAXB DetailedReport
	 * 
	 * @return the JAXB {@code DetailedReport}
	 */
	public eu.europa.esig.dss.jaxb.detailedreport.DetailedReport getDetailedReportJaxb() {
		return detailedReport;
	}

	/**
	 * For debug purpose.
	 */
	public void print() {
		System.out.println("----------------Diagnostic data-----------------");
		System.out.println(getXmlDiagnosticData());
		System.out.println("----------------Validation report---------------");
		System.out.println(getXmlDetailedReport());
		System.out.println("----------------Simple report-------------------");
		System.out.println(getXmlSimpleReport());
		System.out.println("------------------------------------------------");
	}

	public abstract String getXmlSimpleReport();

	/**
	 * This method returns the XML representation of the JAXB DiagnosticData String
	 * 
	 * @return a String with the XML content of the JAXB {@code DiagnosticData}
	 */
	public String getXmlDiagnosticData() {
		if (xmlDiagnosticData == null) {
			xmlDiagnosticData = getJAXBObjectAsString(diagnosticData, eu.europa.esig.dss.jaxb.diagnostic.DiagnosticData.class.getPackage().getName(),
					"/xsd/DiagnosticData.xsd");
		}
		return xmlDiagnosticData;
	}

	/**
	 * This method returns the XML representation of the JAXB DetailedReport String
	 * 
	 * @return a String with the XML content of the JAXB {@code DetailedReport}
	 */
	public String getXmlDetailedReport() {
		if (xmlDetailedReport == null) {
			xmlDetailedReport = getJAXBObjectAsString(detailedReport, eu.europa.esig.dss.jaxb.detailedreport.DetailedReport.class.getPackage().getName(),
					"/xsd/DetailedReport.xsd");
		}
		return xmlDetailedReport;
	}

	protected String getJAXBObjectAsString(Object obj, String contextPath, String xsdFile) {
		try {

			JAXBContext context = JAXBContext.newInstance(contextPath);
			Marshaller marshaller = context.createMarshaller();
			marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);

			if (validateXml) {
				SchemaFactory sf = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
				InputStream schemaStream = AbstractReports.class.getResourceAsStream(xsdFile);
				Schema schema = sf.newSchema(new StreamSource(schemaStream));
				marshaller.setSchema(schema);
			}

			StringWriter writer = new StringWriter();
			marshaller.marshal(obj, writer);
			return writer.toString();
		} catch (Exception e) {
			if (validateXml) {
				throw new RuntimeException(e);
			} else {
				LOG.error("Unable to generate string value for context " + contextPath + " : " + e.getMessage(), e);
				return null;
			}
		}
	}

}
