/*
 *  ******************************************************************************
 *  *
 *  *
 *  * This program and the accompanying materials are made available under the
 *  * terms of the Apache License, Version 2.0 which is available at
 *  * https://www.apache.org/licenses/LICENSE-2.0.
 *  *
 *  *  See the NOTICE file distributed with this work for additional
 *  *  information regarding copyright ownership.
 *  * Unless required by applicable law or agreed to in writing, software
 *  * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 *  * License for the specific language governing permissions and limitations
 *  * under the License.
 *  *
 *  * SPDX-License-Identifier: Apache-2.0
 *  *****************************************************************************
 */

package org.datavec.local.transforms.misc;

import lombok.AllArgsConstructor;
import org.datavec.api.writable.Writable;
import org.nd4j.common.function.Function;

import java.util.List;

@AllArgsConstructor
public class SequenceWritablesToStringFunction implements Function<List<List<Writable>>, String> {

    public static final String DEFAULT_DELIMITER = ",";
    public static final String DEFAULT_TIME_STEP_DELIMITER = "\n";

    private final String delimiter;
    private final String timeStepDelimiter;
    private final String quote;

    /**
     * Function with default delimiters ("," and "\n")
     */
    public SequenceWritablesToStringFunction() {
        this(DEFAULT_DELIMITER);
    }

    /**
     * function with default delimiter ("\n") between time steps
     * @param delim Delimiter between values within a single time step
     */
    public SequenceWritablesToStringFunction(String delim) {
        this(delim, DEFAULT_TIME_STEP_DELIMITER, null);
    }

    /**
     *
     * @param delim             The delimiter between column values in a single time step (for example, ",")
     * @param timeStepDelimiter The delimiter between time steps (for example, "\n")
     */
    public SequenceWritablesToStringFunction(String delim, String timeStepDelimiter) {
        this(delim, timeStepDelimiter, null);
    }

    @Override
    public String apply(List<List<Writable>> c) {

        StringBuilder sb = new StringBuilder();
        boolean firstLine = true;
        for (List<Writable> timeStep : c) {
            if (!firstLine) {
                sb.append(timeStepDelimiter);
            }
            WritablesToStringFunction.append(timeStep, sb, delimiter, quote);
            firstLine = false;
        }

        return sb.toString();
    }
}
