/*
 * Decompiled with CFR 0.152.
 */
package org.datatransferproject.transfer.microsoft.media;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.api.client.auth.oauth2.Credential;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.net.URL;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.apache.commons.lang3.tuple.Pair;
import org.datatransferproject.api.launcher.Monitor;
import org.datatransferproject.spi.cloud.storage.TemporaryPerJobDataStore;
import org.datatransferproject.spi.transfer.idempotentexecutor.IdempotentImportExecutor;
import org.datatransferproject.spi.transfer.provider.ImportResult;
import org.datatransferproject.spi.transfer.provider.Importer;
import org.datatransferproject.spi.transfer.types.CopyExceptionWithFailureReason;
import org.datatransferproject.spi.transfer.types.DestinationMemoryFullException;
import org.datatransferproject.spi.transfer.types.PermissionDeniedException;
import org.datatransferproject.transfer.microsoft.DataChunk;
import org.datatransferproject.transfer.microsoft.MicrosoftTransmogrificationConfig;
import org.datatransferproject.transfer.microsoft.common.MicrosoftCredentialFactory;
import org.datatransferproject.types.common.DownloadableFile;
import org.datatransferproject.types.common.models.TransmogrificationConfig;
import org.datatransferproject.types.common.models.media.MediaAlbum;
import org.datatransferproject.types.common.models.media.MediaContainerResource;
import org.datatransferproject.types.transfer.auth.TokensAndUrlAuthData;

public class MicrosoftMediaImporter
implements Importer<TokensAndUrlAuthData, MediaContainerResource> {
    private final OkHttpClient client;
    private final ObjectMapper objectMapper;
    private final TemporaryPerJobDataStore jobStore;
    private final Monitor monitor;
    private final MicrosoftCredentialFactory credentialFactory;
    private final MicrosoftTransmogrificationConfig transmogrificationConfig = new MicrosoftTransmogrificationConfig();
    private Credential credential;
    private final String createFolderUrl;
    private final String uploadMediaUrlTemplate;
    private final String albumlessMediaUrlTemplate;
    private static final String UPLOAD_PARAMS = "?@microsoft.graph.conflictBehavior=rename";

    public MicrosoftMediaImporter(String baseUrl, OkHttpClient client, ObjectMapper objectMapper, TemporaryPerJobDataStore jobStore, Monitor monitor, MicrosoftCredentialFactory credentialFactory) {
        this.createFolderUrl = baseUrl + "/v1.0/me/drive/special/photos/children";
        this.albumlessMediaUrlTemplate = baseUrl + "/v1.0/me/drive/special/photos:/%s:/createUploadSession%s";
        this.uploadMediaUrlTemplate = baseUrl + "/v1.0/me/drive/items/%s:/%s:/createUploadSession%s";
        this.client = client;
        this.objectMapper = objectMapper;
        this.jobStore = jobStore;
        this.monitor = monitor;
        this.credentialFactory = credentialFactory;
        this.credential = null;
    }

    public ImportResult importItem(UUID jobId, IdempotentImportExecutor idempotentImportExecutor, TokensAndUrlAuthData authData, MediaContainerResource resource) throws Exception {
        this.getOrCreateCredential(authData);
        this.logDebugJobStatus("%s before transmogrification", jobId, resource);
        resource.transmogrify((TransmogrificationConfig)this.transmogrificationConfig);
        this.logDebugJobStatus("%s after transmogrification", jobId, resource);
        for (MediaAlbum album : resource.getAlbums()) {
            idempotentImportExecutor.executeAndSwallowIOExceptions(album.getId(), album.getName(), () -> this.createOneDriveFolder(album));
        }
        this.executeIdempotentImport(jobId, idempotentImportExecutor, resource.getVideos());
        this.executeIdempotentImport(jobId, idempotentImportExecutor, resource.getPhotos());
        return ImportResult.OK;
    }

    private void logDebugJobStatus(String format, UUID jobId, MediaContainerResource resource) {
        String statusMessage = String.format("%s: Importing %s albums, %s photos, and %s videos", jobId, resource.getAlbums().size(), resource.getPhotos().size(), resource.getVideos().size());
        this.monitor.debug(() -> String.format(format, statusMessage), new Object[0]);
    }

    private String createOneDriveFolder(MediaAlbum album) throws IOException, CopyExceptionWithFailureReason {
        LinkedHashMap<String, Object> rawFolder = new LinkedHashMap<String, Object>();
        String albumName = Strings.isNullOrEmpty((String)album.getName()) ? "Untitled" : album.getName();
        rawFolder.put("name", albumName);
        rawFolder.put("folder", new LinkedHashMap());
        rawFolder.put("@microsoft.graph.conflictBehavior", "rename");
        Request.Builder requestBuilder = new Request.Builder().url(this.createFolderUrl);
        requestBuilder.header("Authorization", "Bearer " + this.credential.getAccessToken());
        requestBuilder.post(RequestBody.create((MediaType)MediaType.parse((String)"application/json"), (String)this.objectMapper.writeValueAsString(rawFolder)));
        try (Response response = this.client.newCall(requestBuilder.build()).execute();){
            int code = response.code();
            ResponseBody body = response.body();
            if (code == 401) {
                this.credentialFactory.refreshCredential(this.credential);
                this.monitor.info(() -> "Refreshed authorization token successfuly", new Object[0]);
                requestBuilder.header("Authorization", "Bearer " + this.credential.getAccessToken());
                Response newResponse = this.client.newCall(requestBuilder.build()).execute();
                code = newResponse.code();
                body = newResponse.body();
            }
            if (code == 403 && response.message().contains("Access Denied")) {
                throw new PermissionDeniedException("User access to microsoft onedrive was denied", (Throwable)new IOException(String.format("Got error code %d  with message: %s", code, response.message())));
            }
            if (code < 200 || code > 299) {
                throw new IOException("Got error code: " + code + " message: " + response.message() + " body: " + response.body().string());
            }
            if (body == null) {
                throw new IOException("Got null body");
            }
            Map responseData = (Map)this.objectMapper.readValue(body.bytes(), Map.class);
            String folderId = (String)responseData.get("id");
            Preconditions.checkState((!Strings.isNullOrEmpty((String)folderId) ? 1 : 0) != 0, (String)"Expected id value to be present in %s", (Object)responseData);
            String string = folderId;
            return string;
        }
    }

    private void executeIdempotentImport(UUID jobId, IdempotentImportExecutor idempotentImportExecutor, Collection<? extends DownloadableFile> downloadableFiles) throws Exception {
        for (DownloadableFile downloadableFile : downloadableFiles) {
            idempotentImportExecutor.executeAndSwallowIOExceptions(downloadableFile.getIdempotentId(), downloadableFile.getName(), () -> this.importDownloadableItem(downloadableFile, jobId, idempotentImportExecutor));
        }
    }

    private String importDownloadableItem(DownloadableFile item, UUID jobId, IdempotentImportExecutor idempotentImportExecutor) throws Exception {
        BufferedInputStream inputStream = null;
        if (item.isInTempStore()) {
            inputStream = new BufferedInputStream(this.jobStore.getStream(jobId, item.getFetchableUrl()).getStream());
        } else if (item.getFetchableUrl() != null) {
            inputStream = new BufferedInputStream(new URL(item.getFetchableUrl()).openStream());
        } else {
            throw new IllegalStateException("Don't know how to get the inputStream for " + item);
        }
        String itemUploadUrl = this.createUploadSession(item, idempotentImportExecutor);
        List<DataChunk> chunksToSend = DataChunk.splitData(inputStream);
        inputStream.close();
        int totalFileSize = chunksToSend.stream().map(DataChunk::getSize).reduce(0, Integer::sum);
        Preconditions.checkState((chunksToSend.size() != 0 ? 1 : 0) != 0, (String)"Data was split into zero chunks %s.", (Object)item.getName());
        Response chunkResponse = null;
        for (DataChunk chunk : chunksToSend) {
            chunkResponse = this.uploadChunk(chunk, itemUploadUrl, totalFileSize, item.getMimeType());
        }
        if (chunkResponse.code() != 200 && chunkResponse.code() != 201) {
            this.monitor.debug(() -> "Received a bad code on completion of uploading chunks", new Object[]{chunkResponse.code()});
        }
        ResponseBody chunkResponseBody = chunkResponse.body();
        Map chunkResponseData = (Map)this.objectMapper.readValue(chunkResponseBody.bytes(), Map.class);
        return (String)chunkResponseData.get("id");
    }

    private Credential getOrCreateCredential(TokensAndUrlAuthData authData) {
        if (this.credential == null) {
            this.credential = this.credentialFactory.createCredential(authData);
        }
        return this.credential;
    }

    private Pair<Request, Response> tryWithCreds(Request.Builder requestBuilder) throws IOException {
        Request request = requestBuilder.build();
        Response response = this.client.newCall(request).execute();
        if (response.code() != 401) {
            return Pair.of((Object)request, (Object)response);
        }
        this.credentialFactory.refreshCredential(this.credential);
        this.monitor.info(() -> "Refreshed authorization token successfuly", new Object[0]);
        requestBuilder.header("Authorization", "Bearer " + this.credential.getAccessToken());
        request = requestBuilder.build();
        return Pair.of((Object)request, (Object)this.client.newCall(request).execute());
    }

    private String createUploadSession(DownloadableFile item, IdempotentImportExecutor idempotentImportExecutor) throws IOException, CopyExceptionWithFailureReason {
        ResponseBody responseBody;
        Request.Builder createSessionRequestBuilder = this.buildCreateUploadSessionPath(item, idempotentImportExecutor);
        createSessionRequestBuilder.header("Authorization", "Bearer " + this.credential.getAccessToken());
        createSessionRequestBuilder.header("Content-Type", "application/json");
        createSessionRequestBuilder.post(RequestBody.create((MediaType)MediaType.parse((String)"application/json"), (String)this.objectMapper.writeValueAsString((Object)ImmutableMap.of())));
        Pair<Request, Response> reqResp = this.tryWithCreds(createSessionRequestBuilder);
        Response response = (Response)reqResp.getRight();
        int code = response.code();
        if (code == 403 && response.message().contains("Access Denied")) {
            throw new PermissionDeniedException("User access to Microsoft One Drive was denied", (Throwable)new IOException(String.format("Got error code %d  with message: %s", code, response.message())));
        }
        if (code == 507 && response.message().contains("Insufficient Storage")) {
            throw new DestinationMemoryFullException("Microsoft destination storage limit reached", (Throwable)new IOException(String.format("Got error code %d  with message: %s", code, response.message())));
        }
        if (code < 200 || code > 299) {
            throw new IOException(String.format("Got error code: %s\nmessage: %s\nbody: %s\nrequest url: %s\nbearer token: %s\n item: %s\n", code, response.message(), response.body().string(), ((Request)reqResp.getLeft()).url(), this.credential.getAccessToken(), item));
        }
        if (code != 200) {
            this.monitor.info(() -> String.format("Got an unexpected non-200, non-error response code", new Object[0]), new Object[0]);
        }
        Preconditions.checkState(((responseBody = response.body()) != null ? 1 : 0) != 0, (String)"Got Null Body when creating item upload session %s", (Object)item);
        Map responseData = (Map)this.objectMapper.readValue(responseBody.bytes(), Map.class);
        Preconditions.checkState((boolean)responseData.containsKey("uploadUrl"), (Object)"No uploadUrl :(");
        return (String)responseData.get("uploadUrl");
    }

    private Request.Builder buildCreateUploadSessionPath(DownloadableFile item, IdempotentImportExecutor idempotentImportExecutor) {
        String createSessionUrl;
        if (Strings.isNullOrEmpty((String)item.getFolderId())) {
            createSessionUrl = String.format(this.albumlessMediaUrlTemplate, item.getName(), UPLOAD_PARAMS);
        } else {
            String oneDriveFolderId = (String)((Object)idempotentImportExecutor.getCachedValue(item.getFolderId()));
            createSessionUrl = String.format(this.uploadMediaUrlTemplate, oneDriveFolderId, item.getName(), UPLOAD_PARAMS);
        }
        return new Request.Builder().url(createSessionUrl);
    }

    private Response uploadChunk(DataChunk chunk, String photoUploadUrl, int totalFileSize, String mediaType) throws IOException, DestinationMemoryFullException {
        int chunkCode;
        Request.Builder uploadRequestBuilder = new Request.Builder().url(photoUploadUrl);
        uploadRequestBuilder.header("Authorization", "Bearer " + this.credential.getAccessToken());
        RequestBody uploadChunkBody = RequestBody.create((MediaType)MediaType.parse((String)mediaType), (byte[])chunk.getData(), (int)0, (int)chunk.getSize());
        uploadRequestBuilder.put(uploadChunkBody);
        String contentRange = String.format("bytes %d-%d/%d", chunk.getStart(), chunk.getEnd(), totalFileSize);
        uploadRequestBuilder.header("Content-Range", contentRange);
        uploadRequestBuilder.header("Content-Length", String.format("%d", chunk.getSize()));
        Response chunkResponse = this.client.newCall(uploadRequestBuilder.build()).execute();
        Preconditions.checkNotNull((Object)chunkResponse, (Object)"chunkResponse is null");
        if (chunkResponse.code() == 401) {
            this.credentialFactory.refreshCredential(this.credential);
            this.monitor.info(() -> "Refreshed authorization token successfuly", new Object[0]);
            uploadRequestBuilder.header("Authorization", "Bearer " + this.credential.getAccessToken());
            chunkResponse = this.client.newCall(uploadRequestBuilder.build()).execute();
        }
        if ((chunkCode = chunkResponse.code()) == 507 && chunkResponse.message().contains("Insufficient Storage")) {
            throw new DestinationMemoryFullException("Microsoft destination storage limit reached", (Throwable)new IOException(String.format("Got error code %d  with message: %s", chunkCode, chunkResponse.message())));
        }
        if (chunkCode < 200 || chunkCode > 299) {
            throw new IOException("Got error code: " + chunkCode + " message: " + chunkResponse.message() + " body: " + chunkResponse.body().string());
        }
        if (chunkCode == 200 || chunkCode == 201 || chunkCode == 202) {
            this.monitor.info(() -> String.format("Uploaded chunk %s-%s successfuly, code %d", chunk.getStart(), chunk.getEnd(), chunkCode), new Object[0]);
        }
        return chunkResponse;
    }
}

