/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.cryptofs.fh;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.RemovalCause;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.NonWritableChannelException;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.inject.Inject;
import org.cryptomator.cryptofs.CryptoFileSystemStats;
import org.cryptomator.cryptofs.fh.BufferPool;
import org.cryptomator.cryptofs.fh.Chunk;
import org.cryptomator.cryptofs.fh.ChunkLoader;
import org.cryptomator.cryptofs.fh.ChunkSaver;
import org.cryptomator.cryptofs.fh.ExceptionsDuringWrite;
import org.cryptomator.cryptofs.fh.OpenFileScoped;
import org.cryptomator.cryptolib.api.AuthenticationFailedException;

@OpenFileScoped
public class ChunkCache {
    public static final int MAX_CACHED_CLEARTEXT_CHUNKS = 5;
    private final ChunkLoader chunkLoader;
    private final ChunkSaver chunkSaver;
    private final CryptoFileSystemStats stats;
    private final BufferPool bufferPool;
    private final ExceptionsDuringWrite exceptionsDuringWrite;
    private final Cache<Long, Chunk> chunkCache;
    private final ConcurrentMap<Long, Chunk> cachedChunks;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final Lock sharedLock = this.lock.readLock();
    private final Lock exclusiveLock = this.lock.writeLock();

    @Inject
    public ChunkCache(ChunkLoader chunkLoader, ChunkSaver chunkSaver, CryptoFileSystemStats stats, BufferPool bufferPool, ExceptionsDuringWrite exceptionsDuringWrite) {
        this.chunkLoader = chunkLoader;
        this.chunkSaver = chunkSaver;
        this.stats = stats;
        this.bufferPool = bufferPool;
        this.exceptionsDuringWrite = exceptionsDuringWrite;
        this.chunkCache = Caffeine.newBuilder().maximumWeight(5L).weigher(this::weigh).executor(Runnable::run).evictionListener(this::evictStaleChunk).build();
        this.cachedChunks = this.chunkCache.asMap();
    }

    private int weigh(Long index, Chunk chunk) {
        if (chunk.currentAccesses().get() > 0) {
            return 0;
        }
        return 1;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Chunk putChunk(long chunkIndex, ByteBuffer chunkData) throws IllegalArgumentException {
        this.sharedLock.lock();
        try {
            Chunk chunk2 = this.cachedChunks.compute(chunkIndex, (index, chunk) -> {
                if (chunk == null) {
                    chunk = new Chunk(chunkData, true, () -> this.releaseChunk(chunkIndex));
                } else {
                    ByteBuffer dst = chunk.data().clear();
                    Preconditions.checkArgument((chunkData.remaining() == dst.remaining() ? 1 : 0) != 0);
                    dst.put(chunkData).flip();
                    chunk.dirty().set(true);
                }
                chunk.currentAccesses().incrementAndGet();
                return chunk;
            });
            return chunk2;
        }
        finally {
            this.sharedLock.unlock();
        }
    }

    public Chunk getChunk(long chunkIndex) throws IOException {
        this.sharedLock.lock();
        try {
            this.stats.addChunkCacheAccess();
            try {
                Chunk chunk2 = this.cachedChunks.compute(chunkIndex, (idx, chunk) -> {
                    if (chunk == null) {
                        chunk = this.loadChunk((long)idx);
                    }
                    chunk.currentAccesses().incrementAndGet();
                    return chunk;
                });
                return chunk2;
            }
            catch (UncheckedIOException | AuthenticationFailedException e) {
                throw new IOException(e);
            }
        }
        finally {
            this.sharedLock.unlock();
        }
    }

    private Chunk loadChunk(long chunkIndex) throws AuthenticationFailedException, UncheckedIOException {
        this.stats.addChunkCacheMiss();
        try {
            return new Chunk(this.chunkLoader.load(chunkIndex), false, () -> this.releaseChunk(chunkIndex));
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void releaseChunk(long chunkIndex) {
        this.sharedLock.lock();
        try {
            this.cachedChunks.computeIfPresent(chunkIndex, (idx, chunk) -> {
                chunk.currentAccesses().decrementAndGet();
                return chunk;
            });
        }
        finally {
            this.sharedLock.unlock();
        }
    }

    public void flush() throws IOException {
        this.exclusiveLock.lock();
        try {
            this.cachedChunks.forEach((index, chunk) -> {
                try {
                    this.chunkSaver.save((long)index, (Chunk)chunk);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            });
        }
        catch (UncheckedIOException e) {
            throw new IOException(e);
        }
        finally {
            this.exclusiveLock.unlock();
        }
    }

    public void invalidateStale() {
        this.exclusiveLock.lock();
        try {
            this.cachedChunks.entrySet().removeIf(entry -> ((Chunk)entry.getValue()).currentAccesses().get() == 0);
        }
        finally {
            this.exclusiveLock.unlock();
        }
    }

    void evictStaleChunk(Long index, Chunk chunk, RemovalCause removalCause) {
        assert (removalCause != RemovalCause.EXPLICIT);
        assert (chunk.currentAccesses().get() == 0);
        try {
            this.chunkSaver.save(index, chunk);
        }
        catch (IOException | NonWritableChannelException e) {
            this.exceptionsDuringWrite.add(e);
        }
        this.bufferPool.recycle(chunk.data());
    }
}

