/*
 * Decompiled with CFR 0.152.
 */
package org.cognitor.cassandra.migration;

import com.datastax.driver.core.BoundStatement;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ConsistencyLevel;
import com.datastax.driver.core.KeyspaceMetadata;
import com.datastax.driver.core.Metadata;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.SimpleStatement;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.TableMetadata;
import com.datastax.driver.core.VersionNumber;
import com.datastax.driver.core.exceptions.DriverException;
import com.datastax.driver.core.exceptions.InvalidQueryException;
import java.io.Closeable;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Date;
import java.util.Optional;
import java.util.UUID;
import org.cognitor.cassandra.migration.DbMigration;
import org.cognitor.cassandra.migration.MigrationException;
import org.cognitor.cassandra.migration.cql.SimpleCQLLexer;
import org.cognitor.cassandra.migration.keyspace.Keyspace;
import org.cognitor.cassandra.migration.util.Ensure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Database
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(Database.class);
    private static final String SCHEMA_CF = "schema_migration";
    private static final String SCHEMA_LEADER_CF = "schema_migration_leader";
    private static final String INSERT_MIGRATION = "insert into %s(applied_successful, version, script_name, script, executed_at) values(?, ?, ?, ?, ?)";
    private static final String CREATE_MIGRATION_CF = "CREATE TABLE %s %s (applied_successful boolean, version int, script_name varchar, script text, executed_at timestamp, PRIMARY KEY (applied_successful, version))";
    private static final String CREATE_LEADER_CF = "CREATE TABLE %s %s (keyspace_name text, leader uuid, took_lead_at timestamp, leader_hostname text, PRIMARY KEY (keyspace_name))";
    private static final String VERSION_QUERY = "select version from %s where applied_successful = True order by version desc limit 1";
    private static final String TAKE_LEAD_QUERY = "INSERT INTO %s (keyspace_name, leader, took_lead_at, leader_hostname) VALUES (?, ?, dateOf(now()), ?) IF NOT EXISTS USING TTL %s";
    private static final String RELEASE_LEAD_QUERY = "DELETE FROM %s where keyspace_name = ? IF leader = ?";
    private static final String MIGRATION_ERROR_MSG = "Error during migration of script %s while executing '%s'";
    private static final int LEAD_TTL = 300;
    private static final int TAKE_LEAD_WAIT_TIME = 10000;
    private final UUID instanceId = UUID.randomUUID();
    private final String instanceAddress;
    private final String tableName;
    private final String leaderTableName;
    private final String keyspaceName;
    private final Keyspace keyspace;
    private final Cluster cluster;
    private final Session session;
    private ConsistencyLevel consistencyLevel = ConsistencyLevel.QUORUM;
    private final PreparedStatement logMigrationStatement;
    private final PreparedStatement takeMigrationLeadStatement;
    private final PreparedStatement releaseMigrationLeadStatement;
    private final VersionNumber cassandraVersion;
    private boolean tookLead = false;

    public Database(Cluster cluster, Keyspace keyspace) {
        this(cluster, keyspace, "");
    }

    public Database(Cluster cluster, Keyspace keyspace, String tablePrefix) {
        this(cluster, keyspace, null, tablePrefix);
    }

    public Database(Cluster cluster, String keyspaceName) {
        this(cluster, keyspaceName, "");
    }

    public Database(Cluster cluster, String keyspaceName, String tablePrefix) {
        this(cluster, null, keyspaceName, tablePrefix);
    }

    private Database(Cluster cluster, Keyspace keyspace, String keyspaceName, String tablePrefix) {
        String tmpInstanceAddress;
        this.cluster = Ensure.notNull(cluster, "cluster");
        this.keyspace = keyspace;
        this.keyspaceName = Optional.ofNullable(keyspace).map(Keyspace::getKeyspaceName).orElse(keyspaceName);
        this.tableName = Database.createTableName(tablePrefix, SCHEMA_CF);
        this.leaderTableName = Database.createTableName(tablePrefix, SCHEMA_LEADER_CF);
        this.createKeyspaceIfRequired();
        this.session = cluster.connect(this.keyspaceName);
        this.cassandraVersion = cluster.getMetadata().getAllHosts().stream().map(h -> h.getCassandraVersion()).min(VersionNumber::compareTo).get();
        this.ensureSchemaTable();
        this.logMigrationStatement = this.session.prepare(String.format(INSERT_MIGRATION, this.getTableName()));
        this.takeMigrationLeadStatement = this.session.prepare(String.format(TAKE_LEAD_QUERY, this.getLeaderTableName(), 300));
        this.releaseMigrationLeadStatement = this.session.prepare(String.format(RELEASE_LEAD_QUERY, this.getLeaderTableName()));
        try {
            tmpInstanceAddress = InetAddress.getLocalHost().getHostAddress();
        }
        catch (UnknownHostException e) {
            LOGGER.warn("Could not find the local host address. Using default value.");
            tmpInstanceAddress = "unknown";
        }
        this.instanceAddress = tmpInstanceAddress;
    }

    private static String createTableName(String tablePrefix, String tableName) {
        if (tablePrefix == null || tablePrefix.isEmpty()) {
            return tableName;
        }
        return String.format("%s_%s", tablePrefix, tableName);
    }

    private void createKeyspaceIfRequired() {
        if (this.keyspace == null || this.keyspaceExists()) {
            return;
        }
        try (Session session = this.cluster.connect();){
            session.execute(this.keyspace.getCqlStatement());
        }
        catch (DriverException exception) {
            throw new MigrationException(String.format("Unable to create keyspace %s.", this.keyspaceName), exception);
        }
    }

    private boolean keyspaceExists() {
        return this.cluster.getMetadata().getKeyspace(this.keyspace.getKeyspaceName()) != null;
    }

    @Override
    public void close() {
        this.session.close();
    }

    public int getVersion() {
        Statement getVersionQuery = new SimpleStatement(String.format(VERSION_QUERY, this.getTableName())).setConsistencyLevel(this.consistencyLevel);
        ResultSet resultSet = this.session.execute(getVersionQuery);
        Row result = resultSet.one();
        if (result == null) {
            return 0;
        }
        return result.getInt(0);
    }

    public String getKeyspaceName() {
        return this.keyspaceName;
    }

    public String getTableName() {
        return this.tableName;
    }

    public String getLeaderTableName() {
        return this.leaderTableName;
    }

    private void ensureSchemaTable() {
        if (this.schemaTablesIsNotExisting()) {
            this.createSchemaTable();
        }
    }

    private boolean schemaTablesIsNotExisting() {
        Metadata metadata = this.cluster.getMetadata();
        KeyspaceMetadata keyspace = metadata.getKeyspace(this.keyspaceName);
        TableMetadata table = keyspace.getTable(this.getTableName());
        TableMetadata leaderTable = keyspace.getTable(this.getLeaderTableName());
        return table == null || leaderTable == null;
    }

    private void createSchemaTable() {
        String ifNotExistsString = this.isVersionAtLeastV2(this.cassandraVersion) ? "IF NOT EXISTS" : "";
        this.session.execute(String.format(CREATE_MIGRATION_CF, ifNotExistsString, this.getTableName()));
        this.session.execute(String.format(CREATE_LEADER_CF, ifNotExistsString, this.getLeaderTableName()));
    }

    public void execute(DbMigration migration) {
        Ensure.notNull(migration, "migration");
        LOGGER.debug(String.format("About to execute migration %s to version %d", migration.getScriptName(), migration.getVersion()));
        String lastStatement = null;
        try {
            SimpleCQLLexer lexer = new SimpleCQLLexer(migration.getMigrationScript());
            for (String statement : lexer.getCqlQueries()) {
                lastStatement = statement = statement.trim();
                this.executeStatement(statement, migration);
            }
            this.logMigration(migration, true);
            LOGGER.debug(String.format("Successfully applied migration %s to version %d", migration.getScriptName(), migration.getVersion()));
        }
        catch (Exception exception) {
            this.logMigration(migration, false);
            String errorMessage = String.format(MIGRATION_ERROR_MSG, migration.getScriptName(), lastStatement);
            throw new MigrationException(errorMessage, (Throwable)exception, migration.getScriptName(), lastStatement);
        }
    }

    private void executeStatement(String statement, DbMigration migration) {
        if (!statement.isEmpty()) {
            SimpleStatement simpleStatement = new SimpleStatement(statement);
            simpleStatement.setConsistencyLevel(this.consistencyLevel);
            ResultSet resultSet = this.session.execute((Statement)simpleStatement);
            if (!resultSet.getExecutionInfo().isSchemaInAgreement()) {
                throw new MigrationException("Schema agreement could not be reached. You might consider increasing 'maxSchemaAgreementWaitSeconds'.", migration.getScriptName());
            }
        }
    }

    private void logMigration(DbMigration migration, boolean wasSuccessful) {
        BoundStatement boundStatement = this.logMigrationStatement.bind(new Object[]{wasSuccessful, migration.getVersion(), migration.getScriptName(), migration.getMigrationScript(), new Date()});
        this.session.execute((Statement)boundStatement);
    }

    public ConsistencyLevel getConsistencyLevel() {
        return this.consistencyLevel;
    }

    public Database setConsistencyLevel(ConsistencyLevel consistencyLevel) {
        this.consistencyLevel = Ensure.notNull(consistencyLevel, "consistencyLevel");
        return this;
    }

    boolean takeLeadOnMigrations(int repositoryLatestVersion) {
        if (!this.isVersionAtLeastV2(this.cassandraVersion)) {
            return true;
        }
        while (repositoryLatestVersion > this.getVersion()) {
            try {
                LOGGER.debug("Trying to take lead on schema migrations");
                BoundStatement boundStatement = this.takeMigrationLeadStatement.bind(new Object[]{this.getKeyspaceName(), this.instanceId, this.instanceAddress});
                ResultSet lwtResult = this.session.execute((Statement)boundStatement);
                if (lwtResult.wasApplied()) {
                    LOGGER.debug("Took lead on schema migrations");
                    this.tookLead = true;
                    return true;
                }
                LOGGER.info("Schema migration is locked by another instance. Waiting for it to be released...");
                this.waitFor(10000);
            }
            catch (InvalidQueryException e1) {
                LOGGER.info("All required tables do not exist yet, waiting for them to be created...");
                this.waitFor(10000);
            }
        }
        return false;
    }

    private void waitFor(int waitTime) {
        try {
            Thread.sleep(waitTime);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException(e);
        }
    }

    void removeLeadOnMigrations() {
        if (this.tookLead) {
            LOGGER.debug("Trying to release lead on schema migrations");
            BoundStatement boundStatement = this.releaseMigrationLeadStatement.bind(new Object[]{this.getKeyspaceName(), this.instanceId});
            ResultSet lwtResult = this.session.execute((Statement)boundStatement);
            if (lwtResult.wasApplied()) {
                LOGGER.debug("Released lead on schema migrations");
                this.tookLead = false;
                return;
            }
            LOGGER.warn("Could not release lead on schema migrations");
            return;
        }
    }

    public boolean isVersionAtLeastV2(VersionNumber cassandraVersion) {
        return cassandraVersion.compareTo(VersionNumber.parse((String)"2.0")) >= 0;
    }
}

