/*
 * Decompiled with CFR 0.152.
 */
package org.codelibs.elasticsearch.runner;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import org.codelibs.elasticsearch.runner.ClusterRunnerException;
import org.elasticsearch.action.ActionResponse;
import org.elasticsearch.action.ShardOperationFailedException;
import org.elasticsearch.action.admin.cluster.health.ClusterHealthRequest;
import org.elasticsearch.action.admin.cluster.health.ClusterHealthResponse;
import org.elasticsearch.action.admin.cluster.health.ClusterHealthStatus;
import org.elasticsearch.action.admin.cluster.state.ClusterStateResponse;
import org.elasticsearch.action.admin.cluster.tasks.PendingClusterTasksResponse;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsResponse;
import org.elasticsearch.action.admin.indices.flush.FlushResponse;
import org.elasticsearch.action.admin.indices.optimize.OptimizeResponse;
import org.elasticsearch.action.admin.indices.refresh.RefreshResponse;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.AdminClient;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.Requests;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.logging.log4j.LogConfigurator;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.node.Node;
import org.elasticsearch.node.internal.InternalNode;
import org.elasticsearch.search.sort.SortBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

public class ElasticsearchClusterRunner {
    private static final ESLogger logger = Loggers.getLogger(ElasticsearchClusterRunner.class);
    protected static final String LOGGING_YAML = "logging.yml";
    protected static final String ELASTICSEARCH_YAML = "elasticsearch.yml";
    protected static final String WORK_DIR = "work";
    protected static final String DATA_DIR = "data";
    protected static final String LOGS_DIR = "logs";
    protected static final String PLUGINS_DIR = "plugins";
    protected static final String CONFIG_DIR = "config";
    protected List<Node> nodeList = new ArrayList<Node>();
    @Option(name="-basePath", usage="Base path for Elasticsearch.")
    protected String basePath;
    @Option(name="-numOfNode", usage="The number of Elasticsearch node.")
    protected int numOfNode = 3;
    @Option(name="-baseTransportPort", usage="Base transport port.")
    protected int baseTransportPort = 9300;
    @Option(name="-baseHttpPort", usage="Base http port.")
    protected int baseHttpPort = 9200;
    @Option(name="-clusterName", usage="Cluster name.")
    protected String clusterName = "elasticsearch-cluster-runner";
    @Option(name="-indexStoreType", usage="Index store type.")
    protected String indexStoreType = "default";
    @Option(name="-useStdOut", usage="Print logs to stdout.")
    protected boolean useStdOut = true;
    @Option(name="-throwOnFailure", usage="Throw an exception on a failure.")
    protected boolean throwOnFailure = true;

    public static void main(String[] args) {
        final ElasticsearchClusterRunner runner = new ElasticsearchClusterRunner();
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                runner.close();
            }
        });
        runner.build(args);
        while (!runner.isClosed()) {
            try {
                Thread.sleep(5000L);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    public boolean isClosed() {
        for (Node node : this.nodeList) {
            if (node.isClosed()) continue;
            return false;
        }
        return true;
    }

    public void close() {
        for (Node node : this.nodeList) {
            node.close();
        }
        this.print("Closed all nodes.");
    }

    public void clean() {
        Path bPath = FileSystems.getDefault().getPath(this.basePath, new String[0]);
        try {
            Files.walkFileTree(bPath, (FileVisitor<? super Path>)new FileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.delete(file);
                    return this.checkIfExist(file);
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    throw exc;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    if (exc == null) {
                        Files.delete(dir);
                        if (!Files.exists(dir, new LinkOption[0])) {
                            return FileVisitResult.CONTINUE;
                        }
                        throw new IOException();
                    }
                    throw exc;
                }

                private FileVisitResult checkIfExist(Path path) throws IOException {
                    if (Files.exists(path, new LinkOption[0])) {
                        throw new IOException("Failed to delete " + path);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
            this.print("Deleted " + this.basePath);
        }
        catch (IOException e) {
            this.print("Failed to delete " + this.basePath);
        }
    }

    public void build(String ... args) {
        Path logConfPath;
        if (args != null) {
            CmdLineParser parser = new CmdLineParser((Object)this);
            parser.setUsageWidth(80);
            try {
                parser.parseArgument(args);
            }
            catch (CmdLineException e) {
                throw new ClusterRunnerException("Failed to parse args: " + Strings.arrayToDelimitedString((Object[])args, (String)" "));
            }
        }
        if (this.basePath == null) {
            try {
                this.basePath = Files.createTempDirectory("es-cluster", new FileAttribute[0]).toAbsolutePath().toString();
            }
            catch (IOException e) {
                throw new ClusterRunnerException("Could not create $ES_HOME.", e);
            }
        }
        Path esBasePath = Paths.get(this.basePath, new String[0]);
        this.createDir(esBasePath);
        Path confPath = Paths.get(this.basePath, CONFIG_DIR);
        this.createDir(confPath);
        Path esConfPath = confPath.resolve(ELASTICSEARCH_YAML);
        if (!Files.exists(esConfPath, new LinkOption[0])) {
            try (InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream("config/elasticsearch.yml");){
                Files.copy(is, esConfPath, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new ClusterRunnerException("Could not create: " + esConfPath, e);
            }
        }
        if (!Files.exists(logConfPath = confPath.resolve(LOGGING_YAML), new LinkOption[0])) {
            try (InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream("config/logging.yml");){
                Files.copy(is, logConfPath, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new ClusterRunnerException("Could not create: " + logConfPath, e);
            }
        }
        this.print("----------------------------------------");
        this.print("Cluster Name: " + this.clusterName);
        this.print("Base Path:    " + this.basePath);
        this.print("Num Of Node:  " + this.numOfNode);
        this.print("----------------------------------------");
        for (int i = 0; i < this.numOfNode; ++i) {
            this.nodeList.add(this.buildNode(i + 1));
        }
    }

    protected Node buildNode(int number) {
        Path confPath = Paths.get(this.basePath, CONFIG_DIR);
        Path pluginsPath = Paths.get(this.basePath, PLUGINS_DIR);
        Path logsPath = Paths.get(this.basePath, LOGS_DIR, "node_" + number);
        Path dataPath = Paths.get(this.basePath, DATA_DIR, "node_" + number);
        Path workPath = Paths.get(this.basePath, WORK_DIR, "node_" + number);
        this.createDir(logsPath);
        this.createDir(dataPath);
        this.createDir(workPath);
        ImmutableSettings.Builder settingsBuilder = ImmutableSettings.settingsBuilder();
        settingsBuilder.put("path.conf", confPath.toAbsolutePath().toString());
        settingsBuilder.put("path.data", dataPath.toAbsolutePath().toString());
        settingsBuilder.put("path.work", workPath.toAbsolutePath().toString());
        settingsBuilder.put("path.logs", logsPath.toAbsolutePath().toString());
        settingsBuilder.put("path.plugins", pluginsPath.toAbsolutePath().toString());
        String nodeName = "Node " + number;
        int transportPort = this.baseTransportPort + number;
        int httpPort = this.baseHttpPort + number;
        settingsBuilder.put("cluster.name", this.clusterName);
        settingsBuilder.put("node.name", nodeName);
        settingsBuilder.put("node.master", true);
        settingsBuilder.put("node.data", true);
        settingsBuilder.put("http.enabled", true);
        settingsBuilder.put("transport.tcp.port", transportPort);
        settingsBuilder.put("http.port", httpPort);
        settingsBuilder.put("index.store.type", this.indexStoreType);
        this.print("Node Name:      " + nodeName);
        this.print("HTTP Port:      " + httpPort);
        this.print("Transport Port: " + transportPort);
        this.print("Data Directory: " + dataPath);
        this.print("Log Directory:  " + logsPath);
        this.print("----------------------------------------");
        Settings settings = settingsBuilder.build();
        LogConfigurator.configure((Settings)settings);
        InternalNode node = new InternalNode(settings, true);
        node.start();
        return node;
    }

    public Node getNode(int i) {
        return this.nodeList.get(i);
    }

    public int getNodeSize() {
        return this.nodeList.size();
    }

    protected void print(String line) {
        if (this.useStdOut) {
            System.out.println(line);
        } else {
            logger.info(line, new Object[0]);
        }
    }

    protected void createDir(Path path) {
        if (!Files.exists(path, new LinkOption[0])) {
            this.print("Creating " + path);
            try {
                Files.createDirectories(path, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new ClusterRunnerException("Failed to create " + path, e);
            }
        }
    }

    public Client client() {
        return this.getNode(0).client();
    }

    public AdminClient admin() {
        return this.client().admin();
    }

    public ClusterHealthStatus ensureGreen(String ... indices) {
        ClusterHealthResponse actionGet = (ClusterHealthResponse)this.client().admin().cluster().health(Requests.clusterHealthRequest((String[])indices).waitForGreenStatus().waitForEvents(Priority.LANGUID).waitForRelocatingShards(0)).actionGet();
        if (actionGet.isTimedOut()) {
            this.onFailure("ensureGreen timed out, cluster state:\n" + ((ClusterStateResponse)this.client().admin().cluster().prepareState().get()).getState().prettyPrint() + "\n" + ((PendingClusterTasksResponse)this.client().admin().cluster().preparePendingClusterTasks().get()).prettyPrint(), (ActionResponse)actionGet);
        }
        return actionGet.getStatus();
    }

    public ClusterHealthStatus ensureYellow(String ... indices) {
        ClusterHealthResponse actionGet = (ClusterHealthResponse)this.client().admin().cluster().health(Requests.clusterHealthRequest((String[])indices).waitForRelocatingShards(0).waitForYellowStatus().waitForEvents(Priority.LANGUID)).actionGet();
        if (actionGet.isTimedOut()) {
            this.onFailure("ensureYellow timed out, cluster state:\n\n" + ((ClusterStateResponse)this.client().admin().cluster().prepareState().get()).getState().prettyPrint() + "\n" + ((PendingClusterTasksResponse)this.client().admin().cluster().preparePendingClusterTasks().get()).prettyPrint(), (ActionResponse)actionGet);
        }
        return actionGet.getStatus();
    }

    public ClusterHealthStatus waitForRelocation() {
        ClusterHealthRequest request = Requests.clusterHealthRequest((String[])new String[0]).waitForRelocatingShards(0);
        ClusterHealthResponse actionGet = (ClusterHealthResponse)this.client().admin().cluster().health(request).actionGet();
        if (actionGet.isTimedOut()) {
            this.onFailure("waitForRelocation timed out, cluster state:\n\n" + ((ClusterStateResponse)this.client().admin().cluster().prepareState().get()).getState().prettyPrint() + "\n" + ((PendingClusterTasksResponse)this.client().admin().cluster().preparePendingClusterTasks().get()).prettyPrint(), (ActionResponse)actionGet);
        }
        return actionGet.getStatus();
    }

    public FlushResponse flush() {
        this.waitForRelocation();
        FlushResponse actionGet = (FlushResponse)this.client().admin().indices().prepareFlush(new String[0]).execute().actionGet();
        ShardOperationFailedException[] shardFailures = actionGet.getShardFailures();
        if (shardFailures != null && shardFailures.length != 0) {
            this.onFailure(shardFailures.toString(), (ActionResponse)actionGet);
        }
        return actionGet;
    }

    public RefreshResponse refresh() {
        this.waitForRelocation();
        RefreshResponse actionGet = (RefreshResponse)this.client().admin().indices().prepareRefresh(new String[0]).execute().actionGet();
        ShardOperationFailedException[] shardFailures = actionGet.getShardFailures();
        if (shardFailures != null && shardFailures.length != 0) {
            this.onFailure(shardFailures.toString(), (ActionResponse)actionGet);
        }
        return actionGet;
    }

    public OptimizeResponse optimize(boolean forece) {
        this.waitForRelocation();
        OptimizeResponse actionGet = (OptimizeResponse)this.client().admin().indices().prepareOptimize(new String[0]).setForce(forece).execute().actionGet();
        ShardOperationFailedException[] shardFailures = actionGet.getShardFailures();
        if (shardFailures != null && shardFailures.length != 0) {
            this.onFailure(shardFailures.toString(), (ActionResponse)actionGet);
        }
        return actionGet;
    }

    public CreateIndexResponse createIndex(String index, Settings settings) {
        CreateIndexResponse actionGet = (CreateIndexResponse)this.client().admin().indices().prepareCreate(index).setSettings(settings != null ? settings : ImmutableSettings.Builder.EMPTY_SETTINGS).execute().actionGet();
        if (!actionGet.isAcknowledged()) {
            this.onFailure("Failed to create " + index + ".", (ActionResponse)actionGet);
        }
        return actionGet;
    }

    public boolean indexExists(String index) {
        IndicesExistsResponse actionGet = (IndicesExistsResponse)this.client().admin().indices().prepareExists(new String[]{index}).execute().actionGet();
        return actionGet.isExists();
    }

    public IndexResponse insert(String index, String type, String id, String source) {
        IndexResponse actionGet = (IndexResponse)this.client().prepareIndex(index, type, id).setSource(source).setRefresh(true).execute().actionGet();
        if (!actionGet.isCreated()) {
            this.onFailure("Failed to insert " + id + " into " + index + "/" + type + ".", (ActionResponse)actionGet);
        }
        return actionGet;
    }

    public DeleteResponse delete(String index, String type, String id) {
        DeleteResponse actionGet = (DeleteResponse)this.client().prepareDelete(index, type, id).setRefresh(true).execute().actionGet();
        if (!actionGet.isFound()) {
            this.onFailure("Failed to delete " + id + " from " + index + "/" + type + ".", (ActionResponse)actionGet);
        }
        return actionGet;
    }

    public SearchResponse search(String index, String type, QueryBuilder queryBuilder, SortBuilder sort, int from, int size) {
        SearchResponse actionGet = (SearchResponse)this.client().prepareSearch(new String[]{index}).setTypes(new String[]{type}).setQuery((QueryBuilder)(queryBuilder != null ? queryBuilder : QueryBuilders.matchAllQuery())).addSort((SortBuilder)(sort != null ? sort : SortBuilders.scoreSort())).setFrom(from).setSize(size).execute().actionGet();
        return actionGet;
    }

    private void onFailure(String message, ActionResponse response) {
        if (this.throwOnFailure) {
            throw new ClusterRunnerException(message, response);
        }
        this.print(message);
    }
}

